/*
    acltool, a GTK editor for AFS ACLs
    Copyright (C) 2000  Nohup AB

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>	/* BUFSIZ */
#include <string.h>
#define __USE_XOPEN_EXTENDED
#include <ftw.h>	/* for recursive operation */

#ifdef HAVE_SYS_IOCCOM_H
#include <sys/ioccom.h>
#endif
#include <sys/types.h>
#include <sys/ioctl.h>
#include <kafs.h>	/* AFS access */

#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"

#define AFSNAMEMAX 256

enum BITS {
	B_READ		= 1<<0,	/* read files */
	B_WRITE		= 1<<1,	/* write files & write-lock existing files */
	B_INSERT	= 1<<2,	/* insert & write-lock new files */
	B_LOOKUP	= 1<<3,	/* Enumerate files and examine ACL */
	B_DELETE	= 1<<4,	/* Remove files */
	B_LOCK		= 1<<5,	/* read-lock files */
	B_ADMINISTER	= 1<<6	/* Set access list of directory */
};

int BIT[I_MAX]={B_READ, B_LOOKUP, B_INSERT, B_DELETE, B_WRITE, B_LOCK,
  B_ADMINISTER };

const char LETTER[I_PLUS1]="RLIDWKA";

const char OFF='-', ALLOFF[I_PLUS1]="-------";

const char *npermbuts[I_PLUS1]={"nread", "nlookup", "ninsert", "ndelete",
  "nwrite", "nlock", "nadminister", NULL};

int success=0, failure=0;

char *path=NULL;

void setperms(GtkWidget *, const char *, int);
void dodialog(const char *);

struct ac {
  	int row;
	char name[AFSNAMEMAX];
	char pos[I_PLUS1];
	char neg[I_PLUS1];
} *currentac=NULL;

gint acnameis(gconstpointer a, gconstpointer name)
 {
  return strcmp(((struct ac *)a)->name, (const char *)name);
 }

void addactoclist(gpointer a, gpointer clist)
 {
  gint row;
  gchar *content[4];

  content[0]=((struct ac *)a)->name;
  content[1]=((struct ac *)a)->pos;
  content[2]=((struct ac *)a)->neg;
  content[3]=NULL;
  row=gtk_clist_append(GTK_CLIST(clist), content);
  gtk_clist_set_row_data(GTK_CLIST(clist), row, a);
  ((struct ac *)a)->row=row;
 }

GList *afsacl=NULL;

gboolean
doquit                                 (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
 {
  gtk_exit(0);
  return FALSE;
 }


void
on_negative_toggled                    (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  GtkWidget *button;
  int i;

  if(togglebutton->active)
   {
    for(i=0;npermbuts[i];i++)
     {
      button=lookup_widget(GTK_WIDGET(togglebutton), npermbuts[i]);
      gtk_widget_show(button);
     }
   }
  else
   {
    for(i=0;npermbuts[i];i++)
     {
      button=lookup_widget(GTK_WIDGET(togglebutton), npermbuts[i]);
      gtk_widget_hide(button);
     }
    setperms(GTK_WIDGET(togglebutton), ALLOFF, 0);
   }
}


void
do_reread                              (GtkWidget     *widget,
                                        gpointer         user_data)
 {
  GtkCList *groups;
  GtkEntry *directory;
  GList *glp;
  char buffer[BUFSIZ], name[AFSNAMEMAX], *p;
  struct ViceIoctl a_params;
  int positives, negatives, i, step, j, k, access;
  struct ac *a;

  directory=GTK_ENTRY(lookup_widget(widget, "directory"));
  g_free(path);
  p=gtk_entry_get_text(directory);
  path=g_strdup(p);
  a_params.in_size=0;
  a_params.out_size=BUFSIZ;
  a_params.in=NULL;
  a_params.out=buffer;
  if(k_pioctl(path,VIOCGETAL,&a_params,1)==-1)
   {
    strcpy(buffer, "0\n0\n");
    p=g_strdup_printf("Couldn't get ACL for %s - setting it probably won't work either", path);
    dodialog(p);
    g_free(p);
   }
  sscanf(buffer, "%d\n%d\n%n", &positives, &negatives, &i);
  g_list_foreach(afsacl, (GFunc)g_free, NULL);
  g_list_free(afsacl);
  afsacl=NULL;
  for(j=0;j<positives;j++)
   {
    sscanf(buffer+i, "%s\t%d\n%n", name, &access, &step);
    i+=step;
    a=g_malloc0(sizeof(struct ac));
    strcpy(a->name, name);
    strcpy(a->neg, ALLOFF);
    for(k=0;k<I_MAX;k++)
      a->pos[k]=access&BIT[k]?LETTER[k]:OFF;
    afsacl=g_list_append(afsacl, a);
   }
  for(j=0;j<negatives;j++)
   {
    sscanf(buffer+i, "%s\t%d\n%n", name, &access, &step);
    i+=step;
    glp=g_list_find_custom(afsacl, name, acnameis);
    if(glp)
      a=glp->data;
    else
     {
      a=g_malloc0(sizeof(struct ac));
      strcpy(a->neg, ALLOFF);
      strcpy(a->name, name);
     }
    for(k=0;k<I_MAX;k++)
      a->neg[k]=access&BIT[k]?LETTER[k]:OFF;
    if(!glp)
      afsacl=g_list_append(afsacl, a);
   }
  groups=GTK_CLIST(lookup_widget(widget, "groups"));
  gtk_clist_freeze(groups);
  gtk_clist_clear(groups);
  g_list_foreach(afsacl, addactoclist, groups);
  gtk_clist_moveto(groups, 0, 0, 0, 0);
  currentac=NULL;
  gtk_clist_select_row(groups, 0, 0);
  gtk_clist_thaw(groups);
 }


void
on_group_activate                      (GtkEditable     *editable,
                                        gpointer         user_data)
 {
  struct ac *a;
  GList *glp;
  GtkCList *groups;

  glp=g_list_find_custom(afsacl, gtk_entry_get_text(GTK_ENTRY(editable)), acnameis);
  groups=GTK_CLIST(lookup_widget(GTK_WIDGET(editable), "groups"));
  if(glp)
   {
    a=glp->data;
   }
  else
   {
    a=g_malloc0(sizeof(struct ac));
    strcpy(a->name, gtk_entry_get_text(GTK_ENTRY(editable)));
    strcpy(a->pos, ALLOFF);
    strcpy(a->neg, ALLOFF);
    afsacl=g_list_append(afsacl, a);
    addactoclist(a, groups);
   }
  gtk_clist_select_row(groups, a->row, 0);
 }

void setperms(GtkWidget *w, const char *perms, int pos)
 {
  GtkToggleButton *cb;
  int i;

  for(i=0;npermbuts[i]&&perms[i];i++)
   {
    cb=GTK_TOGGLE_BUTTON(lookup_widget(w, npermbuts[i]+pos));
    gtk_toggle_button_set_active(cb, perms[i]!=OFF);
   }
 }

void
on_qnone_clicked                       (GtkButton       *button,
                                        gpointer         user_data)
{
  setperms(GTK_WIDGET(button), ALLOFF, 1);
}


void
on_qread_clicked                       (GtkButton       *button,
                                        gpointer         user_data)
{
  setperms(GTK_WIDGET(button), "RL-----", 1);
}


void
on_qwrite_clicked                      (GtkButton       *button,
                                        gpointer         user_data)
{
  setperms(GTK_WIDGET(button), "RLIDWK-", 1);
}


void
on_qinsert_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
  setperms(GTK_WIDGET(button), "-LI--K-", 1);
}


void
on_qall_clicked                        (GtkButton       *button,
                                        gpointer         user_data)
{
  setperms(GTK_WIDGET(button), "RLIDWKA", 1);
}


void
on_groups_select_row                   (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  GtkToggleButton *pb, *nb;
  GtkEntry *group;
  struct ac *a;
  int i, neg=0;

  a=(struct ac *) gtk_clist_get_row_data(clist, row);
  currentac=a;
  for(i=0;i<I_MAX;i++)
   {
    nb=GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(clist), npermbuts[i]));
    pb=GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(clist), npermbuts[i]+1));
    gtk_toggle_button_set_active(nb, a->neg[i]!=OFF);
    gtk_toggle_button_set_active(pb, a->pos[i]!=OFF);
    if(a->neg[i]!=OFF)
     neg=1;
   }
  nb=GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(clist), "negative"));
  gtk_toggle_button_set_active(nb, neg);
  group=GTK_ENTRY(lookup_widget(GTK_WIDGET(clist), "group"));
  gtk_entry_set_text(group, a->name);
}


void
on_applyQuit_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
  on_apply_clicked(button,user_data);
  if(success&&!failure)
    doquit(NULL,NULL,NULL);
  success=failure=0;
}

void countpos(gpointer a, gpointer i)
 {
  if(strcmp(ALLOFF, ((struct ac *)a)->pos))
    (*(int*)i)++;
 }

void countneg(gpointer a, gpointer i)
 {
  if(strcmp(ALLOFF, ((struct ac *)a)->neg))
    (*(int*)i)++;
 }

struct ob {
  char buffer[BUFSIZ];
  char pos;
  int offset;
  size_t free;
};

void printac(gpointer a1, gpointer a2)
 {
  struct ac *a=(struct ac *)a1;
  struct ob *o=(struct ob *)a2;
  int i, access=0;

  for(i=0;i<I_MAX;i++)
    if((o->pos?a->pos[i]:a->neg[i])!=OFF)
      access|=BIT[i];
  if(access)
   {
    i=snprintf(o->buffer+o->offset, o->free, "%s\t%d\n", a->name, access);
    if(i>0 && i<o->free)
     {
      o->offset+=i;
      o->free-=i;
     }
    else
      o->free=0;
   }
 }

void dodialog(const char *message)
 {
  static GtkWidget *dialog=NULL;
  GtkLabel *text;

  if(!dialog)
    dialog=create_dialog();
  text=GTK_LABEL(lookup_widget(dialog, "text"));
  gtk_label_set(text, message);
  gtk_widget_show(dialog);
 }

struct ViceIoctl *aparms;

int recapply(const char *file, const struct stat *sb, int flag, struct FTW *s)
 {
  switch(flag)
   {
    case FTW_DP:	/* shouldn't happen, we don't use FTW_DEPTH */
    case FTW_D:
    case FTW_DNR:
    case FTW_NS:	/* stat failure.. */
      if(k_pioctl((char*)file, VIOCSETAL, aparms, 0)==-1)
        failure++;
      else
	success++;
   }
  return 0;
 }

void
on_apply_clicked                       (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkOptionMenu *recursive;
  GtkMenuShell *menu;
  GList *list;
  gpointer item;
  gint do_rec;
  struct ViceIoctl a_params;
  int i, pos=0, neg=0;
  struct ob o;
  GtkEntry *directory;

  /* read option menu - ugly, this is what Galeon does */
  recursive=GTK_OPTION_MENU(lookup_widget(GTK_WIDGET(button), "recursive"));
  menu=GTK_MENU_SHELL(recursive->menu);
  list=menu->children;
  item=gtk_menu_get_active(GTK_MENU(menu));
  do_rec=g_list_index(list, item);
  o.free=sizeof(o.buffer)-1;
  o.buffer[o.free]=0;
  o.offset=0;
  o.pos=1;
  g_list_foreach(afsacl, countpos, (gpointer)&pos);
  g_list_foreach(afsacl, countneg, (gpointer)&neg);
  i=snprintf(o.buffer, o.free, "%d\n%d\n", pos, neg);
  if(i>0 && i<o.free)
   {
    o.offset=i;
    o.free-=i;
   }
  else
    dodialog("ACL too big");
  a_params.out=NULL;
  a_params.out_size=0;
  a_params.in=o.buffer;
  g_list_foreach(afsacl, printac, (gpointer)&o);
  o.pos=0;
  g_list_foreach(afsacl, printac, (gpointer)&o);
  if(o.free)
   {
    a_params.in_size=strlen(o.buffer);
    directory=GTK_ENTRY(lookup_widget(GTK_WIDGET(button), "directory"));
    gtk_entry_set_text(directory, path);
    if(!do_rec)
     {
      if(k_pioctl(path, VIOCSETAL, &a_params, 1)==-1)
       {
        success=0;
        dodialog("k_pioctl() failed - no administer permission?");
       }
      else
       {
        do_reread(GTK_WIDGET(button), user_data);
        success=1;
       }
     }
    else
     {
      success=0;
      failure=0;
      aparms=&a_params;
      if(nftw(path, recapply, 16, FTW_PHYS))
	failure++;
      if(success&&failure)
	dodialog("Failed to apply the ACL to some directories");
      else if(failure&&!success)
	dodialog("Couldn't apply the ACL to any directory");
      else if(!success&&!failure)
	dodialog("Didn't find a directory to apply ACL to?");
     }
   }
  else
    dodialog("ACL too big");
}


void
on_pos_toggled                        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
 {
  gint i=GPOINTER_TO_INT(user_data);

  if(currentac)
   {
    GtkCList *groups;

    currentac->pos[i]=togglebutton->active?LETTER[i]:OFF;
    groups=GTK_CLIST(lookup_widget(GTK_WIDGET(togglebutton), "groups"));
    gtk_clist_set_text(groups, currentac->row, 1, currentac->pos);
   }
 }

void
on_neg_toggled                       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
 {
  gint i=GPOINTER_TO_INT(user_data);

  if(currentac)
   {
    GtkCList *groups;

    currentac->neg[i]=togglebutton->active?LETTER[i]:OFF;
    groups=GTK_CLIST(lookup_widget(GTK_WIDGET(togglebutton), "groups"));
    gtk_clist_set_text(groups, currentac->row, 2, currentac->neg);
   }
 }

