import os.path
import re
import stat
import os
import signal
import fcntl

from gtk import *
from GDK import *

from rox.SaveBox import SaveBox
from support import *
from rox import choices
import string

child_pid = None

archive_formats = [
	# Extension, extract, create
	('.tar', "tar xf '%s'", "tar cf '%(dst)s' '%(src)s'"),
	('.zip', "unzip '%s'", "zip -r '%(dst)s' '%(src)s'"),
	('.deb', "ar x '%s'", None),
	('.rar', "rar x '%s'", "rar a '%(dst)s' '%(src)s'"),

	('.tgz', "gunzip -c '%s' | tar xf -",
				"tar cf - '%(src)s' | gzip > '%(dst)s'"),
	('.tar.gz', "gunzip -c '%s' | tar xf -",
				"tar cf - '%(src)s' | gzip > '%(dst)s'"),

	('.tbz', "bunzip2 -c '%s' | tar xf -",
			"tar cf - '%(src)s' | bzip2 > '%(dst)s'"),
	('.tar.bz', "bunzip2 -c '%s' | tar xf -",
			"tar cf - '%(src)s' | bzip2 > '%(dst)s'"),
	('.tbz2', "bunzip2 -c '%s' | tar xf -",
			"tar cf - '%(src)s' | bzip2 > '%(dst)s'"),
	('.tar.bz2', "bunzip2 -c '%s' | tar xf -",
			"tar cf - '%(src)s' | bzip2 > '%(dst)s'")
	]

compressed_formats = [
	# Ext, extract, compress, type
	('.gz', "gunzip -c '%s'", "gzip -c '%s'", 'x-gzip'),
	('.bz', "bunzip2 -ck '%s'", "bzip2 -c '%s'", 'x-bzip'),
	('.bz2', "bunzip2 -ck '%s'", "bzip2 -c '%s'", 'x-bzip')
]

def get_format(path, formats):
	for f in formats:
		ext = f[0]
		if ext == string.lower(path[-len(ext):]):
			return f
	return None

def esc(text):
	"""Return text with \ and ' escaped"""
	return re.sub("'", "'\"'\"'", text)

def bg_system(command, out = None):
	"system(command), but still process GUI events while waiting..."
	"If 'out' is set then the child's stdout goes to that FD. 'out' is "
	"closed in the parent process."
	global child_pid

	(r, w) = os.pipe()
	
	child_pid = fork()
	if child_pid == -1:
		os.close(r)
		os.close(w)
		if out != None:
			os.close(out)
		report_error("fork() failed!")
		return 127
	if child_pid == 0:
		# Child
		try:
			if out != None:
				os.dup2(out, 1)

			# Collect stderr...
			if w != 2:
				os.dup2(w, 2)
				os.close(w)
			
			os.setpgid(0, 0)	# Start a new process group

			os.close(r)
			
			error = os.system(command) != 0
			os._exit(error)
		except:
			pass
		os._exit(127)
	
	if out != None:
		os.close(out)
	os.close(w)

	done = [""]
	def cb(src, cond, done = done):
		data = os.read(src, 100)
		if data:
			done[0] = done[0] + data
		else:
			done.append(1)
	tag = input_add(r, INPUT_READ, cb)

	while len(done) < 2:
		mainiteration()

	input_remove(tag)

	os.close(r)
	(pid, status) = waitpid(child_pid, 0)
	child_pid = None

	str = string.strip(done[0])
	if status or str:
		if str:
			report_error("Error: " + str)
		else:
			report_error("Operation failed")

	return status

def make_archiver(path):
	while path[-1:] == '/':
		path = path[:-1]
	if os.path.isdir(path):
		window = ArchiveDir(path)
	else:
		window = None
		f = get_format(path, archive_formats)
		if f:
			window = ExtractDir(path, f)
		else:
			f = get_format(path, compressed_formats)
			if f:
				window = ExtractFile(path, f)
			else:
				window = ArchiveFile(path)
	
	window.connect('destroy', lambda w: rox_toplevel_unref())
	rox_toplevel_ref()
	window.show()

def pull_up(dir):
	"If dir contains only one subdir, move its contents into dir."
	list = os.listdir(dir)
	if len(list) != 1:
		return
	
	subdir = os.path.join(dir, list[0])
	if not os.path.isdir(subdir):
		return

	tmp = '<Unknown>'
	try:
		tmp = os.tempnam(dir)
		os.rename(subdir, tmp)
		subdir = tmp
	except:
		print "Warning: failed to rename(%s, %s)\n" % (subdir, tmp)
	 
	for file in os.listdir(subdir):
		bg_system("mv '%s' ." % esc(os.path.join(subdir, file)))
	os.rmdir(subdir)

def report_known(formats):
	txt = "Allowed extensions are:\n"
	for f in formats:
		if f[2]:
			txt = txt + f[0] + '\n'
	report_error(txt)

class Archive(SaveBox):
	def __init__(self, win, type):
		SaveBox.__init__(self, self, self.uri, type);
		self.connect('destroy', self.destroyed)
	
	def destroyed(self, widget):
		global child_pid

		if child_pid:
			os.kill(-child_pid, signal.SIGTERM)
		
	def save_as_file(self, path):
		self.set_sensitive(FALSE)
		success = FALSE
		try:
			success = self.do_save(path)
		except:
			success = FALSE
			report_exception()
		self.set_sensitive(TRUE);
		return success
	
	def set_uri(self, uri):
		path = get_local_path(uri)
		if path:
			child('rox', '-x', path)
		pass

# save_as(path) - write data to file, TRUE on success
# set_uri(uri) - data is safely saved to this location

class ExtractDir(Archive):
	def __init__(self, path, format):
		self.path = path
		ext = format[0]
		self.extract = format[1]
		self.uri = path[:-len(ext)]
		Archive.__init__(self, self, 'special/directory')
	
	def do_save(self, path):
		os.mkdir(path)
		os.chdir(path)
		if bg_system(self.extract % esc(self.path)):
			return FALSE
		else:
			pull_up(path)
			return TRUE

class ExtractFile(Archive):
	def __init__(self, path, format):
		self.path = path
		ext = format[0]
		self.extract = format[1]
		self.uri = path[:-len(ext)]
		Archive.__init__(self, self, 'text/plain')
	
	def do_save(self, path):
		if os.path.exists(path):
			report_error("File `%s' already exists!" % path)
			return FALSE
		stats = os.stat(self.path)
		mode = stat.S_IMODE(stats[stat.ST_MODE]) & 0x1ff;
		out = os.open(path, os.O_WRONLY | os.O_CREAT, mode)
		return bg_system(self.extract % esc(self.path), out = out) == 0
	
	def save_as_selection(self, selection_data):
		(r, w) = os.pipe()
		self.data = ""
		
		def cb(src, cond, self = self):
			self.data = self.data + os.read(src, 1024)
		input_add(r, INPUT_READ, cb)
		
		bg_system(self.extract % esc(self.path), out = w)

		while 1:
			new = os.read(r, 1024)
			if not new:
				break
			self.data = self.data + new
		os.close(r)
		selection_data.set(selection_data.target, 8, self.data)
		self.data = ""

class ArchiveFile(Archive):
	def __init__(self, path):
		self.path = path
		self.uri = path + '.gz'
		Archive.__init__(self, self, 'application/x-gzip')
	
	def do_save(self, path):
		format = get_format(path, compressed_formats)
		if not format or not format[2]:
			report_known(compressed_formats)
			return FALSE

		if os.path.exists(path):
			report_error("File `%s' already exists!" % path)
			return FALSE
		stats = os.stat(self.path)
		mode = stat.S_IMODE(stats[stat.ST_MODE]) & 0x1ff;
		out = os.open(path, os.O_WRONLY | os.O_CREAT, mode)
		return bg_system(format[2] % esc(self.path), out = out) == 0

class ArchiveDir(Archive):
	def __init__(self, path):
		self.path = path
		self.uri = path + '.tgz'
		Archive.__init__(self, self, 'application/x-compressed-tar')
	
	def do_save(self, path):
		format = get_format(path, archive_formats)
		if not format or not format[2]:
			report_known(archive_formats)
			return FALSE
		
		os.chdir(os.path.dirname(self.uri))
		retval = bg_system(format[2] % {
				'src' : esc(os.path.basename(self.path)),
				'dst' : esc(path)
				})
		return retval == 0
