/*
 * Copyright (c) 1995, 1996, 1997, 1998, 1999, 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 *
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>

RCSID("$Id: xfs_vfsops-macos.c,v 1.4 2000/05/07 00:27:09 assar Exp $");

#include <xfs/xfs_common.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_vfsops.h>
#include <xfs/xfs_vfsops-bsd.h>
#include <xfs/xfs_vnodeops.h>

static vop_t **xfs_dead_vnodeop_p;

int
make_dead_vnode(struct mount *mp, struct vnode **vpp)
{
    XFSDEB(XDEBNODE, ("make_dead_vnode mp = %lx\n", (unsigned long)mp));

    return getnewvnode(VT_NON, mp, xfs_dead_vnodeop_p, vpp);
}

static struct vnodeopv_entry_desc xfs_dead_vnodeop_entries[] = {
    {&vop_default_desc, (vop_t *) xfs_eopnotsupp},
    {&vop_lookup_desc,	(vop_t *) xfs_dead_lookup},
    {&vop_reclaim_desc, (vop_t *) xfs_returnzero},
    {&vop_lock_desc,    (vop_t *) nullop},
    {&vop_unlock_desc,  (vop_t *) nullop},
    {&vop_islocked_desc,(vop_t *) nullop},
    {NULL, NULL}};

static struct vnodeopv_desc xfs_dead_vnodeop_opv_desc =
{&xfs_dead_vnodeop_p, xfs_dead_vnodeop_entries};

extern struct vnodeopv_desc xfs_vnodeop_opv_desc;

struct vnodeopv_desc *xfs_vnodeopv_descs[] = {
    &xfs_vnodeop_opv_desc,
    NULL,
};

struct vnodeopv_desc *xfs_dead_vnodeopv_descs[] = {
    &xfs_dead_vnodeop_opv_desc,
    NULL
};


/*
 * If the vfs_opv_descs wasn't included in `struct vfsops' it couldn't
 * get initialized by vfs_attach and we need to do it here.
 */

void vfs_opv_init_explicit __P((struct vnodeopv_desc *));
void vfs_opv_init_default __P((struct vnodeopv_desc *));

/*
 * vfs_init.c
 *
 * Allocate and fill in operations vectors.
 *
 * An undocumented feature of this approach to defining operations is that
 * there can be multiple entries in vfs_opv_descs for the same operations
 * vector. This allows third parties to extend the set of operations
 * supported by another layer in a binary compatibile way. For example,
 * assume that NFS needed to be modified to support Ficus. NFS has an entry
 * (probably nfs_vnopdeop_decls) declaring all the operations NFS supports by
 * default. Ficus could add another entry (ficus_nfs_vnodeop_decl_entensions)
 * listing those new operations Ficus adds to NFS, all without modifying the
 * NFS code. (Of couse, the OTW NFS protocol still needs to be munged, but
 * that is a(whole)nother story.) This is a feature.
 */

extern int vfs_opv_numops;

/*
 * Init the vector, if it needs it.
 * Also handle backwards compatibility.
 */
void
vfs_opv_init_explicit(vfs_opv_desc)
	struct vnodeopv_desc *vfs_opv_desc;
{
	int (**opv_desc_vector) __P((void *));
	struct vnodeopv_entry_desc *opve_descp;

	opv_desc_vector = *(vfs_opv_desc->opv_desc_vector_p);

	if (opv_desc_vector == NULL) {
	    opv_desc_vector = xfs_alloc(vfs_opv_numops * sizeof(int (*)(void *)));
	    bzero(opv_desc_vector, vfs_opv_numops * sizeof(int (*)(void *)));
	    *(vfs_opv_desc->opv_desc_vector_p) = opv_desc_vector;
	}

	for (opve_descp = vfs_opv_desc->opv_desc_ops;
	     opve_descp->opve_op;
	     opve_descp++) {
		/*
		 * Sanity check:  is this operation listed
		 * in the list of operations?  We check this
		 * by seeing if its offest is zero.  Since
		 * the default routine should always be listed
		 * first, it should be the only one with a zero
		 * offset.  Any other operation with a zero
		 * offset is probably not listed in
		 * vfs_op_descs, and so is probably an error.
		 *
		 * A panic here means the layer programmer
		 * has committed the all-too common bug
		 * of adding a new operation to the layer's
		 * list of vnode operations but
		 * not adding the operation to the system-wide
		 * list of supported operations.
		 */
		if (opve_descp->opve_op->vdesc_offset == 0 &&
		    opve_descp->opve_op->vdesc_offset != VOFFSET(vop_default)) {
			printf("operation %s not listed in %s.\n",
			    opve_descp->opve_op->vdesc_name, "vfs_op_descs");
			panic ("vfs_opv_init: bad operation");
		}

		/*
		 * Fill in this entry.
		 */
		opv_desc_vector[opve_descp->opve_op->vdesc_offset] =
		    opve_descp->opve_impl;
	}
}

extern int vfs_opv_numops;

void
vfs_opv_init_default(vfs_opv_desc)
	struct vnodeopv_desc *vfs_opv_desc;
{
	int j;
	int (**opv_desc_vector) __P((void *));

	opv_desc_vector = *(vfs_opv_desc->opv_desc_vector_p);

	/*
	 * Force every operations vector to have a default routine.
	 */
	if (opv_desc_vector[VOFFSET(vop_default)] == NULL)
		panic("vfs_opv_init: operation vector without default routine.");

	for (j = 0; j < vfs_opv_numops; j++)
		if (opv_desc_vector[j] == NULL)
			opv_desc_vector[j] = 
			    opv_desc_vector[VOFFSET(vop_default)];
}

static int
xfs_init(struct vfsconf *vfsp)
{
    XFSDEB(XDEBVFOPS, ("xfs_init\n"));
    vfs_opv_init_explicit(&xfs_vnodeop_opv_desc);
    vfs_opv_init_default(&xfs_vnodeop_opv_desc);
    vfs_opv_init_explicit(&xfs_dead_vnodeop_opv_desc);
    vfs_opv_init_default(&xfs_dead_vnodeop_opv_desc);

    return 0;
}

static struct vfsops
xfs_vfsops = {
    xfs_mount,
    xfs_start,
    xfs_unmount,
    xfs_root,
    xfs_quotactl,
    xfs_statfs,
    xfs_sync,
    xfs_vget,
    xfs_fhtovp,
    xfs_vptofh,
    xfs_init,
    NULL			/* sysctl */
};

int
xfs_install_filesys(void)
{

    struct vfsconf *vfsp;
    struct vfsconf **vfspp;


    /* Check if filesystem already known */
    for (vfspp = &vfsconf, vfsp = vfsconf;
	 vfsp;
	 vfspp = &vfsp->vfc_next, vfsp = vfsp->vfc_next)
	if (strncmp(vfsp->vfc_name,
		    "xfs", MFSNAMELEN) == 0)
	    return (EEXIST);

    /* Allocate and initialize */
    MALLOC(vfsp, struct vfsconf *, sizeof(struct vfsconf),
	   M_MOUNT, M_WAITOK);

    vfsp->vfc_vfsops = &xfs_vfsops;
    strncpy(vfsp->vfc_name, "xfs", MFSNAMELEN);
    vfsp->vfc_typenum = 0;
    vfsp->vfc_refcount = 0;
    vfsp->vfc_flags = 0;
    vfsp->vfc_mountroot = 0;
    vfsp->vfc_next = NULL;

    maxvfsconf++;

    /* Add to the end of the list */
    *vfspp = vfsp;

    /* Call vfs_init() */
    XFSDEB(XDEBVFOPS, ("calling vfs_init\n"));
    (*(vfsp->vfc_vfsops->vfs_init)) (vfsp);

    /* done! */

    return 0;
}

int
xfs_uninstall_filesys(void)
{
    struct vfsconf *vfsp;
    struct vfsconf **vfspp;
    int ret = 0;

    XFSDEB(XDEBLKM, ("xfs_uninstall_filesys\n"));

    /* Find our vfsconf struct */
    for (vfspp = &vfsconf, vfsp = vfsconf;
	 vfsp;
	 vfspp = &vfsp->vfc_next, vfsp = vfsp->vfc_next)
	if (strncmp(vfsp->vfc_name,
		    "xfs",
		    MFSNAMELEN) == 0)
	    break;

    if (!vfsp) {			       /* Not found */
	ret = ENOENT;
	goto out;
    }
      
    if (vfsp->vfc_refcount) {	       /* In use */
	ret = EBUSY;
	goto out;
    }

    /* Remove from list and free  */
    *vfspp = vfsp->vfc_next;
    FREE(vfsp, M_MOUNT);

    maxvfsconf--;

 out:
    return ret;
}

int
xfs_stat_filesys (void)
{
    return 0;
}
