/*
 * Copyright (c) 1995, 1996, 1997 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 *
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_deb.h>


RCSID("$Id: xfs_wrap.c,v 1.4 2000/03/04 18:26:09 map Exp $");

#if defined(__NetBSD__) || defined(__OpenBSD__) || defined(__APPLE__)

#if !defined(__APPLE__)
MOD_MISC("xfs");
#endif

extern struct vnodeopv_desc xfs_vnodeop_opv_desc;

#elif defined(__FreeBSD__)

MOD_MISC(xfs);

#endif

static int
xfs_uninstall(void)
{
    int err, ret = 0;

    if ((err = xfs_uninstall_device()) != 0)
	ret = err;
    if ((err = xfs_uninstall_syscalls()) != 0)
	ret = err;
    if ((err = xfs_uninstall_filesys()) != 0)
	ret = err;

    return ret;
}

static int
xfs_install(void)
{
    int err = 0;

    if ((err = xfs_install_device()) ||
	(err = xfs_install_syscalls()) ||
	(err = xfs_install_filesys()))
	xfs_uninstall();
    return err;
}

static int
xfs_stat(void)
{
    int err = 0;

    if ((err = xfs_stat_device()) != 0)
	return err;
    else if ((err = xfs_stat_syscalls()) != 0)
	return err;
    else if ((err = xfs_stat_filesys()) != 0)
	return err;
    return err;
}

void vfs_opv_init_explicit __P((struct vnodeopv_desc *));
void vfs_opv_init_default __P((struct vnodeopv_desc *));

/*
 * vfs_init.c
 *
 * Allocate and fill in operations vectors.
 *
 * An undocumented feature of this approach to defining operations is that
 * there can be multiple entries in vfs_opv_descs for the same operations
 * vector. This allows third parties to extend the set of operations
 * supported by another layer in a binary compatibile way. For example,
 * assume that NFS needed to be modified to support Ficus. NFS has an entry
 * (probably nfs_vnopdeop_decls) declaring all the operations NFS supports by
 * default. Ficus could add another entry (ficus_nfs_vnodeop_decl_entensions)
 * listing those new operations Ficus adds to NFS, all without modifying the
 * NFS code. (Of couse, the OTW NFS protocol still needs to be munged, but
 * that is a(whole)nother story.) This is a feature.
 */

extern int vfs_opv_numops;

/*
 * Init the vector, if it needs it.
 * Also handle backwards compatibility.
 */
void
vfs_opv_init_explicit(vfs_opv_desc)
	struct vnodeopv_desc *vfs_opv_desc;
{
	int (**opv_desc_vector) __P((void *));
	struct vnodeopv_entry_desc *opve_descp;

	opv_desc_vector = *(vfs_opv_desc->opv_desc_vector_p);

	if (opv_desc_vector == NULL) {
	    opv_desc_vector = xfs_alloc(vfs_opv_numops * sizeof(int (*)(void *)));
	    bzero(opv_desc_vector, vfs_opv_numops * sizeof(int (*)(void *)));
	    *(vfs_opv_desc->opv_desc_vector_p) = opv_desc_vector;
	}

	for (opve_descp = vfs_opv_desc->opv_desc_ops;
	     opve_descp->opve_op;
	     opve_descp++) {
		/*
		 * Sanity check:  is this operation listed
		 * in the list of operations?  We check this
		 * by seeing if its offest is zero.  Since
		 * the default routine should always be listed
		 * first, it should be the only one with a zero
		 * offset.  Any other operation with a zero
		 * offset is probably not listed in
		 * vfs_op_descs, and so is probably an error.
		 *
		 * A panic here means the layer programmer
		 * has committed the all-too common bug
		 * of adding a new operation to the layer's
		 * list of vnode operations but
		 * not adding the operation to the system-wide
		 * list of supported operations.
		 */
		if (opve_descp->opve_op->vdesc_offset == 0 &&
		    opve_descp->opve_op->vdesc_offset != VOFFSET(vop_default)) {
			printf("operation %s not listed in %s.\n",
			    opve_descp->opve_op->vdesc_name, "vfs_op_descs");
			panic ("vfs_opv_init: bad operation");
		}

		/*
		 * Fill in this entry.
		 */
		opv_desc_vector[opve_descp->opve_op->vdesc_offset] =
		    opve_descp->opve_impl;
	}
}

extern int vfs_opv_numops;

void
vfs_opv_init_default(vfs_opv_desc)
	struct vnodeopv_desc *vfs_opv_desc;
{
	int j;
	int (**opv_desc_vector) __P((void *));

	opv_desc_vector = *(vfs_opv_desc->opv_desc_vector_p);

	/*
	 * Force every operations vector to have a default routine.
	 */
	if (opv_desc_vector[VOFFSET(vop_default)] == NULL)
		panic("vfs_opv_init: operation vector without default routine.");

	for (j = 0; j < vfs_opv_numops; j++)
		if (opv_desc_vector[j] == NULL)
			opv_desc_vector[j] = 
			    opv_desc_vector[VOFFSET(vop_default)];
}

/*
 *
 */

extern int (**xfs_vnodeop_p) (void *);

#if defined(__APPLE__)
int xfs_modload()
#else
static int
xfs_modload(struct lkm_table *lkmtp, int cmd)
#endif
{
    int error = 0;

    printf("Loading mod...\n");
    vfs_opv_init_explicit(&xfs_vnodeop_opv_desc);
    vfs_opv_init_default(&xfs_vnodeop_opv_desc);
    error = xfs_install();
    xfs_stat();
    return error;
}

/*
 *
 */

#if defined(__APPLE__)
int xfs_modunload()
#else
static int
xfs_modunload(struct lkm_table * lkmtp, int cmd)
#endif
{
    int error = 0;

    printf("xfs: trying to unload\n");
    error = xfs_uninstall();
    if (!error)
	printf("xfs: unload successful\n");
    else
	printf("xfs: unload unsuccessful, system unstable\n");
    return error;
}

/*
 *
 */

#if defined(__APPLE__)
int xfs_modstat()
#else
static int
xfs_modstat(struct lkm_table * lkmtp, int cmd)
#endif
{
    int error = 0;

    printf("stating\n");
    error = xfs_stat();
    return error;
}

#if !defined(__APPLE__)
int
xfs_mod(struct lkm_table * lkmtp,
	int cmd,
	int ver)
{
#if defined(__NetBSD__) || defined(__OpenBSD__)
    vfs_opv_init_explicit(&xfs_vnodeop_opv_desc);
    vfs_opv_init_default(&xfs_vnodeop_opv_desc);
#endif
    DISPATCH(lkmtp, cmd, ver, xfs_modload, xfs_modunload, xfs_modstat);
}
#endif /* !__APPLE__ */
