/*
 * Copyright (c) 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <windows.h>
#include <xfs_ioctl.h>

#define DEBUG 1

/*
 * This is truly revolting
 */

static SOCKET arla_socket;
static HANDLE xfs_device; 
static int exit_helper = 0;
/*
 *
 */

static void
start_socket (void)
{
    int ret;
    WORD ver_req;
    WSADATA data;
    struct sockaddr_in addr;
    SOCKET listen_socket;

    ver_req = MAKEWORD( 1, 1 );
    ret = WSAStartup(ver_req, &data);
    if (ret) {
	printf ("WSAStartup returned %d\n", GetLastError());
	exit (1);
    }
    
    listen_socket = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (listen_socket == INVALID_SOCKET) {
	printf ("socket returned %d\n", WSAGetLastError());
	exit (1);
    }

    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(0x7f000001); /* 127.0.0.1 */
    addr.sin_port = htons(5000);
    
    ret = bind (listen_socket, (struct sockaddr *) &addr, sizeof(addr));
    if (ret == SOCKET_ERROR) {
	printf ("bind returned %d\n", WSAGetLastError());
	exit(1);
    }

    ret = listen (listen_socket, 1);
    if (ret != 0) {
	printf ("listen returned %d\n", WSAGetLastError());
	exit (1);
    }
    
    ret = sizeof(addr);
    arla_socket = accept (listen_socket, (struct sockaddr *)&addr, &ret);
    if (arla_socket == INVALID_SOCKET) {
	printf ("accept returned %d\n", WSAGetLastError());
	exit (1);
    }
#ifdef DEBUG
    printf ("got a connection from: %s", 
	    inet_ntoa(addr.sin_addr));
#endif
    closesocket (listen_socket);
}

/*
 *
 */

static void
open_device (char *devname)
{
    DWORD threadid;
    int ret;
    
    ret = DefineDosDevice(DDD_RAW_TARGET_PATH, "xfsdev",
			  devname);
    if (!ret) {
	printf ("DefineDosDevice returned %d\n", GetLastError());
	exit (2);
    }
    
    xfs_device = CreateFile ("\\\\.\\xfsdev", 
			     GENERIC_READ|GENERIC_WRITE,
			     FILE_SHARE_READ|FILE_SHARE_WRITE , NULL,
			     OPEN_EXISTING, 0, NULL);
    
    if (xfs_device == INVALID_HANDLE_VALUE) {
	printf ("CreateFile (opening the device) returned %d\n",
		GetLastError());
	exit (2);
    }
}

/*
 *
 */


static DWORD WINAPI 
device_thread(LPVOID foo)
{
    int ret;
    u_long len;
    DWORD ioctl_len;
    char out_len[4];
    char *msg;

#define MAX_XMSG_SIZE (64*1024)

    msg = malloc (MAX_XMSG_SIZE);
    if (msg == NULL) {
	printf ("malloc failed with error: %d\n", GetLastError());
	exit (1);
    }

    while (1) {
	ret = DeviceIoControl (xfs_device,IOCTL_XFS_GETMSG,
			       NULL, 0, msg, sizeof(msg)-1,
			       &ioctl_len,
			       NULL);
	if (!ret) {
	    printf ("DeviceIoControl(GETMSG) returned %d\n",
		    GetLastError());
	    exit_helper = 1;
	    ExitThread(0);
	}
	len = htonl (ioctl_len);
	memcpy (out_len, &len, sizeof(out_len));
	ret = send (arla_socket, out_len, sizeof(out_len), 0);
	if (ret) {
	    printf ("send(out_len) returned %d\n", WSAGetLastError());
	    exit_helper = 1;
	    ExitThread(0);
	}
	ret = send (arla_socket, msg, ioctl_len, 0);
	if (ret) {
	    printf ("send(msg) returned %d\n", WSAGetLastError());
	    exit_helper = 1;
	    ExitThread(0);
	}
    }
}

/*
 *
 */

static void
loop (void)
{
    fd_set rfd;
    int ret;
    char *msg;

    msg = malloc (MAX_XMSG_SIZE);
    if (msg == NULL) {
	printf ("malloc failed with error: %d\n", GetLastError());
	exit (1);
    }

    while (!exit_helper) {
	FD_ZERO (&rfd);
	FD_SET (arla_socket, &rfd);
	
	ret = select (arla_socket + 1, &rfd, NULL, NULL, NULL);
	if (ret == SOCKET_ERROR) {
	    printf ("select returned %d\n", ret);
	    return;
	}

	if (FD_ISSET(arla_socket, &rfd)) {
	    u_long len;
	    DWORD ioctl_len;
	    char out_len[4];

	    ret = recv (arla_socket, out_len, sizeof(out_len), 0);
	    if (ret == SOCKET_ERROR) {
		printf ("recv (out_len) returned %d\n",
			WSAGetLastError());
		return;
	    }
	    memcpy (&len, out_len, sizeof(len));
	    len = ntohl(len);

	    if (len > MAX_XMSG_SIZE) {
		printf ("recv a too large message (%d)\n", len);
		continue;
	    }

	    ret = recv (arla_socket, out_len, sizeof(out_len), 0);
	    if (ret == SOCKET_ERROR) {
		printf ("recv (msg) returned(SOCKET_ERROR) %d\n",
			WSAGetLastError());
		return;
	    } else if (ret != len) {
		printf ("recv (msg) of incorrect size\n");
		return;
	    }

	    ret = DeviceIoControl (xfs_device, IOCTL_XFS_PUTMSG,
				   msg, len, NULL, 0,
				   &ioctl_len,
				   NULL);
	    if (!ret) {
		printf ("DeviceIoControl(PUTMSG) returned %d\n",
			GetLastError());
		return;
	    }
	    
	} else {
	    printf ("select behavies strange\n");
	}
	
    }
}

/*
 *
 */

int
main (int argc, char **argv)
{
    int ret;
    DWORD nread;
    char *msg;
    DWORD tid;
    HANDLE thread;

    open_device ("\\Device\\XFS");
    start_socket ();

    thread = CreateThread (NULL, 0, device_thread, NULL, 0, &tid);
    if (thread == INVALID_HANDLE_VALUE) {
	printf ("CreateThread failed with %d\n", GetLastError());
	return 0;
    } 

    loop();

    CloseHandle (xfs_device);
#if 0
    closesocket (arla_socket);
#endif
    return 0;
}
