/*
 * Copyright (c) 1999, 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* $Id: xfs_misc.c,v 1.2 2000/05/08 05:20:51 lha Exp $ */

#include <xfs_locl.h>

/*
 * Allocate a ccb structure for either a zone or non-paged pool.
 * Set approprivate flags.
 */

XFS_CCB *
xfs_get_ccb (void)
{
    XFS_CCB *ccb = NULL;
    BOOLEAN zonep = TRUE;
    KIRQL   iqrl;
    
    KeAcquireSpinLock(&XFSGlobalData.ZoneAllocationSpinLock, &iqrl);

    if (!ExIsFullZone(&XFSGlobalData.CCBZoneHeader)) {
	ccb = ExAllocateFromZone (&XFSGlobalData.CCBZoneHeader);
	KeReleaseSpinLock(&XFSGlobalData.ZoneAllocationSpinLock, iqrl);
    } else {
	KeReleaseSpinLock(&XFSGlobalData.ZoneAllocationSpinLock, iqrl);

	ccb = ExAllocatePool(NonPagedPool, AlignPointer(sizeof(*ccb)));
	zonep = FALSE;
    }

    if (ccb == NULL)
	XFSPanic (STATUS_INSUFFICIENT_RESOURCES, AlignPointer(sizeof(*ccb)), 0);
    
    RtlZeroMemory(ccb, AlignPointer(sizeof(*ccb)));
    
    if (!zonep)
	XFS_SETFLAGS(ccb->flags, XFS_CCB_NOT_FROM_ZONE);
    
    return ccb;
}

/*
 * Free ccb structure
 */

void
xfs_release_ccb (XFS_CCB *ccb)
{
    KIRQL iqrl;

    ASSERT(ccb);

    if (! XFS_TESTFLAGS(ccb->flags, XFS_CCB_NOT_FROM_ZONE)) {
	KeAcquireSpinLock(&XFSGlobalData.ZoneAllocationSpinLock, &iqrl);
	
	ExFreeToZone(&XFSGlobalData.CCBZoneHeader, ccb);

	KeReleaseSpinLock(&XFSGlobalData.ZoneAllocationSpinLock, iqrl);
    } else {
	ExFreePool(ccb);
    }
}

/*
 *
 */

struct xfs_link *
xfs_alloc_link (struct xfs_channel *chan)
{
    struct xfs_link *link = NULL;
    BOOLEAN zonep = TRUE;
    KIRQL   iqrl;
    
    KeAcquireSpinLock(&chan->ZoneAllocationSpinLock, &iqrl);

    if (!ExIsFullZone(&chan->LinkZoneHeader)) {
	link = ExAllocateFromZone (&chan->LinkZoneHeader);
	KeReleaseSpinLock(&chan->ZoneAllocationSpinLock, iqrl);
    } else {
	KeReleaseSpinLock(&chan->ZoneAllocationSpinLock, iqrl);

	link = ExAllocatePool(NonPagedPool, AlignPointer(sizeof(*link)));
	zonep = FALSE;
    }

    if (link == NULL)
	XFSPanic (STATUS_INSUFFICIENT_RESOURCES, AlignPointer(sizeof(*link)), 0);
    
    RtlZeroMemory(link, AlignPointer(sizeof(*link)));
    
    if (!zonep)
	XFS_SETFLAGS(link->flags, XFS_LINK_NOT_FROM_ZONE);
    
    return link;
}

/*
 *
 */

void
xfs_free_link (struct xfs_channel *chan, struct xfs_link *link)
{
    KIRQL iqrl;

    ASSERT(link && chan);

    if (! XFS_TESTFLAGS(link->flags, XFS_LINK_NOT_FROM_ZONE)) {
	KeAcquireSpinLock(&chan->ZoneAllocationSpinLock, &iqrl);
	
	ExFreeToZone(&chan->LinkZoneHeader, link);

	KeReleaseSpinLock(&chan->ZoneAllocationSpinLock, iqrl);
    } else {
	ExFreePool(link);
    }
}

/*
 *
 */

void *
xfs_alloc (size_t size)
{
    return ExAllocatePool (NonPagedPool, size);
}

/*
 *
 */

void
xfs_free (void *ptr, size_t size)
{
    ExFreePool (ptr);
}
