/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_vfsops.h>
#include <xfs/xfs_msg_locl.h>
#include <xfs/xfs_dev.h>

RCSID("$Id: xfs_message.c,v 1.26.2.2 2001/02/14 20:32:28 lha Exp $");

int
xfs_message_installroot(int fd,
			struct xfs_message_installroot *message,
			u_int size)
{
    int error = 0;

    XFSDEB(XDEBMSG, ("xfs_message_installroot\n"));

    if (xfs[fd].root != 0)
    {
	printf("XFS PANIC Warning: xfs_message_installroot again\n");
	error = EBUSY;
    }
    else
    {
	xfs[fd].root = new_xfs_node(&xfs[fd], NULL, &message->node); /*VN_HOLD's*/
	xfs[fd].root->vn.v_flag |= VROOT;
	mutex_exit(&xfs[fd].root->node_lock);
    }
    return error;
}

int
xfs_message_installnode(int fd,
			struct xfs_message_installnode *message,
			u_int size)
{
    int error = 0;
    struct xfs_node *n, *dp;

    XFSDEB(XDEBMSG, ("xfs_message_installnode\n"));

    dp = xfs_node_find(&xfs[fd], &message->parent_handle);
    if (dp) 
    {
	/* XXXSMP here it could be so that we lock child->parent
	 * and that might not be a good idea */
	n = new_xfs_node(&xfs[fd], dp, &message->node); /* VN_HOLD's */
	dnlc_remove (XNODE_TO_VNODE(dp), message->name);
	xfs_dnlc_enter(XNODE_TO_VNODE(dp), message->name, XNODE_TO_VNODE(n));
	if (dp != n) /* in case of "." */
	    mutex_exit(&dp->node_lock);
	mutex_exit(&n->node_lock);
	VN_RELE(XNODE_TO_VNODE(n));
    }
    else
    {
	printf("XFS PANIC Warning: xfs_message_install could not find parent\n");
	error = ENOENT;
    }
    return error;
}

int
xfs_message_installattr(int fd,
			struct xfs_message_installattr *message,
			u_int size)
{
    int error = 0;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_installattr\n"));

    t = xfs_node_find(&xfs[fd], &message->node.handle);
    if (t != 0)
    {
	t->tokens = message->node.tokens;
	xfs_attr2vattr(&message->node.attr, &t->attr, 0);
	bcopy((caddr_t)message->node.id,
	      (caddr_t)t->id, sizeof(t->id));
	bcopy((caddr_t)message->node.rights,
	      (caddr_t)t->rights, sizeof(t->rights));
	t->anonrights = message->node.anonrights;
	mutex_exit(&t->node_lock);
    }
    else
    {
	XFSDEB(XDEBMSG, ("xfs_message_installattr: no such node\n"));
    }
    return error;
}

int
xfs_message_installdata(int fd,
			struct xfs_message_installdata *message,
			u_int size)
{
    struct xfs_node *t;
    struct vnode *vp;
    int error = 0;
    
    XFSDEB(XDEBMSG, ("xfs_message_installdata\n"));
    
    t = xfs_node_find(&xfs[fd], &message->node.handle);
    if (t != 0) 
    {
	struct xfs_fh_args *fh_args = 
	    (struct xfs_fh_args *)&message->cache_handle;
	
	VN_HOLD(XNODE_TO_VNODE(t));
	message->cache_name[sizeof(message->cache_name)-1] = '\0';
	XFSDEB(XDEBMSG, ("cache_name = '%s'\n",
			 message->cache_name));
	
	if (message->flag & XFS_ID_HANDLE_VALID) {
	    error = xfs_fhlookup (fh_args->fsid, fh_args->fid, &vp);
	} else {
	    error = lookupname(message->cache_name, UIO_SYSSPACE,
			       NO_FOLLOW, NULL, &vp);
	}
	if (error == 0) {
	    if (DATA_FROM_XNODE(t)) {
		VN_RELE(DATA_FROM_XNODE(t));
	    }
	    if (XNODE_TO_VNODE(t)->v_type == VDIR
		&& (message->flag & XFS_ID_INVALID_DNLC)) {
		dnlc_purge_vp(XNODE_TO_VNODE(t));
	    }
	    ASSERT(vp != NULL);
	    DATA_FROM_XNODE(t) = vp;
	    t->tokens = message->node.tokens;
	    xfs_attr2vattr(&message->node.attr, &t->attr, 1);
	    bcopy((caddr_t)message->node.id,
		  (caddr_t)t->id, sizeof(t->id));
	    bcopy((caddr_t)message->node.rights,
		  (caddr_t)t->rights, sizeof(t->rights));
	    t->anonrights = message->node.anonrights;
	} else {
	    printf("XFS PANIC Warning: xfs_message_installdata failed to "
		   "lookup cache file = %s, error = %d\n",
		   message->cache_name, error);
	}
	mutex_exit(&t->node_lock);
	VN_RELE(XNODE_TO_VNODE(t));
    } else {
	printf("XFS PANIC Warning: "
	       "xfs_message_installdata didn't find node!\n");
	error = ENOENT;
    }
    
    return error;
}

int
xfs_message_invalidnode(int fd,
			struct xfs_message_invalidnode *message,
			u_int size)
{
    int error = 0;
    struct xfs_node *t;
  
    XFSDEB(XDEBMSG, ("xfs_message_invalidnode\n"));

    t = xfs_node_find(&xfs[fd], &message->handle);
    if (t != 0)
    {
	struct vnode *vp;
	/* XXX Really need to put back dirty data first. */
	if (DATA_FROM_XNODE(t))
	{
	    VN_RELE(DATA_FROM_XNODE(t));
	    DATA_FROM_XNODE(t) = (struct vnode *) 0;
	}
	XFS_TOKEN_CLEAR(t, ~0,
			XFS_OPEN_MASK | XFS_ATTR_MASK |
			XFS_DATA_MASK | XFS_LOCK_MASK);
	vp = XNODE_TO_VNODE(t);
	mutex_exit(&t->node_lock);
	dnlc_purge_vp (vp);
    } 
    else
    {
#if 0
	printf("XFS PANIC Warning: xfs_message_invalidnode didn't find node!\n");
#endif
	error = ENOENT;
    }
    return error;
}

int
xfs_message_updatefid(int fd,
		      struct xfs_message_updatefid *message,
		      u_int size)
{
    int error = 0;
    struct xfs_node *t;
  
    XFSDEB(XDEBMSG, ("xfs_message_updatefid\n"));

    t = xfs_node_find(&xfs[fd], &message->old_handle);
    if (t != 0)
    {
	t->handle = message->new_handle;
	mutex_exit(&t->node_lock);
    }
    else
    {
	printf("XFS PANIC Warning: xfs_message_updatefid didn't find node!\n");
	error = ENOENT;
    }
    return error;
}

static void
gc_vnode (struct vnode *vp)
{
    mutex_enter (&vp->v_lock);
    if (vp->v_count <= 1)
    {
	mutex_exit (&vp->v_lock);
	XFSDEB(XDEBMSG, ("xfs_message_gc: try\n"));
	dnlc_purge_vp(vp);
    } 
    else 
    {
	int count = vp->v_count;
	mutex_exit (&vp->v_lock);
	XFSDEB(XDEBMSG, ("xfs_message_gc: used (%d)\n", count));
    }
}

int
xfs_message_gc_nodes(int fd,
		     struct xfs_message_gc_nodes *message,
		     u_int size)
{
    struct xfs *xfsp = &xfs[fd];
    XFSDEB(XDEBMSG, ("xfs_message_gc\n"));

    if (message->len == 0) {
	struct xfs_node *t;
	
	for (t = xfs_node_iter_start(xfsp);
	     t != NULL; 
	     t = xfs_node_iter_next(xfsp))
	{
	    struct vnode *vp = XNODE_TO_VNODE(t);
	    gc_vnode (vp);
	}

    } else {
	int i;
	
	for (i = 0; i < message->len; i++) {
	    struct xfs_node *t;
	    struct vnode *vp;

	    t = xfs_node_find (xfsp, &message->handle[i]);
	    if (t == NULL)
		continue;
	    vp = XNODE_TO_VNODE(t);
	    mutex_exit(&t->node_lock);

	    gc_vnode(vp);
	}
    }

    return 0;
}


/*
 * Probe what this xfs support
 */

int
xfs_message_version(int fd,
		    struct xfs_message_version *message,
		    u_int size)
{
    struct xfs_message_wakeup msg;
    int ret;

    ret = XFS_VERSION;

    msg.header.opcode = XFS_MSG_WAKEUP;
    msg.sleepers_sequence_num = message->header.sequence_num;
    msg.error = ret;

    return xfs_message_send(fd, (struct xfs_message_header *) &msg, sizeof(msg));
}
