import string, re, sys

sitelib = '::LIBDIR::'
sys.path.append(sitelib)
from afs_utils import FullPartitionName
import afs_voslistvol, afs_vosexamine, afs_paths, afs_utils

class QuotaPartinfoEntry:

    def __init__(self, partition, kfree, ktotaldisk, kuncommitted):
        self.partition = partition
        self.kfree = long(kfree)
        self.ktotaldisk = long(ktotaldisk)
        self.kuncommitted = long(kuncommitted)

    def getKFree(self):
        return self.kfree

    def getPartition(self):
        return self.partition

    def getKUncommitted(self):
        return self.kuncommitted

    def getKTotalDisk(self):
        return self.ktotaldisk


class QuotaPartinfoData:

    def __init__(self, server, partition=''):
        """ Represent something like 'vos partinfo' but show K uncommitted
            instead of K free on disk.

            * server - An AFS file server hostname
            * partition - A partition on server (optional)
        """
        self.server = server
        self.partition = partition
        self.entries = []

    def load(self):
        """ Load the data based on the info passed to the constructor
            when the object was instantiated.

            Returns 0 on failure, 1 on successful load and parse
        """
        data = self._loadraw()
        if not data:
            return 0

        whole_re = re.compile('^Free space on partition\s+(.+):\s+(\d+) K blocks out of total\s+(\d+)')

        for rawline in data:
            if rawline == '\n':
                continue
            match = whole_re.match(rawline)
            if match != None:
                (part, kfree, ktotal) = match.groups()
                kfree = long(kfree)
                ktotal = long(ktotal)

                # Now compute the K committed
                vlvdata = afs_voslistvol.VosListvolData(self.getServer(), part)
                vlvdata.load()
                kcommitted = 0L
                for entry in vlvdata.getEntries():
                    # Only add up RO and RW vols as BK vols take little space
                    if entry.getVType() != "BK":
                        vname = entry.getVName()
                        vedata = afs_vosexamine.VosExamineData(vname)
                        vedata.load()
                        q = vedata.getMaxQuota()
                        kcommitted = kcommitted + q

                kuncommitted = ktotal - kcommitted
                entry = QuotaPartinfoEntry(part, kfree, ktotal, kuncommitted)
                self.addEntry(entry)
                continue

        return 1

    def _loadraw(self):
        """ Call 'vos partinfo' and actually load the data we need.  Called by
            load().  Do not call this yourself. 
        """
        cmd = afs_paths.AFS_vosCMD + ' partinfo ' + self.getServer()
        if self.getPartition():
            cmd = cmd + ' ' + self.getPartition()
        return afs_utils.RunCommandReturnOutput(cmd)

    def printReport(self):
        """ Print an ASCII report to standard output """
        for e in self.getEntries():
            partition = FullPartitionName(e.getPartition())
            print "Uncommitted space on partition %s: %d K blocks out of total %d" % (partition, e.getKUncommitted(), e.getKTotalDisk())


    def getServer(self):
        return self.server

    def getPartition(self):
        return self.partition


    def addEntry(self, entry):
        self.entries.append(entry)

    def getEntries(self):
        return self.entries

if __name__ == '__main__':
    qpidata = QuotaPartinfoData('bunky', 'c')
    qpidata.load()
    qpidata.printReport()

#-----------------------------------------------------------------------------#
#                (c) Copyright 2000 The MITRE Corporation                     #
#-----------------------------------------------------------------------------#
