/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */
/* Balsa E-Mail Client
 *
 * Copyright (C) 1997-2002 Stuart Parmenter and others,
 *                         See the file AUTHORS for a list.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option) 
 * any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

/*
 * A external source (program opened via popen) address book
 */

#include "config.h"

#include <gnome.h>

#include <stdio.h>
#include <sys/stat.h>

#include "address-book.h"
#include "address-book-extern.h"
#include "information.h"
#include "abook-completion.h"

/* FIXME: Arbitrary constant */
#define LINE_LEN 256

static GtkObjectClass *parent_class = NULL;

static void libbalsa_address_book_externq_class_init(LibBalsaAddressBookExternClass *klass);
static void libbalsa_address_book_externq_init(LibBalsaAddressBookExtern *ab);
static void libbalsa_address_book_externq_destroy(GtkObject * object);

static void libbalsa_address_book_externq_load(LibBalsaAddressBook * ab, 
					       LibBalsaAddressBookLoadFunc 
                                               callback, 
                                               gpointer closure);
static void libbalsa_address_book_externq_store_address(LibBalsaAddressBook *ab,
                                                        LibBalsaAddress *new_address);

static void libbalsa_address_book_externq_save_config(LibBalsaAddressBook *ab,
                                                      const gchar * prefix);
static void libbalsa_address_book_externq_load_config(LibBalsaAddressBook *ab,
                                                      const gchar * prefix);
static void load_externq_file(LibBalsaAddressBook *ab);

static GList *parse_externq_file(LibBalsaAddressBookExtern *addr_externq, gchar *pattern);

static GList *libbalsa_address_book_externq_alias_complete(LibBalsaAddressBook *ab, 
                                                           const gchar * prefix,
                                                           gchar ** new_prefix);

GtkType libbalsa_address_book_externq_get_type(void)
{
    static GtkType address_book_externq_type = 0;

    if (!address_book_externq_type) {
	static const GtkTypeInfo address_book_externq_info = {
	    "LibBalsaAddressBookExtern",
	    sizeof(LibBalsaAddressBookExtern),
	    sizeof(LibBalsaAddressBookExternClass),
	    (GtkClassInitFunc) libbalsa_address_book_externq_class_init,
	    (GtkObjectInitFunc) libbalsa_address_book_externq_init,
	    /* reserved_1 */ NULL,
	    /* reserved_2 */ NULL,
	    (GtkClassInitFunc) NULL,
	};

	address_book_externq_type =
	    gtk_type_unique(libbalsa_address_book_get_type(),
			    &address_book_externq_info);
    }

    return address_book_externq_type;

}

static void
libbalsa_address_book_externq_class_init(LibBalsaAddressBookExternClass *
                                         klass)
{
    LibBalsaAddressBookClass *address_book_class;
    GtkObjectClass *object_class;

    parent_class = gtk_type_class(LIBBALSA_TYPE_ADDRESS_BOOK);

    object_class = GTK_OBJECT_CLASS(klass);
    address_book_class = LIBBALSA_ADDRESS_BOOK_CLASS(klass);

    object_class->destroy = libbalsa_address_book_externq_destroy;

    address_book_class->load = libbalsa_address_book_externq_load;
    address_book_class->store_address =
	libbalsa_address_book_externq_store_address;

    address_book_class->save_config =
	libbalsa_address_book_externq_save_config;
    address_book_class->load_config =
	libbalsa_address_book_externq_load_config;

    address_book_class->alias_complete =
	libbalsa_address_book_externq_alias_complete;

}

static void
libbalsa_address_book_externq_init(LibBalsaAddressBookExtern * ab)
{
    ab->load = NULL;
    ab->save = NULL;
    ab->address_list = NULL;
    ab->mtime = 0;

    ab->name_complete  = 
	g_completion_new((GCompletionFunc)completion_data_extract);
    ab->alias_complete = 
	g_completion_new((GCompletionFunc)completion_data_extract);
}

static void
libbalsa_address_book_externq_destroy(GtkObject * object)
{
    LibBalsaAddressBookExtern *addr_externq;

    addr_externq = LIBBALSA_ADDRESS_BOOK_EXTERN(object);

    g_free(addr_externq->load);
    g_free(addr_externq->save);
	
    g_list_foreach(addr_externq->address_list, (GFunc) gtk_object_unref, NULL);
    g_list_free(addr_externq->address_list);
    addr_externq->address_list = NULL;

    if (GTK_OBJECT_CLASS(parent_class)->destroy)
	(*GTK_OBJECT_CLASS(parent_class)->destroy) (GTK_OBJECT(object));

}

LibBalsaAddressBook *
libbalsa_address_book_externq_new(const gchar * name, const gchar * load,
                                  const gchar *save)
{
    LibBalsaAddressBookExtern *abvc;
    LibBalsaAddressBook *ab;

    abvc = gtk_type_new(LIBBALSA_TYPE_ADDRESS_BOOK_EXTERN);
    ab = LIBBALSA_ADDRESS_BOOK(abvc);

    ab->name = g_strdup(name);
    abvc->load = g_strdup(load);
    abvc->save = g_strdup(save);

    return ab;
}

static void
libbalsa_address_book_externq_load(LibBalsaAddressBook * ab, 
                                   LibBalsaAddressBookLoadFunc callback, 
                                   gpointer closure)
{
    GList *lst;

    load_externq_file(ab);

    for (lst = LIBBALSA_ADDRESS_BOOK_EXTERN(ab)->address_list; 
	 lst; lst = g_list_next(lst)) {
	if ( callback ) 
	    callback(ab, LIBBALSA_ADDRESS(lst->data), closure);
    }
    callback(ab, NULL, closure);
}

static void
load_externq_file(LibBalsaAddressBook *ab)
{
    LibBalsaAddressBookExtern *addr_externq;
    addr_externq = LIBBALSA_ADDRESS_BOOK_EXTERN(ab);
	
    /* Erase the current address list */
    g_list_foreach(addr_externq->address_list, (GFunc) gtk_object_unref, NULL);
    g_list_free(addr_externq->address_list);
    addr_externq->address_list = parse_externq_file(addr_externq, " ");
}

static GList*
parse_externq_file(LibBalsaAddressBookExtern *addr_externq, gchar *pattern)
{
    FILE *gc;
    gchar string[LINE_LEN];
    char name[LINE_LEN], email[LINE_LEN], tmp[LINE_LEN];
    gchar command[LINE_LEN];
    GList *list = NULL;
    GList *address_list = NULL;

    /* Start the program */
    g_snprintf(command, sizeof(command), "%s \"%s\"", 
               addr_externq->load, pattern);
    
    gc = popen(command,"r");

    if (gc == NULL) {
        libbalsa_information(LIBBALSA_INFORMATION_WARNING, 
                             _("Could not open external query address book %s "
                               "while trying to parse output from: %s"), 
                             addr_externq->parent.name, command); 
        return NULL;
    }
    fgets(string, sizeof(string), gc);
    /* The first line should be junk, just debug output */
#ifdef DEBUG
    printf("%s\n", string);
#endif
	
    while (fgets(string, sizeof(string), gc)) {
        LibBalsaAddress *address;
#ifdef DEBUG
        printf("%s\n", string);
#endif
        if(sscanf(string, "%[^\t]\t%[^\t]%[^\n]", email, name, tmp)<2)
            continue;
#ifdef DEBUG
        printf("%s,%s,%s\n",email,name,tmp);
#endif
        address = libbalsa_address_new();
        /* The externq database doesn't support Id's, sorry! */
        address->id = g_strdup(_("No-Id"));
        address->address_list = g_list_append(address_list, g_strdup(email));

        if (name)address->full_name = g_strdup(name);
        else address->full_name = g_strdup(_("No-Name"));
        list = g_list_prepend(list,address);
    }
    fclose(gc);
    
    list = g_list_sort(list, (GCompareFunc)address_compare);
    return list;
}

static void
libbalsa_address_book_externq_store_address(LibBalsaAddressBook * ab,
                                            LibBalsaAddress * new_address)
{
    gchar command[LINE_LEN];
    LibBalsaAddressBookExtern *ex;
    FILE *gc; 
    g_return_if_fail(LIBBALSA_IS_ADDRESS_BOOK_EXTERN(ab));

    ex = LIBBALSA_ADDRESS_BOOK_EXTERN(ab);

    g_snprintf(command, sizeof(command), "%s \"%s\" \"%s\" \"%s\"", 
               ex->save, 
               (gchar *)g_list_first(new_address->address_list)->data, 
               new_address->full_name, "TODO");

    gc = popen(command, "r");
    if(gc == NULL) {
        libbalsa_information(LIBBALSA_INFORMATION_WARNING, 
                             _("Could not add address to address book %s "
                               "while trying to execute: %s"), 
                             ex->parent.name, command); 
        return;
    }
    fclose(gc);
}

static void
libbalsa_address_book_externq_save_config(LibBalsaAddressBook * ab,
                                          const gchar * prefix)
{
    LibBalsaAddressBookExtern *vc;

    g_return_if_fail(LIBBALSA_IS_ADDRESS_BOOK_EXTERN(ab));

    vc = LIBBALSA_ADDRESS_BOOK_EXTERN(ab);

    gnome_config_set_string("Load", vc->load);
    gnome_config_set_string("Save", vc->save);

    if (LIBBALSA_ADDRESS_BOOK_CLASS(parent_class)->save_config)
	LIBBALSA_ADDRESS_BOOK_CLASS(parent_class)->save_config(ab, prefix);
}

static void
libbalsa_address_book_externq_load_config(LibBalsaAddressBook * ab,
                                          const gchar * prefix)
{
    LibBalsaAddressBookExtern *vc;

    g_return_if_fail(LIBBALSA_IS_ADDRESS_BOOK_EXTERN(ab));

    vc = LIBBALSA_ADDRESS_BOOK_EXTERN(ab);

    g_free(vc->load);
    vc->load = gnome_config_get_string("Load");
    vc->save = gnome_config_get_string("Save");

    if (LIBBALSA_ADDRESS_BOOK_CLASS(parent_class)->load_config)
	LIBBALSA_ADDRESS_BOOK_CLASS(parent_class)->load_config(ab, prefix);
}

static GList*
libbalsa_address_book_externq_alias_complete(LibBalsaAddressBook * ab,
                                             const gchar * prefix, 
                                             gchar ** new_prefix)
{
    LibBalsaAddressBookExtern *ex;
    GList *res = NULL;
    *new_prefix = NULL;

    g_return_val_if_fail(LIBBALSA_IS_ADDRESS_BOOK_EXTERN(ab), NULL);

    ex = LIBBALSA_ADDRESS_BOOK_EXTERN(ab);

    if ( ab->expand_aliases == FALSE )
	return NULL;

    res = parse_externq_file(ex, (gchar *)prefix);
	
    g_list_reverse(res);

    if(res != NULL)
        *new_prefix = libbalsa_address_to_gchar((LibBalsaAddress *)
                                                g_list_first(res)->data, 0);

    return res;
}
