/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <string.h>
#include <netdb.h>
#include <sys/stat.h>
#include <dirent.h>
#include "cheops.h"

#ifdef HAS_GMODULE
#include <gmodule.h>
#else
#ifdef HAS_DLOPEN
#include <dlfcn.h>
#else
#error "Must have dlopen() or g_module_open()"
#endif
#endif

static char *service_syms[] = { "service_connect", NULL };

static char *monitor_syms[] = { "default_level", "monitor", "cleanup", "setup", "setup_cleanup", "setup2str","str2setup", "timeout_callback", NULL };

static char *probe_syms[] = { "getport", "probe", NULL  };

static struct plugin *plist=NULL;

void *get_plugin(char *name, int type, int sym)
{
	void *symbol=NULL;
	struct plugin *p;
	if (sym < MAX_SYMS) {
		p=plist;
		while(p) {
			if (!strcmp(p->name, name)) 
				symbol = p->syms[sym];
			p=p->next;
		}
	}
	return symbol;
}

char *str_plugin(void *data, int type, int sym)
{
	struct plugin *p;
	char *c = NULL;
	if (sym < MAX_SYMS) {
		p=plist;
		while(p) {
			if (p->syms[sym] == data)
				c = p->name;
			p=p->next;
		}
	}
	return c;
}

char *label_plugin(char *name)
{
	struct plugin *p;
	char *res=NULL;
	p = plist;
	while(p) {
		if (!strcmp(name, p->name)) {
			res=p->label;
		}
		p=p->next;
	}
	return res;
}

GList *list_plugin(int type)
{
	struct plugin *p;
	GList *l = NULL;
	p = plist;
	while(p) {
		if (p->type == type)
			l = g_list_append(l, p->name);
		p=p->next;
	}
	return l;
}

#ifdef HAS_GMODULE
#define LOAD_MODULE(handle,string, symbol) \
		g_module_symbol(handle, string, (void *)&(symbol))
#define CLOSE_MODULE(mod) g_module_close(mod)
#define ERROR_MODULE g_module_error()
#else
#define LOAD_MODULE(handle,string, symbol) \
		{ symbol = (__typeof__(symbol))(dlsym(handle, string)); }
#define CLOSE_MODULE(mod) dlclose(mod)
#define ERROR_MODULE dlerror()
#endif
static int load_plugin(char *fn)
{
	int (*get_type)();
	char *(*get_name)();
	char *(*get_label)();
	char *c, *l;
	char **syms=NULL;
	int x;
	char *s;
	int t;
	struct plugin *p;
#ifdef HAS_GMODULE
	GModule *mod;
	mod = g_module_open(fn, 0);
#else
	void *mod;
	mod = dlopen(fn, 0);
#endif
	if (!mod) {
		fprintf(stderr, "Error loading plugin '%s': %s\n", fn, ERROR_MODULE);
		return 0;
	};
	LOAD_MODULE(mod, "plugin_name", get_name);
	if (!get_name) {
		fprintf(stderr, "Plugin file %s does not export plugin_name()\n", fn);
		CLOSE_MODULE(mod);
		return 0;
	}
	c = get_name();
	if (!c || !strlen(c)) {
		fprintf(stderr, "Plugin file %s provided an empty name\n", fn);
		CLOSE_MODULE(mod);
		return 0;
	}
	LOAD_MODULE(mod, "plugin_label", get_label);
	if (!get_label) {
		fprintf(stderr, "Plugin %s (%s) does not export plugin_label()\n", c, fn);
		CLOSE_MODULE(mod);
		return 0;
	}
	l = get_label();
	LOAD_MODULE(mod, "plugin_type", get_type);
	if (!get_type) {
		fprintf(stderr, "Plugin %s (%s) does not export plugin_type()\n", c, fn);
		CLOSE_MODULE(mod);
		return 0;
	}
	t = get_type();
	if (get_plugin(c, t, 0)) {
		fprintf(stderr, "Ignoring duplicate module %s\n", c);
		CLOSE_MODULE(mod);
		return 0;
	}
	switch(t) {
	case TYPE_SERVICE:
		syms = service_syms;
		break;
	case TYPE_MONITOR:
		syms = monitor_syms;
		break;
	case TYPE_PROBE:
		syms = probe_syms;
		break;
	case TYPE_SNMP:
	default:
		fprintf(stderr, "Don't know how to handle plugin of type %d\n", t);
		CLOSE_MODULE(mod);
		return 0;
	}
	p = g_new0(struct plugin, 1);
	memset(p, 0, sizeof(struct plugin));
	p->type = t;
	strncpy(p->name, c, sizeof(p->name));
	strncpy(p->label, l, sizeof(p->label));
	x=0;
	do {
		s=syms[x];
		if (s) {
			LOAD_MODULE(mod, s, p->syms[x]);
#if 0
			if (!p->syms[x])
				fprintf(stderr, "Warning: module %s has no function %s\n", c, s);
#endif
			x++;
		}
	} while(s);
#ifdef DEBUG
	printf("loaded module %s, type %d\n", p->name, p->type);
#endif
	p->next = plist;
	plist = p;
	return 1;	
}

static int load_path(char *path)
{
	DIR *dir;
	struct dirent *file;
	int res;
	int cnt=0;
	char buf[256];
#ifdef DEBUG
	printf("Loading modules from %s\n", path);
#endif
	dir = opendir(path);
	if (dir) {
		while((file = readdir(dir))) {
			if (strlen(file->d_name) > 3){
				g_snprintf(buf, sizeof(buf), "%s/%s", path, file->d_name);
				res = load_plugin(buf);
				if (res)
					cnt++;
			}
		};
	}
	return cnt;
}

int load_plugins()
{
	int cnt=0;
	char buf[256];
	char *c;
	if ((c = getenv("HOME"))) {
		g_snprintf(buf, sizeof(buf), "%s/.cheops/plugins", c);
		cnt += load_path(buf);
	}
	load_path(LIB_PATH "/plugins");
	return cnt;
}

