#define GDBM_BLOCK_SIZE 2048

/* Argument processing */
extern int pw_args ();		/* Parses argc/argv and returns pw-handle */
extern char *pw_getpwarg ();

/* Error codes */
enum pw_errcode
{
    PW_EOK = 0,		/* No error */
    PW_ELINETOOLONG,	/* Line to insert is too long */
    PW_ENEWSYNTAX,	/* Line to insert contains a syntax error */
    PW_EOLDSYNTAX,	/* Line to update contains a syntax error */
    PW_EFSFAIL,		/* Some other file system related problem */
    PW_ELOCKED,		/* The password file is locked */
    PW_ENOTFOUND,	/* Relevant entry not found */
    PW_EUIDEXISTS,	/* An entry with this uid already exists */
    PW_ENAMEEXISTS,	/* An entry with this logname already exists */
    PW_EBADPASSWD,	/* The password file is corrupt */
    PW_ENOTSORTED,	/* The password file is not sorted */
    PW_EOPEN,		/* Failed to open the password file */
    PW_ECREAT,		/* Failed to create lockfile */
    PW_EREAD,		/* Read error */
    PW_EWRITE,		/* Write error */
    PW_EHOST,		/* Unknown host */
    PW_ESOCKET,		/* Can't create communication socket(2) */
    PW_EBIND,		/* Can't bind(2) to server */
    PW_ECONNECT,	/* Can't connect(2) to server */
    PW_ESEND,		/* Can't send(2) request to server */
    PW_ERECV,		/* Can't recv(2) reply from server */
    PW_EROOT,		/* Disallowed manipulation of root-entry */
    PW_ETOOMANY,	/* Too many active descriptors already */
    PW_EDESCRIPTOR,	/* Bad descriptor */
    PW_ENOTACTIVE,	/* Descriptor is not active */
    PW_EAUTHENTICATOR,	/* Kerberos authenticator not valid */
    PW_EREQUEST,	/* Request was corrupt at server*/
    PW_EREPLY,		/* Reply from server is corrupt */
    PW_ETICKET,		/* Can't create ticket */
    PW_ECREDENTIALS,	/* Can't read credentials */
    PW_EIMPL,		/* Requested operation is not implemented */
    PW_EBADUID,		/* Bad uid for request */
    PW_EBADNAME,	/* Bad login name for request */
    PW_ETIMEOUT,	/* Server doesn't respond (timeout) */
    PW_EAUTHORITY,	/* No authority for operation */
    PW_EKADMLINK,	/* Can't initialise link to kadmin server */
    PW_EKADMINTKT,	/* Can't get initial ticket to kadmin service */
    PW_EKADMEXISTS,	/* Principal exists */
    PW_EKADMOP,  	/* Kadmin operation failed */
    PW_EBUILDAUTH,	/* Failed to build Kerberos authenticator */
    PW_EMEMORY,  	/* Memory allocation failed */
    PW_ENOAFS,  	/* AFS is not running on this host */
    PW_EAFSTOKEN,  	/* Can't create AFS token */
    PW_ENISDOMAIN,  	/* Can't get NIS domain name */
    PW_ENISMASTER,  	/* Can't locate NIS master */
    PW_EBADPNR	        /* Bad pnr name for request */
};

/* pw_errno will contain the error code after an error */
extern enum pw_errcode pw_errno;

/* pw_krberr will contain additional information for Kerberos errors */
extern int pw_krberr;

/* pw_syntaxerr will contain additional information for syntax errors */
extern int pw_syntaxerr;

/* Error strings */
extern char *pw_errlist[];

/* Number of error strings */
extern int pw_nerr;

/* This function returns true if errno is set with the given error */
extern int pw_is_syserr ();

/* This function returns true if pw_krberr is set with the given error */
extern int pw_is_krberr ();

/* This function returns true if pw_syntaxerr is set with the given error */
extern int pw_is_syntaxerr ();
void pw_abort();
void pw_warning();
void pw_error();

/* The types for pw_open */
#define PW_LOCAL	0	/* Open a file with passwd(5)-format */
#define PW_REMOTE	1	/* Open a connection to a remote server */
#define PW_NDBM		2	/* Open a ndbm-file pair (byuid and byname) */

/* The flags for pw_open */
#define PW_FROOT   	1	/* Permission to change root-entries */
#define PW_FSLOPPY   	2	/* Allow entries with sloppy syntax */

/* The maximum number of open descriptors (pwfiles/servers) */
#define PW_MAXOPEN	64

/* The fields of a passwd(5)-format line */
#define PW_NAME		0
#define PW_PASSWD	1
#define PW_UID		2
#define PW_GID		3
#define PW_GECOS	4
#define PW_DIR		5
#define PW_SHELL	6
#define PW_NFIELDS	7	/* The number of fields */

#define PW_BUFLEN	512

/* The operations on password files (server) */
extern int pw_open ();		/* Open a password file (server) */
extern void pw_close ();	/* Close a password file (server) */
extern void pw_create ();	/* Add a new line */
extern void pw_delete ();	/* Delete a line */
extern void pw_update ();	/* Replace a line with a new */
extern int pw_firstuid ();	/* Return first free uid no smaller than arg */
extern int pw_nextuid ();	/* Return first used uid no smaller than arg */
extern char *pw_prefetched ();	/* You don't want to know... */
extern char *pw_getpwuid ();	/* Return line with this uid */
extern char *pw_getpwnam ();	/* Return line with this logname */

extern void pnr_create (int d, char * pnr, char * name);
              /* Add a new pnr,name */
extern void pnr_delete (int d, char * name);
              /* Delete name */
extern void comment_change (int d, char *pnr, char *comment);
              /* Replace a comment with a new */
extern void comment_delete (int d, char * pnr);
              /* Delete a comment */
extern char *info_getbypnr (int d, char * pnr);
              /* Return info with this pnr */
extern char *pnr_getbyname (int d, char * name);
              /* Return pnr with this logname */
extern char *names_getbypnr (int d, char *pnr);
              /* Get a , separeted list of names */
extern char *comment_getbypnr (int d, char *pnr);
              /* Get the comment */
extern int pnr_checkpnr(char *pnr);
              /* Controll the pnr */



/* Password line syntax */

extern int pw_syntax ();
extern char *pw_syntax_message ();

#define PW_SYNTAX_NONE			0
#define PW_SYNTAX_SLOPPY		10
#define PW_SYNTAX_NORMAL		20
#define PW_SYNTAX_PEDANTIC		30

#define PW_SYNTAX_EOK			0
#define PW_SYNTAX_ENFIELDS		1
#define PW_SYNTAX_ENAMETOOLONG		2
#define PW_SYNTAX_EUID			3
#define PW_SYNTAX_EGID			4
#define PW_SYNTAX_ENONAME		5
#define PW_SYNTAX_EHASPASSWD		6
#define PW_SYNTAX_ENOPASSWD		7
#define PW_SYNTAX_ECOLON		8

#define GC_SYNTAX_ENFIELDS		101
#define GC_SYNTAX_ECREATIONDATE		102
#define GC_SYNTAX_EEXPIRATIONDATE	103
#define GC_SYNTAX_ECOMMA		104


/* Debug flag */
extern int pw_debug;

/* Some files */
#define PW_PWFILE	"/etc/passwd"
#define PW_SHELLFILE	"/etc/shells"
#define PW_NOLOGINFILE	"/usr/local/etc/shells.nologin"

/* Default port */
#define PW_DEFAULTPORT	200

/* Gecos stuff  */
#define GC_FULLNAME             0
#define GC_ORGANIZATION         1
#define GC_OFFICEPHONE          2
#define GC_HOMEPHONE            3
#define GC_COMMENT              4
#define GC_CREATIONDATE         5
#define GC_CREATOR              6
#define GC_EXPIRATIONDATE       7
#define GC_MAILFORWARD          8
#define GC_ADMINFO              9
#define GC_NFIELDS              10

extern int gc_user_editable ();
extern char *gc_prompt ();


/* Password checking and generation */
extern int pw_checkpasswd ();
extern char *pw_checkpasswd_message ();
extern char *pw_mkpasswd ();

#define PW_CHECK_OK			0
#define PW_CHECK_EMPTY			1
#define PW_CHECK_REVERSEDLOGNAME	2
#define PW_CHECK_LOGNAME		3
#define PW_CHECK_NONPRINT		4
#define PW_CHECK_SHORT			5
#define PW_CHECK_NOALPHA		6
#define PW_CHECK_JUSTALPHA		7
#define PW_CHECK_GROUPING		8

/* Kerberos administration routines */
extern void pw_kadm_init ();
extern void pw_kadm_svc_init ();
extern void pw_kadm_usr_init ();
extern void pw_kadm_destroy_ticket ();
extern void pw_kadm_add_new_key ();
extern void pw_kadm_change_password ();
extern void pw_kadm_zap_principal ();

/* Full name to logname mapping */
extern char *pw_full2log ();

/* */
struct pw_data
{
    char **pwv;
    char **gcv;
};
typedef struct pw_data pw_data;

extern char *pw_data_to_string ();
extern pw_data *pw_string_to_data ();
extern void pw_free_data ();

struct info_data
{
    char **uname;
    char **info;
};
typedef struct info_data info_data;
extern char *info_data_to_string();
extern info_data *info_string_to_data(char *s);
extern info_data *info_delete_comment(info_data *d);
extern info_data *info_change_comment(info_data *d, char * comment);
extern info_data *info_add_name(info_data *d, char *name);
extern info_data *info_del_name(info_data *d, char *name);
extern info_data *info_make_empty(char *pnr);
extern char *info_get_names(info_data *d);
extern char *info_get_comment(info_data *d);
extern void info_free_data(info_data *d);
extern int info_empty (info_data *d);



/* Flags for pw_admin_init */
#define PW_CHPASS	1
#define PW_KADMIN	2
#define PW_AFS		4

extern void pw_admin_init ();
extern void pw_admin_usr_init ();
extern void pw_admin_destroy ();

#define NONIS
#define SERVERNAME "heimdal.e.kth.se"
#define PASSWDPATH "/var/chpass/passwd"

