
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <pwd.h>
#include <setjmp.h>
#include <sys/param.h>
#include <unistd.h>
#include <string.h>

#include <afs/stds.h>
#include <afs/auth.h>
#include <afs/cellconfig.h>

#include <des.h>
#include <krb.h>
#include <krb_db.h>
#include <krb_err.h>
#include <kadm.h>
#include <z/bool.h>
#include <z/error.h>
#include <z/str.h>
#include "pw.h"

#define SIGSYS 12

static char pw_tkt_string[MAXPATHLEN];

static char *this_cell ();
static bool has_afs_syscalls ();
static int get_ad_in_tkt ();

void
pw_admin_init (flags, principal, instance, passwd, srvtab, expir)
    int flags;
    char *principal;
    char *instance;
    char *passwd;
    char *srvtab;
    unsigned int expir;
{
    int res;
    char realm[REALM_SZ];
    int life;
    FILE *fp;
 

    if ((flags & PW_AFS) && !has_afs_syscalls ())
    {
	pw_errno = PW_ENOAFS;
	return;
    }

    init_krb_err_tbl ();


    /*
     * Get local realm name
     */

    res = krb_get_lrealm (realm, 1);
    if (res != KSUCCESS)
        strcpy (realm, KRB_REALM);


    /*
     * Create ticket file
     */

    sprintf (pw_tkt_string, "/tmp/tkt_chpass_%d", getpid ());
    fp = fopen (pw_tkt_string, "w");
    if (fp == NULL)
    {
	pw_errno = PW_ETICKET;
	pw_krberr = TKT_FIL_ACC;
	unlink (pw_tkt_string);
	return;
    }
    if (fputs (principal, fp) == EOF || fputc ('\0', fp) == EOF
    ||  fputs (instance, fp) == EOF || fputc ('\0', fp) == EOF)
    {
	pw_errno = PW_ETICKET;
	pw_krberr = TKT_FIL_ACC;
	fclose (fp);
	unlink (pw_tkt_string);
	return;
    }
    if (fclose (fp) == EOF)
    {
	pw_errno = PW_ETICKET;
	pw_krberr = TKT_FIL_ACC;
	unlink (pw_tkt_string);
	return;
    }
    krb_set_tkt_string (pw_tkt_string);

   
    /*
     * Calculate lifetime value
     */

    life = krb_time_to_life (time (NULL), expir);


    /*
     * Get a ticket to the chpass service if requested
     */

    if (flags & PW_CHPASS)
    {
	res = get_ad_in_tkt (principal, instance, realm,
			     "chpass", "server", life,
			     passwd, srvtab);
	if (res != KSUCCESS)
	{
	    pw_errno = PW_ETICKET;
	    pw_krberr = res;
	    dest_tkt ();
	    return;
	}
    }


    /*
     * Get a ticket to the kadmin service if requested
     */

    if (flags & PW_KADMIN)
    {
	res = get_ad_in_tkt (principal, instance, realm,
			     PWSERV_NAME, KADM_SINST, life,
			     passwd, srvtab);
	if (res != KSUCCESS)
	{
	    pw_errno = PW_ETICKET;
	    pw_krberr = res;
	    dest_tkt ();
	    return;
	}

	res = kadm_init_link (PWSERV_NAME, KRB_MASTER, realm);
	if (res != KADM_SUCCESS)
	{
	    pw_errno = PW_EKADMLINK;
	    pw_krberr = res;
	    return;
	}
    }

    
    /*
     * Get a ticket to the AFS service if requested
     */

    if (flags & PW_AFS)
    {
	res = get_ad_in_tkt (principal, instance, realm,
				AUTH_SUPERUSER, "", life,
				passwd, srvtab);
	if (res != KSUCCESS)
	{
	    pw_errno = PW_ETICKET;
	    pw_krberr = res;
	    dest_tkt ();
	    return;
	}
    }


    /*
     * Create PAG and token if AFS service is requested
     */

    if (flags & PW_AFS)
    {
	char *cell;
	struct ktc_principal afs;
	struct ktc_principal me;
	struct ktc_token token;
	CREDENTIALS cred;
	

	res = krb_get_cred (AUTH_SUPERUSER, "", realm, &cred);

	setpag ();
	
	cell = this_cell ();

	strcpy (afs.name, AUTH_SUPERUSER);
	strcpy (afs.instance, "");
	strcpy (afs.cell, cell);

	strcpy (me.name, principal);
	strcpy (me.instance, instance);
	strcpy (me.cell, cell);

	free (cell);

	token.ticketLen = cred.ticket_st.length;
	bcopy (&cred.ticket_st.dat[0], &token.ticket[0], token.ticketLen);
	bcopy (&cred.session[0], &token.sessionKey, sizeof (token.sessionKey));
	token.kvno = cred.kvno;
	token.startTime = cred.issue_date;
	token.endTime = krb_life_to_time (cred.issue_date, cred.lifetime);

	res = ktc_SetToken (&afs, &token, &me);
	if (res != 0)
	{
	    pw_errno = PW_EAFSTOKEN;
	    pw_krberr = res;
	    dest_tkt ();
	    return;
	}
    
    }

	
    /*
     * All's well that ends well..
     */

    pw_errno = PW_EOK;
    abort_hook = pw_admin_destroy;
}

void
pw_admin_usr_init (flags, expir)
    int flags;
    int expir;
{
    int res;
    char principal[ANAME_SZ];
    char passwd[MAX_KPW_LEN];
    char *logname;
    struct passwd *pwd;
    extern char *getlogin ();
    extern char *getenv ();

    
    /*
     * Get principal name
     */

    logname = getlogin ();
    if (logname == NULL)
        logname = getenv ("USER");
    if (logname != NULL)
    {
        pwd = getpwnam (logname);
        if (pwd == NULL)
            my_abort ("Who are you? (User %s is not known)", logname);
    }
    else
    {
        pwd = getpwuid (getuid ());
        if (pwd == NULL)
            my_abort ("Who are you? (Uid %d is not known)", getuid ());
    }
    strcpy (principal, pwd->pw_name);
    

    /*
     * Get principals admin password
     */

    res = des_read_pw_string (passwd, MAX_KPW_LEN - 1,
                              "Your \"admin\" password: ", 0);
    if (res != 0)
        my_abort ("Failed to read the password");


    /*
     * Do it
     */
    
    pw_admin_init (flags, principal, "admin", passwd, NULL, expir);
    bzero (passwd, sizeof (passwd));

    if (pw_errno != PW_EOK)
	pw_abort ();
}

void
pw_admin_destroy ()
{
    dest_tkt ();
    if (has_afs_syscalls ())
	ktc_ForgetAllTokens ();
}

static int
get_ad_in_tkt (principal, instance, realm,
	       service, sinstance, life,
	       passwd, srvtab)
    char *principal;
    char *instance;
    char *realm;
    char *service;
    char *sinstance;
    int life;
    char *passwd;
    char *srvtab;
{
    int res;
    char real_tkt_string[MAXPATHLEN];
    char tmp_tkt_string[MAXPATHLEN];
    FILE *real;
    FILE *tmp;
    int c;

    strcpy (real_tkt_string, tkt_string ());
    sprintf (tmp_tkt_string, "/tmp/tkt_tmp.%d", getpid ());
    krb_set_tkt_string (tmp_tkt_string);

    if (passwd != NULL)
	res = krb_get_pw_in_tkt (principal, instance, realm,
				 service, sinstance, life, passwd);
    else
	res = krb_get_svc_in_tkt (principal, instance, realm,
				  service, sinstance, life, srvtab);
    if (res != KSUCCESS)
    {
	krb_set_tkt_string (real_tkt_string);
	return res;
    }

    krb_set_tkt_string (real_tkt_string);

    tmp = fopen (tmp_tkt_string, "r");
    if (tmp == NULL)
	return KDC_GEN_ERR;
    
    while (getc (tmp))
	;
    while (getc (tmp))
	;
    
    real = fopen (real_tkt_string, "a");
    if (real == NULL)
	return KDC_GEN_ERR;

    while ((c = getc (tmp)) != EOF)
	putc (c, real);

    fclose (tmp);
    fclose (real);
    unlink (tmp_tkt_string);

    return KSUCCESS;
}

static char *
this_cell ()
{
    FILE *fp;
    char cell[1024];
    
    fp = fopen ("/usr/vice/etc/ThisCell", "r");
    if (fp == NULL)
	return NULL;
    if (fgets (cell, 1024, fp) == NULL)
	return NULL;
    stripline (cell);
    fclose (fp);

    return strdup (cell);
}

static jmp_buf jmpbuf;

static void
sig_sys()
{
  errno = 0;
  longjmp (jmpbuf, 1);
}

/*
 * XXX: Only works for one kind of AFS syscalls!
 */
static bool
has_afs_syscalls ()
{
  void (*func) ();
  bool has_afs;

  has_afs = FALSE;
  
  func = signal (SIGSYS, sig_sys);
  if (setjmp (jmpbuf) == 0)
  {
      syscall (31, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
      errno = 0;
      has_afs = TRUE;
  }
  signal (SIGSYS, func);

  return has_afs;
}
