
/* iteration.c
 *
 * Some utility functions that iterates through a given pw-database.
 */

#include <stdlib.h>

#include <z/bool.h>
#include "pw.h"


/* pw_foreach
 *
 * The given procedure is called once foreach entry.
 */
void
pw_foreach (d, proc)
    int	d;
    void (*proc) ();
{
    int uid;
    char *line;

    uid = 0;
    while ((uid = pw_nextuid (d, uid)) != -1)
    {
	uid++;
	line = pw_prefetched (d);
	if (pw_errno != PW_EOK)
		return;
	(*proc) (line);
	free (line);
    }

    if (pw_errno == PW_ENOTFOUND)
	pw_errno = PW_EOK;
}

/* pw_map
 *
 * Change all entries as specified by the given function.
 */
void
pw_map (d, func)
    int	d;
    char *(*func) ();
{
    int uid;
    char *old;
    char *new;

    uid = 0;
    while ((uid = pw_nextuid (d, uid)) != -1)
    {
	uid++;
	old = pw_prefetched (d);
	if (pw_errno != PW_EOK)
		return;
	new = (*func) (old);
	if (strcmp (old, new) != 0)
	    pw_update (d, old, new);
	free (old);
	free (new);
	if (pw_errno != PW_EOK)
	    return;
    }

    if (pw_errno == PW_ENOTFOUND)
	pw_errno = PW_EOK;
}


/* pw_filter
 *
 * Delete all entries satisfying the given predicate.
 */
void
pw_filter (d, pred)
    int	d;
    bool (*pred) ();
{
    int uid;
    char *line;

    uid = 0;
    while ((uid = pw_nextuid (d, uid)) != -1)
    {
	uid++;
	line = pw_prefetched (d);
	if (pw_errno != PW_EOK)
		return;
	if ((*pred) (line))
	    pw_delete (d, line);
	free (line);
	if (pw_errno != PW_EOK)
	    return;
    }

    if (pw_errno == PW_ENOTFOUND)
	pw_errno = PW_EOK;
}
