
/* pw.c
 *
 */

#include <string.h>
#include <z/bool.h>
#include <z/error.h>
#include "pw.h"
#include "pw.H"
#include "local.H"
#include "remote.H"
#include "ndbm.H"

static bool descriptor_is_ok ();


/* The descriptor table */
slot _pw_dtab[PW_MAXOPEN];

/* Debug flag */
bool pw_debug;

/* descriptor_is_ok
 *
 * This function checks the validity of a generic descriptor.
 */
static bool
descriptor_is_ok (d)
    int d;
{
    if (d < 0 || d >= PW_MAXOPEN)
	return FALSE;
    
    if (!_pw_dtab[d].active)
	return FALSE;

    return TRUE;
}


/*
 * Generic function-tables
 */

static int (*generic_open[]) () =
{ _pw_local_open, _pw_remote_open, _pw_ndbm_open };
static void (*generic_close[]) () =
{ _pw_local_close, _pw_remote_close, _pw_ndbm_close };
static void (*generic_create[]) () =
{ _pw_local_create, _pw_remote_create, _pw_ndbm_create };
static void (*generic_delete[]) () =
{ _pw_local_delete, _pw_remote_delete, _pw_ndbm_delete };
static void (*generic_update[]) () =
{ _pw_local_update, _pw_remote_update, _pw_ndbm_update };
static int (*generic_firstuid[]) () =
{ _pw_local_firstuid, _pw_remote_firstuid, _pw_ndbm_firstuid };
static int (*generic_nextuid[]) () =
{ _pw_local_nextuid, _pw_remote_nextuid, _pw_ndbm_nextuid };
static char *(*generic_prefetched[]) () =
{ _pw_local_prefetched, _pw_remote_prefetched, _pw_ndbm_prefetched };
static char *(*generic_getpwuid[]) () =
{ _pw_local_getpwuid, _pw_remote_getpwuid, _pw_ndbm_getpwuid };
static char *(*generic_getpwnam[]) () =
{ _pw_local_getpwnam, _pw_remote_getpwnam, _pw_ndbm_getpwnam };

static void (*generic_pnr_create[]) () =
{ _pnr_local_create, _pnr_remote_create, _pnr_ndbm_create };
static void (*generic_pnr_delete[]) () = 
{ _pnr_local_delete, _pnr_remote_delete, _pnr_ndbm_delete };
static void (*generic_comment_change[]) () = 
{ _comment_local_change, _comment_remote_change, _comment_ndbm_change };
static void (*generic_comment_delete[]) () = 
{ _comment_local_delete, _comment_remote_delete, _comment_ndbm_delete };
static char*(*generic_info_getbypnr[]) () = 
{ _info_local_getbypnr, _info_remote_getbypnr, _info_ndbm_getbypnr };
static char*(*generic_pnr_getbyname[]) () = 
{ _pnr_local_getbyname, _pnr_remote_getbyname, _pnr_ndbm_getbyname };
static char*(*generic_names_getbypnr[]) () = 
{ _names_local_getbypnr, _names_remote_getbypnr, _names_ndbm_getbypnr };
static char*(*generic_comment_getbypnr[]) () = 
{ _comment_local_getbypnr, _comment_remote_getbypnr, _comment_ndbm_getbypnr };

/* pw_open
 *
 */
int
pw_open (name, type, flags)
    char *	name;
    int		type;
    int		flags;
{
    int		res;		/* Result (all-purpose) */
    int		d;		/* Descriptor */
    

    /* Check type parameter */

    ASSERT (type == PW_LOCAL || type == PW_REMOTE || type == PW_NDBM);
    
    
    /* Get hold of a free table slot */

    d = 0;
    while (_pw_dtab[d].active && d < PW_MAXOPEN)
	d++;

    if (d == PW_MAXOPEN)
    {
	pw_errno = PW_ETOOMANY;
	return -1;
    }

    
    /* Do the actual opening */
    
    pw_errno = PW_EOK;
    res = generic_open[type] (d, name);
    if (res == -1)
	return -1;


    /* Fill in table slot and return descriptor */

    _pw_dtab[d].active = TRUE;
    _pw_dtab[d].type = type;
    _pw_dtab[d].flags = flags;
    return d;
}


/* pw_close
 *
 */
void
pw_close (d)
    int d;
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_close[_pw_dtab[d].type] (d);
	_pw_dtab[d].active = FALSE;
    }
}

void
pw_create (d, new)
    int         d;              /* Descriptor */
    char *      new;            /* New entry to insert */
{
    if (!descriptor_is_ok (d))
        pw_errno = PW_EDESCRIPTOR;
    else
    {
        pw_errno = PW_EOK;
        generic_create[_pw_dtab[d].type] (d, new);
    }
}

void
pw_delete (d, old)
    int		d;		/* Descriptor */
    char *	old;		/* Old entry to delete */
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_delete[_pw_dtab[d].type] (d, old);
    }
}

void
pw_update (d, old, new)
    int		d;		/* Descriptor */
    char *	old;		/* What to change FROM */
    char *	new;		/* What to change TO */
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_update[_pw_dtab[d].type] (d, old, new);
    }
}

int
pw_firstuid (d, uid)
    int		d;		/* Descriptor */
    int		uid;		/* Target uid */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return -1;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_firstuid[_pw_dtab[d].type] (d, uid);
    }
}

int
pw_nextuid (d, uid)
    int		d;		/* Descriptor */
    int		uid;		/* Target uid */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return -1;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_nextuid[_pw_dtab[d].type] (d, uid);
    }
}

char *
pw_prefetched (d)
    int		d;		/* Descriptor */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_prefetched[_pw_dtab[d].type] (d);
    }
}

char *
pw_getpwuid (d, uid)
    int		d;		/* Descriptor */
    int		uid;		/* Uid to match */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_getpwuid[_pw_dtab[d].type] (d, uid);
    }
}

char *
pw_getpwnam (d,lognam)
    int		d;		/* Descriptor */
    char *	lognam;		/* Login name to match */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_getpwnam[_pw_dtab[d].type] (d, lognam);
    }
}

void
pnr_create (d, pnr, uname)
    int		d;		/* Descriptor */
    char *	pnr;		/* New entry to insert */
    char *      uname;

{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_pnr_create[_pw_dtab[d].type] (d, pnr, uname);
    }
}

void 
pnr_delete (d, name)
     int d;       /* Descriptor */
     char * name;/* User name */
{
  if (!descriptor_is_ok (d))
    pw_errno = PW_EDESCRIPTOR;
  else
    {
      pw_errno = PW_EOK;
      generic_pnr_delete[_pw_dtab[d].type] (d, name);
    }
}

void 
comment_change (d,pnr, comment)
     int d;
     char * pnr;
     char * comment;
{
  if (!descriptor_is_ok (d))
    pw_errno = PW_EDESCRIPTOR;
  else
    {
      pw_errno = PW_EOK;
      generic_comment_change[_pw_dtab[d].type] (d, pnr, comment);
    }
}
 
void 
comment_delete (d, pnr)
     int d;       /* Descriptor */
     char * pnr;/* User name */
{
  if (!descriptor_is_ok (d))
    pw_errno = PW_EDESCRIPTOR;
  else
    {
      pw_errno = PW_EOK;
      generic_comment_delete[_pw_dtab[d].type] (d, pnr);
    }
}


char *
info_getbypnr (int d, char * pnr)
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_info_getbypnr[_pw_dtab[d].type] (d, pnr);
    }
}

char *
pnr_getbyname (int d, char * name)
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_pnr_getbyname[_pw_dtab[d].type] (d, name);
    }
}

char *
names_getbypnr (int d, char * pnr)
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_names_getbypnr[_pw_dtab[d].type] (d, pnr);
    }
}

char *
comment_getbypnr (int d, char * pnr)
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_comment_getbypnr[_pw_dtab[d].type] (d, pnr);
    }
}






