#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <z/bool.h>
#include <z/error.h>
#include <sys/time.h>

#include "pw.h"

/* The "alphabets" below has ambigous characters removed */
static char *lc = "abcdefghijkmnopqrstuvwxyz";
static char *uc = "ABCDEFGHJKLMNPQRSTUVWXYZ"; 
static char *nu = "23456789";                 
static char *sp = "!?%/=+$;";

/*
   init_random
     Initializes the random seed. Called by pw_mkpasswd once but might
     be useful to call to re-seed the pseudo random generator.  
*/

static void
init_random(void) 
{
    struct timeval tv;
    int pid = getpid();

    gettimeofday(&tv, (struct timezone *) NULL);
    srandom(tv.tv_sec ^ tv.tv_usec ^ pid);
}

int
pw_checkpasswd (pword, name)
    char *pword;
    char *name;
{
    int transitions;
    int nonprint;
    int nonalpha;
    int alpha;
    int wasalpha;
    int l;
    int i;
    char rword[1024];
    
    transitions = 0;
    nonprint = 0;
    alpha = 0;
    nonalpha = 0;
    wasalpha = isalpha (pword[0]);
    l = strlen (pword);
    if (l == 0)
	return PW_CHECK_EMPTY;
    for (i = 0; i < l; i++)
    {
	rword[l - 1 - i] = pword[i];
	if (!isprint(pword[i]))
	    nonprint++;
	if (wasalpha ^ isalpha(pword[i]))
	    transitions++;
	wasalpha = isalpha(pword[i]);
	if (wasalpha)
	    alpha++;
	else
	    nonalpha++;
    }
    rword[l] = '\0';
    if (nonprint > 0)
	return PW_CHECK_NONPRINT;
    else if (l < 6)
	return PW_CHECK_SHORT;
/*
    else if (alpha == 0)
	return PW_CHECK_NOALPHA;
    else if (nonalpha == 0)
	return PW_CHECK_JUSTALPHA;
    else if (transitions < 2)
	return PW_CHECK_GROUPING;
    else if (strcmp(pword, name) == 0)
	return PW_CHECK_LOGNAME;
    else if (strcmp(rword, name) == 0)
	return PW_CHECK_REVERSEDLOGNAME;
*/
    else
	return PW_CHECK_OK;
}

char *
pw_checkpasswd_message (code)
    int code;
{
    switch (code)
    {
    case PW_CHECK_OK:
	return "Password is OK!\n";
	
    case PW_CHECK_EMPTY:
        return "Empty password.\n";

    case PW_CHECK_REVERSEDLOGNAME:
	return "Password reverse of login name.\n";

    case PW_CHECK_LOGNAME:
	return "Password same as login name.\n";

    case PW_CHECK_NONPRINT:
	return "Password contains non-printing (control) characters.\n";

    case PW_CHECK_SHORT:
	return "Password is too short (minimum 6 characters).\n";

    case PW_CHECK_NOALPHA:
	return "Password contains no alphabetic character.\n";

    case PW_CHECK_JUSTALPHA:
	return "Password contains no non-alphabetic character.\n";

    case PW_CHECK_GROUPING:
	return "\
Alpabetic and non-alphabetic must be mixed.\n\
It is not allowed to have all the alphabetic characters\n\
at the beginning and all the non-alphabetic characters at\n\
the end, nor vice versa.\n";

    default:
	ASSERT(FALSE);
	return NULL;
    }
}

/* returns an 8 character password */

static char * characters = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!?%/=+$;";

struct spell {
    char shortname;
    char * longname;
};


struct spell spelllist[] = {
    {'a',	"adam"},
    {'b',	"bertil"},
    {'c',	"caesar"},
    {'d',	"david"},
    {'e',	"erik"},
    {'f',	"filip"},
    {'g',	"gustav"},
    {'h',	"helge"},
    {'i',	"ivar"},
    {'j',	"johan"},
    {'k',	"kalle"},
    {'l',	"ludvig"},
    {'m',	"martin"},
    {'n',	"niklas"},
    {'o',	"olof"},
    {'p',	"petter"},
    {'q',	"quintus"},
    {'r',	"rudolf"},
    {'s',	"sigurd"},
    {'t',	"tore"},
    {'u',	"urban"},
    {'v',	"viktor"},
    {'w',	"willhelm"},
    {'x',	"xerxes"},
    {'y',	"yngve"},
    {'z',	"zta"},
    {'A',	"ADAM"},
    {'B',	"BERTIL"},
    {'C',	"CAESAR"},
    {'D',	"DAVID"},
    {'E',	"ERIK"},
    {'F',	"FILIP"},
    {'G',	"GUSTAV"},
    {'H',	"HELGE"},
    {'I',	"IVAR"},
    {'J',	"JOHAN"},
    {'K',	"KALLE"},
    {'L',	"LUDVIG"},
    {'M',	"MARTIN"},
    {'N',	"NIKLAS"},
    {'O',	"OLOF"},
    {'P',	"PETTER"},
    {'Q',	"QUINTUS"},
    {'R',	"RUDOLF"},
    {'S',	"SIGURD"},
    {'T',	"TORE"},
    {'U',	"URBAN"},
    {'V',	"VIKTOR"},
    {'W',	"WILLHELM"},
    {'X',	"XERXES"},
    {'Y',	"YNGVE"},
    {'Z',	"ZTA"},
    {'!',	"utropstecken"},
    {'#',	"brdgrd"},
    {'?',	"frgetecken"},
    {'%',	"procenttecken"},
    {'&',	"ampersand"},
    {'/',	"snedstreck"},
    {'(',	"vnsterparentes"},
    {')',	"hgerparentes"},
    {'=',	"likamed"},
    {'+',	"plus"},
    {'$',	"dollar"},
    {'^',	"circumflex"},
    {';',	"semikolon"},
    {'0',	"nolla"},
    {'1',	"etta"},
    {'2',	"tva"},
    {'3',	"trea"},
    {'4',	"fyra"},
    {'5',	"femma"},
    {'6',	"sexa"},
    {'7',	"sjua"},
    {'8',	"tta"},
    {'9',	"nia"}
};

char *
pw_convchartoword(char character) {
    int i;
    for(i =0 ; i < sizeof (spelllist); i++) {
	if(spelllist[i].shortname == character)
	    return strdup(spelllist[i].longname);
    } 
    return NULL;
}

/*
    fisher_yates
      Do a fisher_yates shuffle on a string. Returns a newly allocated
      string as the result and NULL if it cannot allocate memory for
      the new string.
*/

static char *
fisher_yates(char *string) 
{
    int len = strlen(string);
    char *shuffled;
    int i, j;
    char c;

    if ((shuffled = malloc(sizeof(char) * len + 1)) == NULL) 
    {
        return NULL;
    }

    for(i = 0; i < len; i++) 
    {
        shuffled[i] = string[i];
    }

    for (i=len - 1 ; i > 0; i--) 
    {
        j = random()%i;
        if (i == j) continue;

        c = shuffled[i];
        shuffled[i] = shuffled[j];
        shuffled[j] = c;
    }

    shuffled[len] = '\0';

    return shuffled;
}

/*  
    pw_mkpasswd
      Returns an eight character password that is guaranteed to
      contain both upper and lower case letters as well as numerical
      and special characters conforming to the recommendations given
      in the agreement for using the computer facilities at the School
      of Electrical Engineering, KTH.
*/

char *
pw_mkpasswd(char *logname)
{
    static int random_initialized = 0;
    char passwd[9];
    int lc_len = strlen(lc);
    int uc_len = strlen(uc);
    int nu_len = strlen(nu);
    int sp_len = strlen(sp);
    int no_lc, no_uc, no_nu, no_sp, i, idx;

    if (!random_initialized) 
    {
        init_random();
        for(i=0; i < 17; i++) random();
        random_initialized = 1;
    }
    
    no_sp = 1 + random()%2;             /* 1-2 special chars */
    no_nu = 1 + random()%2;             /* 1-2 numeric chars */
    no_uc = 1 + random()%3;             /* 1-3 uppercase chars */
    no_lc = 8 - no_uc - no_nu - no_sp;  /* 1-5 lowercase chars */

    for(i=0, idx=0; i < no_sp ; i++, idx++) 
    {
        passwd[idx] = sp[random()%sp_len];
    }

    for(i=0; i < no_nu ; i++, idx++) 
    {
        passwd[idx] = nu[random()%nu_len];
    }

    for(i=0; i < no_uc ; i++, idx++) 
    {
        passwd[idx] = uc[random()%uc_len];
    }

    for(i=0; i < no_lc ; i++, idx++) 
    {
        passwd[idx] = lc[random()%lc_len];
    }

    passwd[8] = '\0';
    
    return fisher_yates(passwd);
}
