
/* userask.c
 *
 */

#include <stdio.h>

#include <z/bool.h>
#include "userask.h"

extern char *index ();

/*
 *
 * Here all checking for different types is made.
 *
 * All types is in a static array of structs. One struct for each
 * type, so if you want to add a type, just add one entry in the array
 * named types[], and maybe a function for some special checking.
 *
 * The entry in types[] must contain:
 *    - One name of the type, which must be the same as the second field
 *      of an entry in passwd.defs
 *    - One string of all legal characters which will be passed 
 *      to your function as an argument.
 *    - One pointer to a function wich returns TRUE or FALSE depending
 *      on if the string passed to the function is legal or not.
 *      The function takes two arguments, the first is the string
 *      to check, and the second is the string with legal characters.
 *
 * The program itself does NOT check the legal characters, you must
 * do that in your own function. Your function could look like this:
 *
 *    int foo(buffer, chars)
 *    char *buffer, *chars;
 *    {
 *         if(!ua_charcheck(buffer, chars))
 *                return(FALSE);
 * Your special checking
 *         return(TRUE);
 *    }
 *
 * If you don't want to, or don't have to write your own function
 * put ua_charcheck as the checking function in your entry in types[].
 * 
 */

/*
 * This part is for the different typedefs in passwd.defs
 */

typedef struct {
    char *name;
    char *chars;
    int  (*func)();
} TYPES;

static int ua_charcheck();
static int ua_datecheck();
static int ua_nada_adminfo();

TYPES types[] = {

/* gecosstring */
    "gecosstring",
    " !\"#$%&'()*+-./0123456789;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_`abcdefghijklmnopqrstuvwxyz{|}~",  /* Not <:>, <\> and <,> */
    ua_charcheck,
/* nobackslashstring */
    "nobackslashstring",
    " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_`abcdefghijklmnopqrstuvwxyz{|}~", /* Not <\> */
    ua_charcheck,
/* allcharstring */
    "allcharstring",
    " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~", /* All! */
    ua_charcheck,
/* date */
    "date",
    "-0123456789",
    ua_datecheck,
/* number */
    "number",
    "0123456789",
    ua_charcheck,
/* phone */
    "phonenumber",
    " 0123456789+-",
    ua_charcheck,
/* nada_adminfo */
    "nada_adminfo",
    "",
    ua_nada_adminfo,
/* sentinel */
    "",
    "",
};


/* ua_charcheck
 *
 * This small routine returns TRUE only iff each character in 'buffer'
 * is a member of the string 'chars'.
 */
static int
ua_charcheck(buffer, chars)
char *buffer, *chars;
{
    char *i;

    i=buffer;
    while(*i != '\0' && index(chars,*i)!=NULL)
	i++;
    return(*i=='\0');
}

/*
 * Special checker functions
 */

#ifdef YYMMDD
int date_index[] = { 0,1,2 };
#else
#ifdef YYDDMM
int date_index[] = { 0,2,1 };
#else
#ifdef DDYYMM
int date_index[] = { 2,0,1 };
#else
#ifdef DDMMYY
int date_index[] = { 2,1,0 };
#else
#ifdef MMYYDD
int date_index[] = { 1,0,2 };
#else
#ifdef MMDDYY
int date_index[] = { 1,2,0 };
#else
You_must_define_a_date_format
#endif
#endif
#endif
#endif
#endif
#endif

int maxdays[] = { 31,28,31,30,31,30,31,31,30,31,30,31 };

#define MAXDATELEN 10

/* ua_datecheck
 *
 * This datecheck function accepts dates written as 1989-08-16, 89-08-16 or
 * 890816 (in this case YYMMDD was #define'd).
 */
static int
ua_datecheck(buffer, chars)
char *buffer, *chars;
{
    int  day, month, year;
    char *i, *j;
    int datefield[3];
    char tmpbuf[MAXDATELEN + 1];

    if (strlen (buffer) > MAXDATELEN)
	return FALSE;
    if(!ua_charcheck(buffer, chars))
	return(FALSE);
    strcpy(tmpbuf, buffer);
    if((i=index(tmpbuf,'-')) != NULL) {
	if(strlen(buffer)!=8 && strlen(buffer)!=10)
	    return(FALSE);
	j=buffer;
	*i++ = '\0';
	datefield[0]=atoi(tmpbuf);
	j=i;
	if((i=index(j,'-')) == NULL)
	    return(FALSE); /* There must be two '-' in each date string */
	*i++ = '\0';
	datefield[1]=atoi(j);
	datefield[2]=atoi(i);
    } else {
	if(strlen(tmpbuf) != 6)
	    return(FALSE);	/* Now it must be 6 digits */
	datefield[2]=atoi(&tmpbuf[4]);
	tmpbuf[4]='\0';
	datefield[1]=atoi(&tmpbuf[2]);
	tmpbuf[2]='\0';
	datefield[0]=atoi(tmpbuf);
    }
    year  = datefield[date_index[0]];
    month = datefield[date_index[1]];
    day   = datefield[date_index[2]];

    if((year % 4) == 0 &&	        /* Leap year, February    */
       ((year % 100) != 0 || (year % 400) == 0))
	maxdays[1] = 29;

    if(month < 1 || month > 12) 
	return(FALSE);

    if(day < 1 || day > maxdays[month-1]) 
	return(FALSE);

    return(TRUE);
}


/* ua_nada_adminfo
 *
 * checks if data is a valid entry for the adminfo gecos-fields used
 * at NADA.
 */
static int
ua_nada_adminfo (buf, chars)
    char *buf;
    char *chars;	/* Not used */
{
    if (strcmp (buf, "warn") == 0)
	return TRUE;
    if (strcmp (buf, "arch") == 0)
	return TRUE;
    if (strcmp (buf, "expir") == 0)
	return TRUE;

    return FALSE;
}


/*
 * Now for some useful functions ...
 */

/* ua_knowntype
 *
 * Returns TRUE if the given string is a type identifier which is known -
 * ie has an entry in the table types[] .
 */
int 
ua_knowntype (type)
    char *type;
{
    int i;
    
    i = 0;
    while (strcmp (types[i].name, "") != 0)
    {
	if (strcmp (types[i].name, type) == 0)
	    return TRUE;
	i++;
    }
    return FALSE;
}


#define STREXISTS(S)	((S) != (char *) 0)
#define STRNOTEMPTY(S)	((S) != (char *) 0 && *(S) != '\0')

#define BUFLEN 256

extern char *index();

int ua_errorcode;

static int
ua_error(ua_ecod)
int ua_ecod;
{
    ua_errorcode = ua_ecod;
    return(FALSE);
}


/*
 * This routine checks that 'string' is of the named type.
 *
 */
int
ua_typecheck (string, name)
char *string, *name;
{
    TYPES *righttype = NULL;
    int i=0;
    
    while(strcmp(types[i].name,"") != 0) {
	if (strcmp(types[i].name,name)==0) {
	    righttype = &(types[i]);
	    break;
	}
	i++;
    }
    if (righttype == NULL) {
	fprintf(stderr,"The type %s is illegal\n",name);
	exit(1);
    }
    return(righttype->func(string, righttype->chars));
}


int
ua_typerestrict(str, uadata)
char* str;
struct useraskdata uadata;
{
    int slen;

    if (STRNOTEMPTY(uadata.help2) && strcmp(str, uadata.help2) == 0)
	return(ua_error(UAERR_WRONGTYPE));

    slen=strlen(str);
    if (slen == 0 && uadata.emptyallowed)
	return(TRUE);
    else if (slen < uadata.minlen)
	return(ua_error(UAERR_TOOSHORT));
    else if (slen > uadata.maxlen)
	return(ua_error(UAERR_TOOLONG));
    else if (!ua_typecheck(str, uadata.type))
	return(ua_error(UAERR_WRONGTYPE));
    else
	return(TRUE);
}

int
ua_setrestrict(str, uadata)
char* str;
struct useraskdata uadata;
{
    char *cp, *cpend;

    if (STRNOTEMPTY(uadata.help2) && strcmp(str, uadata.help2) == 0)
	return(ua_error(UAERR_NOTINSET));

    cp=uadata.otherdata;
    cpend=index(cp, '\n');
    while (*cp != '\0') {
	if (cpend == (char *) NULL) {
	    if (strcmp(str, cp) == 0) return(TRUE);
	    else return(ua_error(UAERR_NOTINSET));
	}	
	else
	    if (strncmp(str, cp, cpend-cp) == 0 && strlen(str) == cpend-cp)
		return(TRUE);
	cp=cpend+1;
	cpend=index(cp, '\n');
    }
    return(ua_error(UAERR_NOTINSET));
}

void
ua_helpfunc(ua_ecod, uadata)
int ua_ecod;
struct useraskdata uadata;
{
    switch (ua_ecod) {
    case UAERR_TOOLONG:
	fprintf(stderr,
		"Your answer must not be longer than %d characters\n",
		uadata.maxlen);
	break;
    case UAERR_TOOSHORT:
	fprintf(stderr,
		"Your answer must be at least %d characters\n",
		uadata.minlen);
	break;
    case UAERR_WRONGTYPE:
	fprintf(stderr,
		"Your answer must be of type '%s'\n", uadata.type);
	break;
    case UAERR_NOTINSET:
	fprintf(stderr,
		"Your answer must be one of these:\n%s\n",
		uadata.otherdata);
	break;
    case UAERR_HELP1:
	if (STREXISTS(uadata.helptext) && strlen(uadata.helptext) > 0)
	    fprintf(stderr, "%s\n", uadata.helptext);
	else
	    fprintf(stderr, "No help available\n");
	break;
    default:
	fprintf(stderr, "Can't help...\n");
	break;
    }
}

int
userask(prompt, default_value,
	 uadata, restrictfunc, helpfunc, the_value)
char *prompt, *default_value, *the_value;
struct useraskdata uadata;
int (*restrictfunc)();
void (*helpfunc)();
{
    int ok=FALSE, defaultanswer, inputlen;
    char buffer[BUFLEN];

    while(ok != TRUE) {
	defaultanswer=FALSE;
	if STREXISTS(default_value)
	    printf("%s <%s>:", prompt, default_value);
	else
	    printf("%s :", prompt);
	fflush(stdout);
	fgets(buffer, BUFLEN, stdin);
	if(fflush(stdin)==EOF)
	    return(FALSE);
	if((inputlen=strlen(buffer)) > BUFLEN-2) {
	    (*helpfunc)(UAERR_TOOLONG, uadata);
	    continue;
	}
	else 
	    ok=TRUE;
	buffer[inputlen-1]='\0'; /* Remove trailing newline */
	inputlen--;
	if(inputlen == 0 && STREXISTS(default_value)) {
	    strcpy(buffer, default_value);
	    defaultanswer=TRUE;
	}

	if(strcmp(buffer,"none")==0) {
	    strcpy(buffer, "");
	    defaultanswer=(!STRNOTEMPTY(default_value));
	} 

	if (STRNOTEMPTY(uadata.help1) && strcmp(buffer, uadata.help1) == 0) {
	    ok = FALSE;
	    (*helpfunc)(UAERR_HELP1, uadata);
	}
	else {
	    ok=(*restrictfunc)(buffer, uadata);
	    if(ok != TRUE)
		(*helpfunc)(ua_errorcode, uadata);
	}
    }
    strcpy(the_value, buffer);
    return(!defaultanswer);
}

void
ua_showtypes(filed)
FILE *filed;
{
    int i=0;
    
    while(strcmp(types[i].name,"") != 0) {
	fprintf(filed,"%s\n",types[i].name);
	i++;
    }
}
