
#include <stdio.h>
#include <pwd.h>
#include "../../readline/readline.h"
#include <z/bool.h>
#include <z/error.h>
#include <pw.h>

char *progname = "chsh";

#define MAXSHELLS	64
static char *shells[MAXSHELLS];
static int nshells;

extern char *strdup ();
extern char *strcpy ();

char *chsh ();
void readshells ();
char *select_shell ();
char *shell_completer ();

main (argc, argv)
    int argc;
    char *argv[];
{
    int d;
    char *old;
    char *new;
    char *logname;
    struct passwd *pwd;
    extern char *getlogin ();
    extern char *getenv ();
    
    d = pw_args (&argc, &argv);

    if (argc == 1)
    {
	logname = getlogin ();
	if (logname == NULL)
	    logname = getenv ("USER");
	if (logname != NULL)
	{
	    pwd = getpwnam (logname);
	    if (pwd == NULL)
		my_abort ("Who are you? (User %s is not known)", logname);
	}
	else
	{
	    pwd = getpwuid (getuid ());
	    if (pwd == NULL)
		my_abort ("Who are you? (Uid %d is not known)", getuid ());
	}

	old = pw_getpwnam (d, pwd->pw_name);
    }
    else
	old = pw_getpwarg (d, &argc, &argv);
    if (old == NULL)
        pw_abort ();
    

    new = chsh (old);

    if (strcmp (old, new) == 0)
	my_abort ("No change");

    pw_update (d, old, new);
    if (pw_errno != PW_EOK)
	pw_abort ();

    exit (0);
}

char *
chsh (old)
    char *old;
{
    pw_data *pwd;
    char *shell;
    
    pwd = pw_string_to_data (old);
    if (pwd == NULL)
	my_abort ("Memory allocation fails");
    if (pw_syntax (pwd, PW_SYNTAX_SLOPPY) != PW_SYNTAX_EOK)
	my_abort ("Your passwd-entry is corrupt");

    printf ("Changing login shell for \"%s\"\n", pwd->pwv[PW_NAME]);

    readshells ();

    shell = select_shell (pwd->pwv[PW_SHELL]);
    
    pwd->pwv[PW_SHELL] = shell;
    return pw_data_to_string (pwd);
}

void
readshells ()
{
    FILE *fp;
    char buf[PW_BUFLEN];

    nshells = 0;

    fp = fopen (PW_SHELLFILE, "r");
    if (fp == NULL)
	my_sysabort ("%s", PW_SHELLFILE);

    while (fgets (buf, PW_BUFLEN, fp) != NULL)
    {
        stripline (buf);
	shells[nshells++] = strdup (buf);
    }

    fclose (fp);

    if (getuid () == 0)
    {
	fp = fopen (PW_NOLOGINFILE, "r");
	if (fp == NULL)
	    my_sysabort ("%s", PW_NOLOGINFILE);
	
	while (fgets (buf, PW_BUFLEN, fp) != NULL)
	{
	    stripline (buf);
	    shells[nshells++] = strdup (buf);
	}
	
	fclose (fp);
    }
}

char *
select_shell (oldshell)
    char *oldshell;
{
    char prompt[1024];
    char *newshell;
    char *cp;
    int i;

    printf ("Current login shell is %s\n", oldshell);
    sprintf (prompt, "New login shell [%s]: ", oldshell);
    rl_completion_entry_function = (Function *) shell_completer ;

    for (;;)
    {

	newshell = readline (prompt);

	if (newshell == NULL)
	    exit (1);

	while (*newshell == ' ')
	    newshell++;
	cp = newshell + strlen (newshell);
	while (*--cp == ' ')
	    *cp = '\0';

	if (strlen (newshell) == 0)
	    newshell = oldshell;

	if (strcmp (oldshell, newshell) == 0)
	    exit (0);

	i = 0;
	while (i < nshells && strcmp (shells[i], newshell) != 0)
	    i++;

	if (i != nshells)
	    return shells[i];

	printf ("\nThese are your alternatives:\n\n");
	for (i = 0; i < nshells; i ++)
	    printf ("  %s\n", shells[i]);
	putchar ('\n');
    }
}

char *
shell_completer (text, state)
    char *text;
    int state;
{
    static int i;

    if (!state)
        i = 0;

    while (shells[i] != NULL)
    {
        if (strncasecmp (shells[i], text, strlen (text)) == 0)
            return strdup (shells[i++]);
        i++;
    }

    return NULL;
}
