
/* pw.c
 *
 */

#include <string.h>
#include <z/bool.h>
#include <z/error.h>
#include "pw.h"
#include "info.h"
#include "info.H"
//#include "local.H"
#include "inforemote.H"
#include "infondbm.H"

static bool descriptor_is_ok ();


/* The descriptor table */
slot _info_dtab[PW_MAXOPEN];

/* Debug flag */
bool pw_debug;

/* descriptor_is_ok
 *
 * This function checks the validity of a generic descriptor.
 */
static bool
descriptor_is_ok (d)
    int d;
{
    if (d < 0 || d >= PW_MAXOPEN)
	return FALSE;
    
    if (!_info_dtab[d].active)
	return FALSE;

    return TRUE;
}


/*
 * Generic function-tables
 */

static int (*generic_info_open[]) () =
{ info_local_open, info_remote_open, info_ndbm_open };

static void (*generic_info_close[]) () =
{ info_local_close, info_remote_close, info_ndbm_close };

static void (*generic_info_create[]) () =
{ info_local_create, info_remote_create, info_ndbm_create };

static void (*generic_pnr_delete[]) () =
{ pnr_local_delete, pnr_remote_delete, pnr_ndbm_delete };

static void (*generic_pnr_create[]) () =
{ pnr_local_create, pnr_remote_create, pnr_ndbm_create };

static void (*generic_info_delete[]) () =
{ info_local_delete, info_remote_delete, info_ndbm_delete };

static void (*generic_info_update[]) () =
{ info_local_update, info_remote_update, info_ndbm_update };

static int (*generic_info_firstuid[]) () =
{ info_local_firstuid, info_remote_firstuid, info_ndbm_firstuid };

static int (*generic_info_nextuid[]) () =
{ info_local_nextuid, info_remote_nextuid, info_ndbm_nextuid };

static char *(*generic_info_prefetched[]) () =
{ info_local_prefetched, info_remote_prefetched, info_ndbm_prefetched };

static char *(*generic_info_getpwuid[]) () =
{ info_local_getpwuid, info_remote_getpwuid, info_ndbm_getpwuid };

static char *(*generic_info_getpwnam[]) () =
{ info_local_getpwnam, info_remote_getpwnam, info_ndbm_getpwnam };



/* info_open
 *
 */
int
info_open (name, type, flags)
    char *	name;
    int		type;
    int		flags;
{
    int		res;		/* Result (all-purpose) */
    int		d;		/* Descriptor */
    

    /* Check type parameter */

    ASSERT (type == PW_LOCAL || type == PW_REMOTE || type == PW_NDBM);
    
    
    /* Get hold of a free table slot */

    d = 0;
    while (_info_dtab[d].active && d < PW_MAXOPEN)
	d++;

    if (d == PW_MAXOPEN)
    {
	pw_errno = PW_ETOOMANY;
	return -1;
    }

    
    /* Do the actual opening */
    
    pw_errno = PW_EOK;
    res = generic_info_open[type] (d, name);
    if (res == -1)
	return -1;


    /* Fill in table slot and return descriptor */

    _info_dtab[d].active = TRUE;
    _info_dtab[d].type = type;
    _info_dtab[d].flags = flags;
    return d;
}


/* info_close
 *
 */
void
info_close (d)
    int d;
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_info_close[_info_dtab[d].type] (d);
	_info_dtab[d].active = FALSE;
    }
}


void
pnr_create (d, pnr, uname)
    int		d;		/* Descriptor */
    char *	pnr;		/* New entry to insert */
    char *      uname; 
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_pnr_create[_info_dtab[d].type] (d, pnr, uname);
    }
}

void
pnr_delete (d, old)
    int		d;		/* Descriptor */
    char *	old;		/* Old entry to delete */
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_pnr_delete[_info_dtab[d].type] (d, old);
    }
}



void
info_create (d, new)
    int		d;		/* Descriptor */
    char *	new;		/* New entry to insert */
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_info_create[_info_dtab[d].type] (d, new);
    }
}

void
pw_delete (d, old)
    int		d;		/* Descriptor */
    char *	old;		/* Old entry to delete */
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_delete[_pw_dtab[d].type] (d, old);
    }
}

void
pw_update (d, old, new)
    int		d;		/* Descriptor */
    char *	old;		/* What to change FROM */
    char *	new;		/* What to change TO */
{
    if (!descriptor_is_ok (d))
	pw_errno = PW_EDESCRIPTOR;
    else
    {
	pw_errno = PW_EOK;
	generic_update[_pw_dtab[d].type] (d, old, new);
    }
}

int
pw_firstuid (d, uid)
    int		d;		/* Descriptor */
    int		uid;		/* Target uid */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return -1;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_firstuid[_pw_dtab[d].type] (d, uid);
    }
}

int
pw_nextuid (d, uid)
    int		d;		/* Descriptor */
    int		uid;		/* Target uid */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return -1;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_nextuid[_pw_dtab[d].type] (d, uid);
    }
}

char *
pw_prefetched (d)
    int		d;		/* Descriptor */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_prefetched[_pw_dtab[d].type] (d);
    }
}

char *
pw_getpwuid (d, uid)
    int		d;		/* Descriptor */
    int		uid;		/* Uid to match */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_getpwuid[_pw_dtab[d].type] (d, uid);
    }
}

char *
pw_getpwnam (d,lognam)
    int		d;		/* Descriptor */
    char *	lognam;		/* Login name to match */
{
    if (!descriptor_is_ok (d))
    {
	pw_errno = PW_EDESCRIPTOR;
	return NULL;
    }
    else
    {
	pw_errno = PW_EOK;
	return generic_getpwnam[_pw_dtab[d].type] (d, lognam);
    }
}
