
/* local.c
 *
 */

#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>

#ifdef HAVE_SYS_SIGNAL_H
#include <sys/signal.h>
#endif

#include <sys/stat.h>
#include <z/bool.h>
#include <z/basename.h>
#include "pw.h"
#include "pw.H"
#include "local.H"
#define max(a,b)	(a > b ? a : b)


/* _pw_local_open
 *
 */
int
_pw_local_open (d, filename)
    int		d;			/* Descriptor */
    char *	filename;		/* Name of passwd(5) file to open */
{
    FILE *	fp;   			/* passwd file pointer */
    int		res;			/* General purpose (result code) */
    struct stat	statinfo;		/* For stat(2) */

    if (filename == NULL)
	filename = PW_PWFILE;

    res = stat (filename, &statinfo);
    if (res == -1)
    {
	pw_errno = PW_EOPEN;
	return -1;
    }
    
    fp = fopen (filename, "r");
    if (fp == NULL)
    {
	pw_errno = PW_EOPEN;
	return -1;
    }

    _pw_dtab[d].module.local.passwd = strdup (filename);
    _pw_dtab[d].module.local.ptmp = samedir (filename, "ptmp");
    _pw_dtab[d].module.local.fp = fp;
    _pw_dtab[d].module.local.prefetch_uid = -1;
    return 0;
}

/* _pw_local_close
 *
 */
/*ARGSUSED*/
void
_pw_local_close (d)
    int d;
{
    fclose (_pw_dtab[d].module.local.fp);
    free (_pw_dtab[d].module.local.passwd);
    free (_pw_dtab[d].module.local.ptmp);
}


static enum pw_errcode _pw_local_do_update ();


/* _pw_local_create
 */
void
_pw_local_create (d, new)
    int			d;
    char           	*new;
{
    pw_errno = _pw_local_do_update (d, (char *) NULL, new);
}


/* _pw_local_delete
 */
void
_pw_local_delete (d, old)
    int		d;
    char	*old;
{
    pw_errno = _pw_local_do_update (d, old, (char *) NULL);
}

/* _pw_local_update
 */
void
_pw_local_update (d, old, new)
    int d;
    char *old;
    char *new;
{
    pw_errno = _pw_local_do_update (d, old, new);
}


/* _pw_local_do_update
 *
 * This function does all the work for the functions
 * '_pw_local_create', '_pw_local_delete' and '_pw_local_update' above.
 */
static enum pw_errcode
_pw_local_do_update (d, old, new)
    int		d;		/* Descriptor */
    char *	old;		/* Old passwd(5)-line */
    char *	new;		/* New passwd(5)-line */
{
    static char this[PW_BUFLEN];/* Buffer space for current entry */	
    int new_uid;		/* Uid of entry to insert (update) */
    int this_uid;		/* Uid of current entry */
    static char new_name[9];	/* Login name of entry to insert (update) */
    static char this_name[9];	/* Login name of current entry */
    int prev_uid;		/* Uid of previous entry */
    pw_data *pwd;
    int mask;			/* For blocking signals */
    int oldmask;		/* Saves old signal blocking status */
    FILE *ifp;			/* Input file (passwd) */
    FILE *ofp;			/* Output file (ptmp, becomes new passwd) */
    bool found_old;		/* TRUE when the old entry has been found */
    bool inserted_new;		/* TRUE when the new entry has been inserted */
    int res;			/* General purpose result code */
    char *passwd;		/* The passwd file (from descriptor) */
    char *ptmp;			/* The lock file (from descriptor) */
    int level;

    /*
     * Check new line and extract info from it
     */

    if (new != NULL)
    {
	if (strlen (new) > PW_BUFLEN - 2)
	    return PW_ELINETOOLONG;
	pwd = pw_string_to_data (new);
	if (pwd == NULL)
	    return PW_EMEMORY;
	level = _pw_dtab[d].flags & PW_FSLOPPY ?
	    PW_SYNTAX_SLOPPY : PW_SYNTAX_NORMAL;
	if ((res = pw_syntax (pwd, level)) != PW_SYNTAX_EOK)
	{
	    pw_free_data (pwd);
	    pw_syntaxerr = res;
	    return PW_ENEWSYNTAX;
	}
	strcpy (new_name, pwd->pwv[PW_NAME]);
	new_uid = atoi (pwd->pwv[PW_UID]);
	pw_free_data (pwd);
	if (!(_pw_dtab[d].flags & PW_FROOT)
	&& (new_uid == 0 || strcmp (new_name, "root") == 0))
	    return PW_EROOT;
    }
    

    /*
     * Block out terminal signals while doing this
     */

#ifdef _POSIX_VERSION
    sigaddset(&mask, SIGINT); 
    sigaddset(&mask, SIGQUIT); 
    sigaddset(&mask, SIGTSTP);
    sigprocmask(SIG_BLOCK, &mask, &oldmask);
#else
    mask = sigmask (SIGINT) | sigmask (SIGQUIT) | sigmask (SIGTSTP);
    oldmask = sigblock (mask);
#endif

    /*
     * Create lock file - this file will be used to build the new
     * passwd-file
     */

    ptmp = _pw_dtab[d].module.local.ptmp;
    (void) umask (0);
    res = open (ptmp, O_WRONLY | O_CREAT | O_EXCL, 0644);
    if (res == -1)
    {
#ifdef _POSIX_VERSION
	sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *)0);
#else
	sigsetmask (oldmask);
#endif	
	if (errno == EEXIST)
	    return PW_ELOCKED;
	else
	    return PW_ECREAT;
    }
    ofp = fdopen (res, "w");
    if (ofp == NULL)
    {
	unlink (ptmp);
#ifdef _POSIX_VERSION
        sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *)0);
#else
	sigsetmask (oldmask);
#endif
	return PW_EFSFAIL;
    }


    /*
     * The passwd-file is opened in '_pw_local_open'
     */

    ifp = _pw_dtab[d].module.local.fp;
    if (ifp == NULL)
    {
	ifp = fopen (_pw_dtab[d].module.local.passwd, "r");
	if (ifp == NULL)
	{
	    pw_errno = PW_EOPEN;
	    return -1;
	}
    }
    rewind (ifp);
    passwd = _pw_dtab[d].module.local.passwd;


    /*
     * A practical clean-up macro
     */

#ifdef _POSIX_VERSION
#define Return(x) \
return fclose(ofp), unlink(ptmp), sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *)0), x
#else
#define Return(x) \
return fclose(ofp), unlink(ptmp), sigsetmask(oldmask), x
#endif


    /*
     * The main loop of this function merely copies lines from the
     * passwd-file to the lock-file, which will eventually become
     * to new passwd-file. If 'new' is not NULL that line will be
     * inserted at the proper place. If 'old' is not NULL a line
     * EXACTLY matching it will be removed. It is assumed that the
     * passwd-file is sorted in uid-order.
     */

    prev_uid = -1;		/* Need a value less than the smallest uid */
    found_old = FALSE;
    inserted_new = FALSE;
    while (fgets (this, PW_BUFLEN, ifp) != NULL)
    {


	/* '+'-lines (NIS/YP inclusion/exclusion) are just copied */

	if (this[0] == '+')
	{
	    if (fputs (this, ofp) == EOF)
		Return (PW_EWRITE);
	    continue;
	}


	/* Check syntax of entry */

	stripline (this);
	pwd = pw_string_to_data (this);
	if (pwd == NULL)
	{
	    pw_free_data (pwd);
	    Return (PW_EMEMORY);
	}
	if (pw_syntax (pwd, PW_SYNTAX_SLOPPY) != PW_SYNTAX_EOK)
	{
	    pw_free_data (pwd);
	    Return (PW_EBADPASSWD);
	}

	
	/* Extract fields */

	strcpy (this_name, pwd->pwv[PW_NAME]);
	this_uid = atoi (pwd->pwv[PW_UID]);
	pw_free_data (pwd);


	/* If this is the 'old' line - don't copy it */

	if (old != NULL && strcmp (this, old) == 0)
	{
	    if (!(_pw_dtab[d].flags & PW_FROOT)
	    && (this_uid == 0 || strcmp (this_name, "root") == 0))
		Return (PW_EROOT);
	    found_old = TRUE;
	    continue;
	}


	/*
	 * Check if this uid/name is what we are trying to insert - duplicate
	 * uids are only allowed below uid=100 and no duplicate names are
	 * allowed.
	 */

	if (new != NULL)
	{
	    if (this_uid == new_uid && this_uid > 99)
		Return (PW_EUIDEXISTS);
	    if (strcmp (this_name, new_name) == 0)
		Return (PW_ENAMEEXISTS);
	}


	/* Check if passwd-file is sorted */

	if (this_uid < prev_uid)
	    Return (PW_ENOTSORTED);
	

	/* If at right point - insert new line */
	
	if (new != NULL && !inserted_new)
	{
	    if ((found_old && new_uid == this_uid)
	    || (new_uid >= prev_uid && new_uid < this_uid))
	    {
		if (fputs (new, ofp) == EOF || fputc ('\n', ofp) == EOF)
		    Return (PW_EWRITE);
		inserted_new = TRUE;
	    }
	}


	/* Copy passwd-line to ptmp */
	
	if (fputs (this, ofp) == EOF || fputc ('\n', ofp) == EOF)
	    Return (PW_EWRITE);

	prev_uid = this_uid;
    }
    
    if (ferror (ifp))
	Return (PW_EREAD);

    if (old != NULL && !found_old)
	Return (PW_ENOTFOUND);

    if (new != NULL && prev_uid <= new_uid)
    {
	if (fputs (new, ofp) == EOF || fputc ('\n', ofp) == EOF)
	    Return (PW_EWRITE);
    }	

    if (fclose (ofp) == EOF)
    {
	unlink (ptmp);
#ifdef _POSIX_VERSION
        sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *)0);
#else
        sigsetmask (oldmask);
#endif
	return PW_EWRITE;
    }

    res = rename (ptmp, passwd);
    if (res == -1)
    {
	unlink (ptmp);
#ifdef _POSIX_VERSION
        sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *)0);
#else
        sigsetmask (oldmask);
#endif
	return PW_EFSFAIL;
    }
    
#ifdef _POSIX_VERSION
        sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *)0);
#else
        sigsetmask (oldmask);
#endif

    (void) fclose (ifp);
    ifp = fopen (passwd, "r");
    _pw_dtab[d].module.local.fp = ifp;
    
    return PW_EOK;
}

/* _pw_local_firstuid
 */
int
_pw_local_firstuid (d, target)
    int d;
    int target;
{
    static char buf[PW_BUFLEN];	
    int this;
    pw_data *pwd;
    FILE *fp;				
    int prev;
    

    fp = _pw_dtab[d].module.local.fp;
    if (fp == NULL)
    {
	fp = fopen (_pw_dtab[d].module.local.passwd, "r");
	if (fp == NULL)
	{
	    pw_errno = PW_EOPEN;
	    return -1;
	}
    }
    rewind (fp);

    prev = -1;
    while (fgets (buf, PW_BUFLEN, fp) != NULL)
    {
	if (buf[0] =='+')
	    continue;
	stripline (buf);
	pwd = pw_string_to_data (this);
	if (pwd == NULL)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EMEMORY;
	    return -1;
	}
	if (pw_syntax (pwd, PW_SYNTAX_SLOPPY) != PW_SYNTAX_EOK)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EBADPASSWD;
	    return -1;
	}
	this = atoi (pwd->pwv[PW_UID]);
	pw_free_data (pwd);

	if (this < prev)
	{
	    pw_errno = PW_ENOTSORTED;
	    return -1;
	}

	if (this > target && this > prev + 1)
	    return max (prev + 1, target);

	prev = this;
    }
    
    if (ferror (fp))
    {
	pw_errno = PW_EREAD;
	return -1;
    }

    return max (prev + 1, target);
}

/* _pw_local_nextuid
 */
int
_pw_local_nextuid (d, target)
    int d;
    int target;
{
    static char buf[PW_BUFLEN];	
    int this;
    pw_data *pwd;
    FILE *fp;				
    int prev;
    

    fp = _pw_dtab[d].module.local.fp;
    if (fp == NULL)
    {
	fp = fopen (_pw_dtab[d].module.local.passwd, "r");
	if (fp == NULL)
	{
	    pw_errno = PW_EOPEN;
	    return -1;
	}
    }
    rewind (fp);

    prev = -1;
    while (fgets (buf, PW_BUFLEN, fp) != NULL)
    {
	if (buf[0] =='+')
	    continue;
	stripline (buf);
	pwd = pw_string_to_data (buf);
	if (pwd == NULL)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EMEMORY;
	    return -1;
	}
	if (pw_syntax (pwd, PW_SYNTAX_SLOPPY) != PW_SYNTAX_EOK)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EBADPASSWD;
	    return -1;
	}
	this = atoi (pwd->pwv[PW_UID]);
	pw_free_data (pwd);

	if (this < prev)
	{
	    pw_errno = PW_ENOTSORTED;
	    return -1;
	}

	if (this >= target)
	{
	    _pw_dtab[d].module.local.prefetch_uid = this;
	    strcpy (_pw_dtab[d].module.local.prefetch_line, buf);
	    return this;
	}

	prev = this;
    }
    
    if (ferror (fp))
    {
	pw_errno = PW_EREAD;
	return -1;
    }

    pw_errno = PW_ENOTFOUND;
    return -1;
}

char *
_pw_local_prefetched (d)
    int d;
{
    if (_pw_dtab[d].module.local.prefetch_uid == -1)
	return NULL;
    else
	return strdup (_pw_dtab[d].module.local.prefetch_line);
}

/* _pw_local_getpwuid
 */
char *
_pw_local_getpwuid (d, uid)
    int d;
    int uid;
{
    static char buf[PW_BUFLEN];	
    pw_data *pwd;
    FILE *fp;				
    int this;
    int prev;
//    char *line;
    

    fp = _pw_dtab[d].module.local.fp;
    if (fp == NULL)
    {
	fp = fopen (_pw_dtab[d].module.local.passwd, "r");
	if (fp == NULL)
	{
	    pw_errno = PW_EOPEN;
	    return NULL;
	}
    }
    rewind (fp);

    prev = - 1;
    while (fgets (buf, PW_BUFLEN, fp) != NULL)
    {
	if (buf[0] == '+')
	    continue;
	stripline (buf);
	pwd = pw_string_to_data (this);
	if (pwd == NULL)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EMEMORY;
	    return NULL;
	}
	if (pw_syntax (pwd, PW_SYNTAX_SLOPPY) != PW_SYNTAX_EOK)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EBADPASSWD;
	    return NULL;
	}
	this = atoi (pwd->pwv[PW_UID]);
	pw_free_data (pwd);

	if (this < prev)
	{
	    pw_errno = PW_ENOTSORTED;
	    return NULL;
	}

	if (this == uid)
	    return buf;

	if (this > uid)
	{
	    pw_errno = PW_ENOTFOUND;
	    return NULL;
	}
    }
    
    if (ferror (fp))
    {
	pw_errno = PW_EREAD;
	return NULL;
    }

    pw_errno = PW_ENOTFOUND;
    return NULL;
}

/* _pw_local_getpwnam
 */
char *
_pw_local_getpwnam (d, name)
    int		d;
    char *	name;
{
    static char buf[PW_BUFLEN];	
    pw_data *	pwd;
    FILE *	fp;				
    char *	this;

    fp = _pw_dtab[d].module.local.fp;
    if (fp == NULL)
    {
	fp = fopen (_pw_dtab[d].module.local.passwd, "r");
	if (fp == NULL)
	{
	    pw_errno = PW_EOPEN;
	    return NULL;
	}
    }
    rewind (fp);

    while (fgets (buf, PW_BUFLEN, fp) != NULL)
    {
	if (buf[0] == '+')
	    continue;
	stripline (buf);
	pwd = pw_string_to_data (this);
	if (pwd == NULL)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EMEMORY;
	    return NULL;
	}
	if (pw_syntax (pwd, PW_SYNTAX_SLOPPY) != PW_SYNTAX_EOK)
	{
	    pw_free_data (pwd);
	    pw_errno = PW_EBADPASSWD;
	    return NULL;
	}
	this = pwd->pwv[PW_NAME];
	pw_free_data (pwd);

	if (strcmp (name, this) == 0)
	{
	    pw_errno = PW_EOK;
	    return buf;
	}
    }
    
    if (ferror (fp))
    {
	pw_errno = PW_EREAD;
	return NULL;
    }

    pw_errno = PW_ENOTFOUND;
    return NULL;
}

/* _pnr_local_create
 */
void
_pnr_local_create (d, pnr, uname)
    int			d;
    char *	pnr;		/* New entry to insert */
    char *      uname;
{
    pw_errno = PW_EIMPL;
}

/* _pw_local_delete
 */
void
_pnr_local_delete (d, uname)
    int			d;
    char *      uname;
{
    pw_errno = PW_EIMPL;
}

/* _comment_local_create
 */
void
_comment_local_change (d, pnr, comment)
    int			d;
    char *	pnr;		/* New entry to insert */
    char *      comment;
{
    pw_errno = PW_EIMPL;
}

/* _pw_local_delete
 */
void
_comment_local_delete (d, pnr)
    int			d;
    char *      pnr;
{
    pw_errno = PW_EIMPL;
}

char
*_info_local_getbypnr(int d, char *pnr)
{
  pw_errno = PW_EIMPL;
  return NULL;
}

char *
_pnr_local_getbyname(int d, char *name)
{
  pw_errno = PW_EIMPL;
  return NULL;
}

char 
*_names_local_getbypnr(int d, char *pnr)
{
  pw_errno = PW_EIMPL;
  return NULL;
}

char 
*_comment_local_getbypnr(int d, char *pnr)
{
  pw_errno = PW_EIMPL;
  return NULL;
}
