/* remote.c
 *
 */

#include <sys/time.h>
#include <sys/param.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>

#include <des.h>
#include <krb.h>

#include <z/bool.h>
#include <z/str.h>
#include <z/error.h>
#include "pw.h"
#include "proto.h"
#include "pw.H"
#include "remote.H"

static void call_server ();
static void timeout_handler ();

extern int errno;
extern char *index ();


/* Global data */
static struct remote_slot *p;
static char sendbuf[4096];
static char recvbuf[4096];
static struct request req;
static struct reply rep;
static struct itimerval ival;
volatile static int interval;
volatile static int total;
static MSG_DAT msg;
static char hostname[MAXHOSTNAMELEN];
struct sigaction act;

/* _pw_remote_open
 *
 */
int
_pw_remote_open (d, servername)
    int			d;		/* Descriptor */
    char *		servername;	/* Name of server to connect to */
{
    int			res;		/* General return status */
    struct hostent *	host;		/* For gethostbyname(3) */
    struct servent *	serv;		/* For getservbyname(3) */
    int			addrlen;	/* For getsockname(2) */
    char *		tmp;


    /* First set up SIGALRM handler (don't belong here really) */

    act.sa_handler = timeout_handler;
    sigfillset(&act.sa_mask);
#ifdef SV_INTERRUPT
    act.sa_flags = SV_INTERRUPT;
#elif SA_INTERRUPT
    act.sa_flags = SA_INTERRUPT;
#endif    
    (void) sigaction (SIGALRM, &act, NULL);
    

    /* get a handle on our descriptor slot */

    p = &(_pw_dtab[d].module.remote);


    /* default server is NIS master */

    if (servername == NULL)
#ifdef NONIS
        servername = krb_get_phost(SERVERNAME);
#else
    {
	char *domain;

	if (yp_get_default_domain (&domain) != 0)
	{
	    pw_errno = PW_ENISDOMAIN;
	    return -1;
	}
	if (yp_master (domain, "passwd.byuid", &servername) != 0)
	{
	    pw_errno = PW_ENISMASTER;
	    return -1;
	}
    }
#endif


    /* setup server address */

    host = gethostbyname (servername);
    if (host == NULL)
    {
	pw_errno = PW_EHOST;
	return -1;
    }

    bzero (&(p->serv_addr), sizeof (p->serv_addr));
    p->serv_addr.sin_family = AF_INET;
    bcopy (host->h_addr, &(p->serv_addr.sin_addr), host->h_length);
    serv = getservbyname ("chpass", "udp");
    p->serv_addr.sin_port = serv != NULL ? serv->s_port : PW_DEFAULTPORT;

    /* create a socket */

    p->socket = socket (AF_INET, SOCK_DGRAM, 0);
    if (p->socket == -1)
    {
	pw_errno = PW_ESOCKET;
	return -1;
    }

    /* setup local address */

    bzero (&(p->clnt_addr), sizeof (p->clnt_addr));
    p->clnt_addr.sin_family = AF_INET;
    res = gethostname (hostname, sizeof (hostname));
    ASSERT (res != -1);
    host = gethostbyname (hostname);
    ASSERT (host != NULL);
    bcopy (host->h_addr, &(p->clnt_addr.sin_addr), host->h_length);


    /* bind the socket to a local port */

    res = bind (p->socket, (struct sockaddr *)&(p->clnt_addr), sizeof (p->clnt_addr));
    if (res == -1)
    {
	pw_errno = PW_EBIND;
	res = errno; close (p->socket); errno = res;
	return -1;
    }
    bzero (&(p->clnt_addr), sizeof (p->clnt_addr));
    addrlen = sizeof(p->clnt_addr);
    res = getsockname (p->socket, (struct sockaddr *) &(p->clnt_addr), &addrlen);
    if (res == -1)
    {
	pw_errno = PW_EBIND;
	res = errno; close (p->socket); errno = res;
	return -1;
    }

    /* connect to the server */

    res = connect (p->socket, (struct sockaddr *) &(p->serv_addr), sizeof (p->serv_addr));
    if (res == -1)
    {
	pw_errno = PW_ECONNECT;
	res = errno; close (p->socket); errno = res;
	return -1;
    }


    /* get kerberos realm of server */

    p->serv_realm = krb_realmofhost (servername);


    /* fill in server name and shortname */

    p->servername = strdup (servername);
    tmp = index (servername, '.');
    if (tmp == NULL)
	p->shortname = strdup (servername);
    else
	p->shortname = dupsub (servername, tmp - 1);

    p->cred = NULL;

    p->nextid = 1;
    p->prefetch_uid = -1;

    return 0;
}


static int
get_ticket (p)
    struct remote_slot *p;
{
    int res;
    CREDENTIALS cred;


    /* if we have a valid credentials just return */

    if (p->cred != NULL
    && (time (0) < p->cred->issue_date + 5 * 60 * p->cred->lifetime))
	return 0;


    /* if our ticket has expired we are out of luck  */

    if (p->cred != NULL)
    {

	/** Should get new ticket here **/

	pw_errno = PW_ECREDENTIALS;
	pw_krberr = RD_AP_EXP;
	return -1;
    }


    /* check for a ticket in ticket file */

    res = krb_get_cred ("chpass", p->shortname, p->serv_realm, &cred);

    
    /* if it has expired we are out of luck again */

    if (res == KSUCCESS
    && (time (0) > cred.issue_date + 5 * 60 * cred.lifetime))
    {

	/** Should get new ticket here **/

	pw_errno = PW_ECREDENTIALS;
	pw_krberr = RD_AP_EXP;
	return -1;
    }

    
    /* if there was no ticket in the ticket file - get one */

    if (res == RET_NOTKT)
    {
	res = get_ad_tkt ("chpass", p->shortname, p->serv_realm, 96);
	if (res != AD_OK)
	{
	    pw_errno = PW_ETICKET;
	    pw_krberr = res;
	    return -1;
	}
	
	res = krb_get_cred ("chpass", p->shortname, p->serv_realm, &cred);
    }

    if (res != KSUCCESS)
    {
	pw_errno = PW_ECREDENTIALS;
	pw_krberr = res;
	return -1;
    }


    /* record credentials in our slot */

    if (p->cred != NULL)
	free (p->cred);
    p->cred = (CREDENTIALS *) malloc (sizeof (CREDENTIALS));
    bcopy (&cred, p->cred, sizeof (CREDENTIALS));
    des_key_sched (&p->cred->session, p->sched);

    return 0;
}

/* _pw_remote_close
 *
 * Close a connection to a 'chpassd' server and release the descriptor.
 * Returns 0 on success and -1 on failure (in which case pw_errno
 * is set to indicate the error).
 */
void
_pw_remote_close (d)
    int d;
{
     p = &(_pw_dtab[d].module.remote);

    close (p->socket);
    free (p->servername);
    free (p->shortname);
    free (p->cred);
}

/* _pw_remote_create
 */
void
_pw_remote_create (d, new)
    int			d;
    char *		new;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_CREATE;
    bcopy (new, req.arg.create.new, strlen (new) + 1);
    call_server ();
}


/* _pw_remote_delete
 */
void
_pw_remote_delete (d, old)
    int			d;
    char *		old;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_DELETE;
    bcopy (old, req.arg.delete.old, strlen (old) + 1);
    call_server ();
}


/* _pw_remote_update
 */
void
_pw_remote_update (d, old, new)
    int			d;
    char *		old;
    char *		new;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_UPDATE;
    bcopy (old, req.arg.update.old, strlen (old) + 1);
    bcopy (new, req.arg.update.new, strlen (new) + 1);
    call_server ();
}


/* _pw_remote_firstuid
 */
int
_pw_remote_firstuid (d, target)
    int			d;
    int 		target;
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_FIRSTUID;
    req.arg.firstuid.uid = target;
    call_server ();
    if (pw_errno != PW_EOK)
	return -1;

    return rep.res.uid;
}


/* _pw_remote_nextuid
 */
int
_pw_remote_nextuid (d, target)
    int			d;
    int 		target;
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_NEXTUID;
    req.arg.nextuid.uid = target;
    call_server ();
    if (pw_errno != PW_EOK)
	return -1;

    strcpy (p->prefetch_line, rep.res.line);
    p->prefetch_uid = rep.res.uid;

    return rep.res.uid;
}


char *
_pw_remote_prefetched (d)
    int d;
{
    if (_pw_dtab[d].module.remote.prefetch_uid == -1)
	return NULL;
    else
	return strdup (_pw_dtab[d].module.remote.prefetch_line);
}


/* _pw_remote_getpwuid
 */
char *
_pw_remote_getpwuid (d, uid)
    int			 d;
    int 		uid;
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_GETPWUID;
    req.arg.getpwuid.uid = uid;
    call_server ();
    if (pw_errno != PW_EOK)
	return NULL;

    return strdup (rep.res.line);
}


/* _pw_remote_getpwnam
 */
char *
_pw_remote_getpwnam (d, name)
    int			d;
    char *		name;
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_GETPWNAM;
    bcopy (name, req.arg.getpwnam.name, strlen (name) + 1);
    call_server ();
    if (pw_errno != PW_EOK)
	return NULL;

    return strdup (rep.res.line);
}

static void
call_server ()
{
    int res;
    KTEXT_ST auth;
    int sendlen;
    int len;
    int timeout;
    int tmp;
    char msgbuf[4096];
    int msglen;

    req.id = p->nextid++;

    msglen=sizeof(msgbuf);
    compose_request(&req, msgbuf, &msglen);
    if(msglen == 0) {
	my_sysabort("Couldn't compose request\n");
    }

    switch (req.type)
    {
    case REQ_CREATE:
    case REQ_DELETE:
    case REQ_UPDATE:
    case REQ_PNRCREATE:
    case REQ_PNRDELETE:
    case REQ_COMMENTCHANGE:
    case REQ_COMMENTDELETE:
    case REQ_INFOGETBYPNR:
    case REQ_PNRGETBYNAME:
    case REQ_NAMESGETBYPNR:
    case REQ_COMMENTGETBYPNR:


	/* Build authenticator */

	res = krb_mk_req (&auth, "chpass", p->shortname, p->serv_realm, 0);
	if (res != KSUCCESS)
	{
	    pw_errno = PW_EBUILDAUTH;
	    pw_krberr = res;
	    return;
	}

	/* First two bytes of message is length of authenicator */
	
	sendbuf[0] = (auth.length & 0xFF00) >> 8;
	sendbuf[1] = auth.length & 0xFF;
	sendlen = 2;
    
	/* Second part of message is the authenticator */
	
	bcopy (auth.dat, sendbuf + sendlen, auth.length);
	sendlen += auth.length;

	/* Third and last part of message is the encrypted request */

#ifdef PRIVATE
	len = krb_mk_priv (msgbuf, sendbuf + sendlen, msglen,
			   p->sched, p->cred->session,
			   &(p->clnt_addr), &(p->serv_addr));
#else
	len = krb_mk_safe (msgbuf, sendbuf + sendlen, msglen,
			   &p->cred->session,
			   &(p->clnt_addr), &(p->serv_addr));
#endif
	sendlen += len;

	interval = 10;

	break;


    default:

	sendbuf[0] = 0;
	sendbuf[1] = 0;
	bcopy (msgbuf, sendbuf + 2, msglen);
	sendlen = msglen + 2;

	interval = 1;
    }

    ival.it_interval.tv_sec = 0;
    ival.it_interval.tv_usec = 0;

    total = 0;

 retry:
    res = send (p->socket, sendbuf, sendlen, 0);
    if (res == -1)
    {
	pw_errno = PW_ESEND;
	return;
    }

    /* Turn on timer for timeout */

    ival.it_value.tv_sec = interval;
    ival.it_value.tv_usec = 0;
    (void) setitimer (ITIMER_REAL, &ival, NULL);

 skip:
    res = recv (p->socket, recvbuf, 4096, 0);
    if (res == -1)
    {
	if (errno == EINTR)
	{
	    if (total > 32)
	    {
		pw_errno = PW_ETIMEOUT;
		ival.it_value.tv_sec = 0;
		ival.it_value.tv_usec = 0;
		(void) setitimer (ITIMER_REAL, &ival, NULL);
		return;
	    }
	    else
		goto retry;
	}
	else
	{
	    pw_errno = PW_ERECV;
	    ival.it_value.tv_sec = 0;
	    ival.it_value.tv_usec = 0;
	    (void) setitimer (ITIMER_REAL, &ival, NULL);
	    return;
	}
    }
    len = res;

    if (total > 10 && isatty (fileno (stderr)))
	fprintf (stderr, "chpass server ok\n");


    /* Decode reply - if encrypted */

    if (recvbuf[0] != 0)
    {
#ifdef PRIVATE
	res = krb_rd_priv (recvbuf + 1, len - 1, p->sched, p->cred->session,
			   &(p->serv_addr), &(p->clnt_addr), &msg);
#else
	res = krb_rd_safe (recvbuf + 1, len - 1, &p->cred->session,
			   &(p->serv_addr), &(p->clnt_addr), &msg);
#endif
	if (res != KSUCCESS)
	{
	    pw_errno = PW_EREPLY;
	    pw_krberr = res;
	    ival.it_value.tv_sec = 0;
	    ival.it_value.tv_usec = 0;
	    (void) setitimer (ITIMER_REAL, &ival, NULL);
	    return;
	}

	tmp = msg.app_length;
	decompose_reply(&rep, msg.app_data, &tmp);
	if(tmp == 0) {
	    pw_errno = PW_EREPLY;
	    ival.it_value.tv_sec = 0;
	    ival.it_value.tv_usec = 0;
	    (void) setitimer (ITIMER_REAL, &ival, NULL);
	    return;
	}
    }
    else {
	tmp=len;
	decompose_reply(&rep, recvbuf+1, &tmp);
	if(tmp == 0) {
	    pw_errno = PW_EREPLY;
	    ival.it_value.tv_sec = 0;
	    ival.it_value.tv_usec = 0;
	    (void) setitimer (ITIMER_REAL, &ival, NULL);
	    return;
	}
    }

    if (rep.id != 0 && rep.id != req.id)
	goto skip;

    ival.it_value.tv_sec = 0;
    ival.it_value.tv_usec = 0;
    (void) setitimer (ITIMER_REAL, &ival, NULL);

    pw_errno = rep.pw_errno;
    pw_krberr = rep.xerrno.krberr;
    pw_syntaxerr = rep.xerrno.syntaxerr;
    errno = rep.error;

}

static void
timeout_handler ()
{
    total += interval;
    interval = 10;
    if (total > 10 && isatty (fileno (stderr)))
	fprintf (stderr, "chpass server not responding - still trying\n");
}

/* _pnr_remote_create
 */
void
_pnr_remote_create (d,pnr, name)
    int		d;		/* Descriptor */
    char *	pnr;		/* New entry to insert */
    char *      name;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_PNRCREATE;
    bcopy (pnr, req.arg.pnrcreate.pnr, strlen (pnr) + 1);
    bcopy (name, req.arg.pnrcreate.name, strlen (name) + 1);
    call_server ();
}

/* _pnr_remote_delete
 */
void
_pnr_remote_delete (d,name)
    int		d;		/* Descriptor */
    char *      name;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_PNRDELETE;
    bcopy (name, req.arg.pnrdelete.name, strlen (name) + 1);
    call_server ();
}

/* _comment_remote_change
 */
void
_comment_remote_change (d,pnr,comment)
    int		d;		/* Descriptor */
    char *	pnr;		/* New entry to insert */
    char *      comment;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_COMMENTCHANGE;
    bcopy (pnr, req.arg.commentchange.pnr, strlen (pnr) + 1);
    bcopy (comment, req.arg.commentchange.comment, strlen (comment) + 1);
    call_server ();
}


/* _comment_remote_delete
 */
void
_comment_remote_delete (d,pnr)
    int		d;		/* Descriptor */
    char *      pnr;
{
    p = &(_pw_dtab[d].module.remote);
    if (get_ticket (p) == -1)
	return;
    req.type = REQ_COMMENTDELETE;
    bcopy (pnr, req.arg.commentdelete.pnr, strlen (pnr) + 1);
    call_server ();
}

/* _info_remote_getbypnr
 */
char *_info_remote_getbypnr(int d, char *pnr)
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_INFOGETBYPNR;
    if (get_ticket (p) == -1)
	return NULL;
    bcopy (pnr, req.arg.infogetbypnr.pnr, strlen (pnr) + 1);
    call_server ();
    if (pw_errno != PW_EOK)
	return NULL;

    return strdup (rep.res.line);
}


/* _pnr_remote_getbyname
 */
char *_pnr_remote_getbyname(int d, char *name)
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_PNRGETBYNAME;
    if (get_ticket (p) == -1)
	return NULL;
    bcopy (name, req.arg.pnrgetbyname.name, strlen (name) + 1);
    call_server ();
    if (pw_errno != PW_EOK)
	return NULL;

    return strdup (rep.res.line);
}


/* _names_remote_getbypnr
 */
char *_names_remote_getbypnr(int d, char *pnr)
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_NAMESGETBYPNR;
    if (get_ticket (p) == -1)
	return NULL;
    bcopy (pnr, req.arg.namesgetbypnr.pnr, strlen (pnr) + 1);
    call_server ();
    if (pw_errno != PW_EOK)
	return NULL;

    return strdup (rep.res.line);
}

/* _comment_remote_getbypnr
 */
char *_comment_remote_getbypnr(int d, char *pnr)
{
    p = &(_pw_dtab[d].module.remote);
    req.type = REQ_COMMENTGETBYPNR;
    if (get_ticket (p) == -1)
	return NULL;
    bcopy (pnr, req.arg.commentgetbypnr.pnr, strlen (pnr) + 1);
    call_server ();
    if (pw_errno != PW_EOK)
	return NULL;

    return strdup (rep.res.line);
}

/*
 * struct reply -> char array
 */

void
compose_reply(struct reply *rep, char *buf, int *len)
{
    int i=0;
    u_int32_t tmp;

    tmp = htonl(rep->reqno);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    tmp = htonl(rep->id);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    tmp = htonl(rep->pw_errno);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

/* These two are really the same */

    tmp = htonl(rep->xerrno.krberr);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    tmp = htonl(rep->xerrno.syntaxerr);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    tmp = htonl(rep->error);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    bcopy(&rep->res.line, buf+i, PW_BUFLEN);
    i+=PW_BUFLEN;

    tmp = htonl(rep->res.uid);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    *len=i;
}

void
decompose_reply(struct reply *rep, char *buf, int *len)
{
    int i=0;
    u_int32_t tmp;

    if(7*sizeof(tmp)+PW_BUFLEN > *len) {
	*len = 0;
	return;
    }

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->reqno = ntohl(tmp);
    i+=sizeof(tmp);

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->id = ntohl(tmp);
    i+=sizeof(tmp);

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->pw_errno = ntohl(tmp);
    i+=sizeof(tmp);

/* These two are really the same */

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->xerrno.krberr = ntohl(tmp);
    i+=sizeof(tmp);

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->xerrno.syntaxerr = ntohl(tmp);
    i+=sizeof(tmp);

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->error = ntohl(tmp);
    i+=sizeof(tmp);

    bcopy(buf+i, &rep->res.line, PW_BUFLEN);
    rep->res.line[PW_BUFLEN] = '\0';
    i+=PW_BUFLEN;

    bcopy(buf+i, &tmp, sizeof(tmp));
    rep->res.uid = ntohl(tmp);
    i+=sizeof(tmp);

    *len=i;
}

void
compose_request(struct request *req, char *buf, int *len)
{
    int i=0;
    u_int32_t tmp;

    tmp = htonl(req->reqno);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    tmp = htonl(req->type);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    tmp = htonl(req->id);
    bcopy(&tmp, buf+i, sizeof(tmp));
    i+=sizeof(tmp);

    switch(req->type) 
    {

    case REQ_CREATE:
	bcopy(&req->arg.create.new, buf+i, PW_BUFLEN);
	i+=PW_BUFLEN;
	break;

    case REQ_DELETE:
	bcopy(&req->arg.delete.old, buf+i, PW_BUFLEN);
	i+=PW_BUFLEN;
	break;

    case REQ_UPDATE:
	bcopy(&req->arg.update.old, buf+i, PW_BUFLEN);
	i+=PW_BUFLEN;

	bcopy(&req->arg.update.new, buf+i, PW_BUFLEN);
	i+=PW_BUFLEN;
	break;

    case REQ_GETPWNAM:
	bcopy(&req->arg.getpwnam.name, buf+i, PW_BUFLEN);
	i+=PW_BUFLEN;
	break;

    case REQ_GETPWUID:
	tmp = htonl(req->arg.getpwuid.uid);
	bcopy(&tmp, buf+i, sizeof(tmp));
	i+=sizeof(tmp);
	break;

    case REQ_FIRSTUID:
	tmp = htonl(req->arg.firstuid.uid);
	bcopy(&tmp, buf+i, sizeof(tmp));
	i+=sizeof(tmp);
	break;

    case REQ_NEXTUID:
	tmp = htonl(req->arg.nextuid.uid);
	bcopy(&tmp, buf+i, sizeof(tmp));
	i+=sizeof(tmp);
	break;

    case REQ_PNRCREATE:
	bcopy(&req->arg.pnrcreate.pnr, buf+i, PNRLEN);
	i+=PNRLEN;

	bcopy(&req->arg.pnrcreate.name, buf+i, NAMELEN);
	i+=NAMELEN;
	break;

    case REQ_PNRDELETE:
	bcopy(&req->arg.pnrdelete.name, buf+i, NAMELEN);
	i+=NAMELEN;
	break;

    case REQ_COMMENTCHANGE:
	bcopy(&req->arg.commentchange.pnr, buf+i, PNRLEN);
	i+=PNRLEN;

	bcopy(&req->arg.commentchange.comment, buf+i, COMMENTLEN);
	i+=COMMENTLEN;
	break;

    case REQ_COMMENTDELETE:
	bcopy(&req->arg.commentdelete.pnr, buf+i, PNRLEN);
	i+=PNRLEN;
	break;

    case REQ_INFOGETBYPNR:
	bcopy(&req->arg.infogetbypnr.pnr, buf+i, PNRLEN);
	i+=PNRLEN;
	break;

    case REQ_PNRGETBYNAME:
	bcopy(&req->arg.pnrgetbyname.name, buf+i, NAMELEN);
	i+=NAMELEN;
	break;

    case REQ_NAMESGETBYPNR:
	bcopy(&req->arg.namesgetbypnr.pnr, buf+i, PNRLEN);
	i+=PNRLEN;
	break;

    case REQ_COMMENTGETBYPNR:
	bcopy(&req->arg.commentgetbypnr.pnr, buf+i, PNRLEN);
	i+=PNRLEN;
	break;

    default:
	my_sysabort("Unknown type %d", req->type);
    }

    *len = i;
}

void
decompose_request(struct request *req, char *buf, int *len)
{
    int i=0;
    int tmplen=0;
    u_int32_t tmp;

    if(i+sizeof(tmp) > *len) { *len=0; return;}
    bcopy(buf+i, &tmp, sizeof(tmp));
    req->reqno = ntohl(tmp);
    i+=sizeof(tmp);

    if(i+sizeof(tmp) > *len) { *len=0; return;}
    bcopy(buf+i, &tmp, sizeof(tmp));
    req->type = ntohl(tmp);
    i+=sizeof(tmp);

    if(i+sizeof(tmp) > *len) { *len=0; return;}
    bcopy(buf+i, &tmp, sizeof(tmp));
    req->id = ntohl(tmp);
    i+=sizeof(tmp);

    switch(req->type) 
    {

    case REQ_CREATE:
	if(i+PW_BUFLEN > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.create.new, PW_BUFLEN);
	req->arg.create.new[PW_BUFLEN] = '\0';
	i+=PW_BUFLEN;
	break;

    case REQ_DELETE:
	if(i+PW_BUFLEN > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.delete.old, PW_BUFLEN);
	req->arg.delete.old[PW_BUFLEN] = '\0';
	i+=PW_BUFLEN;
	break;

    case REQ_UPDATE:
	if(i+PW_BUFLEN > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.update.old, PW_BUFLEN);
	req->arg.update.old[PW_BUFLEN] = '\0';
	i+=PW_BUFLEN;

	if(i+PW_BUFLEN > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.update.new, PW_BUFLEN);
	req->arg.update.new[PW_BUFLEN] = '\0';
	i+=PW_BUFLEN;
	break;

    case REQ_GETPWNAM:
	if(i+PW_BUFLEN > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.getpwnam.name, PW_BUFLEN);
	req->arg.getpwnam.name[PW_BUFLEN] = '\0';
	i+=PW_BUFLEN;
	break;

    case REQ_GETPWUID:
	if(i+sizeof(tmp) > *len) { *len=0; return;}
	bcopy(buf+i, &tmp, sizeof(tmp));
	req->arg.getpwuid.uid = ntohl(tmp);
	i+=sizeof(tmp);
	break;

    case REQ_FIRSTUID:
	if(i+sizeof(tmp) > *len) { *len=0; return;}
	bcopy(buf+i, &tmp, sizeof(tmp));
	req->arg.firstuid.uid = ntohl(tmp);
	i+=sizeof(tmp);
	break;

    case REQ_NEXTUID:
	if(i+sizeof(tmp) > *len) { *len=0; return;}
	bcopy(buf+i, &tmp, sizeof(tmp));
	req->arg.nextuid.uid = ntohl(tmp);
	i+=sizeof(tmp);
	break;

    case REQ_PNRCREATE:
	tmplen=PNRLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.pnrcreate.pnr, tmplen);
	req->arg.pnrcreate.pnr[tmplen] = '\0';
	i+=tmplen;

	tmplen=NAMELEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.pnrcreate.name, tmplen);
	req->arg.pnrcreate.name[tmplen] = '\0';
	i+=tmplen;

	break;

    case REQ_PNRDELETE:
	tmplen=NAMELEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.pnrdelete.name, tmplen);
	req->arg.pnrdelete.name[tmplen] = '\0';
	i+=tmplen;

	break;

    case REQ_COMMENTCHANGE:
	tmplen=PNRLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.commentchange.pnr, tmplen);
	req->arg.commentchange.pnr[tmplen] = '\0';
	i+=tmplen;

	tmplen=COMMENTLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.commentchange.comment, tmplen);
	req->arg.commentchange.comment[tmplen] = '\0';
	i+=tmplen;

	break;

    case REQ_COMMENTDELETE:
	tmplen=PNRLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.commentdelete.pnr, tmplen);
	req->arg.commentdelete.pnr[tmplen] = '\0';
	i+=tmplen;
	break;

    case REQ_INFOGETBYPNR:
	tmplen=PNRLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.infogetbypnr.pnr, tmplen);
	req->arg.infogetbypnr.pnr[tmplen] = '\0';
	i+=tmplen;
	break;

    case REQ_PNRGETBYNAME:
	tmplen=NAMELEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.pnrgetbyname.name, tmplen);
	req->arg.pnrgetbyname.name[tmplen] = '\0';
	i+=tmplen;

	break;

    case REQ_NAMESGETBYPNR:
	tmplen=PNRLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.namesgetbypnr.pnr, tmplen);
	req->arg.namesgetbypnr.pnr[tmplen] = '\0';
	i+=tmplen;
	break;

    case REQ_COMMENTGETBYPNR:
	tmplen=PNRLEN;
	if(i+tmplen > *len) { *len=0; return;}
	bcopy(buf+i, &req->arg.commentgetbypnr.pnr, tmplen);
	req->arg.commentgetbypnr.pnr[tmplen] = '\0';
	i+=tmplen;
	break;

    default:
	syslog(LOG_ERR, "Unknown type %d", req->type);
	return;
    }
}
