
/* date.c
 *
 */

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <stdio.h>
#include <ctype.h>
#include "bool.h"
#include "date.h"

extern int atoi ();
extern int strcmp ();
extern int gettimeofday ();
extern struct tm *localtime ();

static bool is_leapyear (int year);


enum 
{ JAN = 1, FEB, MAR, APR, MAY, JUN, JUL, AUG, SEP, NOV, DEC };

static int ndays_in_month[] =
{ 0, 31, 0, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };


/* date2int
 *
 * convert string with date (format YYMMDD) to number of days since 1 Jan 1970
 * returns -1 when fed a bad format string
 */
int
date2int (datestr)
    char *datestr;
{
    int  target_year;
    int  target_month;
    int  target_day;
    int  year;
    int  month;
//    int  day;
//    int  ndays_in_year;
    int  ndays_sofar;
    int  i;
    int  date;
    char *teststr;
    static char buf[] = "YYMMDD";

    /*
     * A first check of format - just make sure algorithm below won't crash.
     */

    if (datestr == NULL)
	return -1;
    for (i = 0; i < 6; i++)
    {
	if (isdigit (datestr[i]))
	    buf[i] = datestr[i];
	else
	    return -1;
    }
    if (datestr[6] != '\0')
	return -1;

    /*
     * Pick up the parts from the date string (a bit ugly).
     */
    
    target_day = atoi (buf + 4);
    buf[4] = '\0';
    target_month = atoi (buf + 2);
    buf[2] = '\0';
    target_year = atoi (buf + 0);

    /* 
     * Years in the interval [0,70) are interpreted as being in the 21th
     * century, years in the interval [70,99] are interpreted as being in
     * the 20th (I almost wrote present) century.
     */

    if (target_year < 70)
	target_year += 2000;
    else
	target_year += 1900;

    /*
     * Start at 1 Jan 1970 and count the days.
     */
    
    ndays_sofar = 0;
    for (year = 1970; year < target_year; year++)
	ndays_sofar += (is_leapyear (year) ? 366 : 365);
    ndays_in_month[FEB] = (is_leapyear (target_year) ? 29 : 28);
    for (month = JAN; month < target_month; month++)
	ndays_sofar += ndays_in_month[month];
    ndays_sofar += target_day - 1;
    date = ndays_sofar;

    /*
     * Second check of format - remove all doubt. We convert the result back to
     * a string and compare with the given argument. If they match we can be
     * sure that the given argument had the right format.
     */

    teststr = date2str (date);
    if (strcmp (datestr, teststr) != 0)
	return -1;

    return date;
}


/* date2str
 *
 * convert a date integer (number of days since 1 Jan 1970) into a date
 * string (YYMMDD).
 */
char *
date2str (int date)
{
    int  year;				/* will to the right year */
    int  month;				/* will grow to the right month */
    int  day;				/* will grow to the right day */
    int  ndays_in_year;			/* # of days in the current year */
    int  ndays_left;			/* Day count */
    static char buf[] = "YYMMDD";	/* Static buffer to return to caller */
    
    /* Can't handle dates before 1 Jan 1970 */

    if (date < 0)
	return NULL;

    /* 
     * Find the right year by decrementing the day count until we have less
     * than the number of days in the year left.
     */

    ndays_left = date;
    year = 1970;
    ndays_in_year = (is_leapyear (year) ? 366 : 365);
    while (ndays_left >= ndays_in_year)
    {
	ndays_left -= ndays_in_year;
	year++;
	ndays_in_year = (is_leapyear (year) ? 366 : 365);
    }

    /* February has 29 days if it's leapyear - 28 otherwise */

    ndays_in_month[FEB] = (is_leapyear (year) ? 29 : 28);

    /* 
     * Find the right month by decrementing the day count until we have less
     * than the number of days in the month left.
     */

    month = JAN;
    while (ndays_left >= ndays_in_month[month])
    {
	ndays_left -= ndays_in_month[month];
	month++;
    }

    /*
     * The remaining day count is the "offset" into the month. If the
     * day count is 0 - it's the first of the month.
     */
    
    day = ndays_left + 1; 

    /*
     * Write the date in YYMMDD format into a static buffer and return a
     * pointer to this buffer.
     */

    sprintf (buf, "%02d%02d%02d", year % 100, month, day);
    return buf;
}

static bool
is_leapyear (int year)
{
    if (year % 4)
	return FALSE;
    if (year % 100)
	return TRUE;
    if (year % 400)
	return FALSE;
    return TRUE;
}

/* datetoday
 *
 * return todays date string
 */
char *
datetoday (void)
{
    static char buf[] = "YYMMDD";
    struct timeval t;
    struct timezone tz;
    struct tm *time;
//    int res;

    gettimeofday (&t, &tz);
    time = localtime (&t.tv_sec);
    sprintf (buf, "%02d%02d%02d",
	     time->tm_year % 100,
	     time->tm_mon + 1,
	     time->tm_mday);
    return buf;
}

