
#include <ctype.h>
#include <z/bool.h>
#include <z/error.h>
#include <pw.h>
#include "params.h"

#define NULL 0

extern char *strdup ();

#define NAME_OK			0
#define NAME_TAKEN		1
#define NAME_IMPOSSIBLE		2
#define NAME_ERROR		3
#define NAME_GIVEUP		4

static int try ();
static int last_resort ();
static char mapchar ();

static char buf[9];
static int pos;
static int d;

#define FIRST	0
#define MIDDLE	1
#define LAST	2

static char *name[3];
static int len[3];

static struct config
{
    int n1, i1;
    int n2, i2;
    int n3, i3;
}
conf[] =
{
    { FIRST, 0, 	MIDDLE, 0, 	LAST, 0 },
    { FIRST, 0, 	LAST, 0, 	LAST, 1 },
    { FIRST, 0, 	FIRST, 1, 	LAST, 0 },
    { LAST, 0, 		LAST, 1, 	LAST, 2 },
    { FIRST, 0, 	FIRST, 1, 	FIRST, 2 },
};
static int nconfs = sizeof (conf) / sizeof (struct config);


/* full2log
 *
 * Returns a "suitable" logname for the given fullname. The returned string
 * is currently not used by any user. If an error occurs pw_errno will be
 * set properly and NULL will be returned. 
 *
 * If NULL is returned and pw_errno is PW_EOK, full2log has not been able to
 * find any logname for the user, all alternatives that have been tried have
 * been found to be in use. This event is most unlikely.
 */
char *
full2log (fullname, prefix, pw_d)
    char *fullname;
    char *prefix;
    int pw_d;
{
    char *cp;
    int i;
    int res;
    extern bool nohyphen;

    d = pw_d;

    /*
     * Add prefix
     */

    bzero (buf, 9);

    for (pos = 0; prefix[pos] != '\0' && pos < (nohyphen ? 5 : 4); pos++)
	buf[pos] = prefix[pos];

    if (!nohyphen)
	buf[pos++] = LOGINDELIM;
    

    /*
     * Split full name into first, middle and last name
     */

    cp = fullname;
    
    for (i = 0; i < 3; i++)
    {
	while (*cp == ' ')
	    cp++;

	name[i] = cp;
	while (*cp != '\0' && *cp != ' ')
	    cp++;
	len[i] = cp - name[i];
    }


    /*
     * If there are only to components, these are the first and last names,
     * not the first and middle names.
     */

    if (len[LAST] == 0)
    {
	name[LAST] = name[MIDDLE];
	len[LAST] = len[MIDDLE];
	len[MIDDLE] = 0;
    }


    /*
     * First try the static configurations
     */

    for (i = 0; i < nconfs; i++)
    {
	res = try (conf[i].n1, conf[i].i1,
		   conf[i].n2, conf[i].i2,
		   conf[i].n3, conf[i].i3);

	switch (res)
	{
	case NAME_OK:
	    return strdup (buf);
	    
	case NAME_ERROR:
	    return NULL;

	case NAME_TAKEN:
	case NAME_IMPOSSIBLE:
	    break;

	default:
	    ASSERT (FALSE);
	}
    }


    /*
     * Now try the two dynamic configurations
     */

    for (i = len[LAST] - 1; i > 1; i--)
    {
	res = try (FIRST, 0, LAST, 0, LAST, i);

	switch (res)
	{
	case NAME_OK:
	    return strdup (buf);
	    
	case NAME_ERROR:
	    return NULL;

	case NAME_TAKEN:
	case NAME_IMPOSSIBLE:
	    break;

	default:
	    ASSERT (FALSE);
	}
    }

    for (i = len[FIRST] - 1; i > 1; i--)
    {
	res = try (FIRST, 0, FIRST, i, LAST, 0);

	switch (res)
	{
	case NAME_OK:
	    return strdup (buf);
	    
	case NAME_ERROR:
	    return NULL;

	case NAME_TAKEN:
	case NAME_IMPOSSIBLE:
	    break;

	default:
	    ASSERT (FALSE);
	}
    }


    /*
     * All "smart" attempts have failed - try the last resort.
     */

    res = last_resort ();
    
    switch (res)
    {
    case NAME_OK:
	return strdup (buf);
	
    case NAME_ERROR:
    case NAME_GIVEUP:
	return NULL;
	
    default:
	ASSERT (FALSE);
    }

}


static int
try (n1, i1, n2, i2, n3, i3)
    int n1, i1;
    int n2, i2;
    int n3, i3;
{
    char *line;

    if (i1 >= len[n1] || i2 >= len[n2] || i3 >= len[n3])
	return NAME_IMPOSSIBLE;

    buf[pos + 0] = mapchar (name[n1][i1]);
    buf[pos + 1] = mapchar (name[n2][i2]);
    buf[pos + 2] = mapchar (name[n3][i3]);

    line = pw_getpwnam (d, buf);
    if (line != NULL)
    {
	free (line);
	return NAME_TAKEN;
    }
    else
    {
	if (pw_errno == PW_ENOTFOUND)
	{
	    pw_errno = PW_EOK;
	    return NAME_OK;
	}
	else
	    return NAME_ERROR;
    }    
}


static int
last_resort ()
{
    char c1, c2, c3;
    char *line;

    for (c1 = 'a'; c1 <= 'z'; c1++)
	for (c2 = 'a'; c2 <= 'z'; c2++)
	    for (c3 = 'a'; c3 <= 'z'; c3++)
	    {
		buf[pos + 0] = c1;
		buf[pos + 1] = c2;
		buf[pos + 2] = c3;

		line = pw_getpwnam (d, buf);
		if (line != NULL)
		    free (line);
		else
		{
		    if (pw_errno == PW_ENOTFOUND)
		    {
			pw_errno = PW_EOK;
			return NAME_OK;
		    }
		    else
			return NAME_ERROR;
		}
	    }

    pw_errno = PW_EOK;
    return NAME_GIVEUP;
}


static char
mapchar (c)
    char c;
{

    switch (c)
    {

    case '{':
    case '[':
    case '}':
    case ']':
    case '':
    case '':
    case '':
    case '':
    case '':
    case '':
    case '':
    case '':
	c = 'a';
	break;
	
    case '':
    case '':
	c = 'd';
	break;
	
    case '':
    case '':
	c = 'e';
	break;
	
    case '':
    case '':
	c = 'i';
	break;
	
    case '':
    case '':
    case '|':
    case '\\':
    case '':
    case '':
    case '':
    case '':
	c = 'o';
	break;
	
    case '':
    case '':
	c = 't';
	break;
	
    case '':
    case '':
	c = 'u';
	break;
	
    case '':
    case '':
	c = 'y';
	break;
	
    default:
	if (c >= 'A' && c <= 'Z')
	    c += 'a' - 'A';
	else if (c < 'a' || c > 'z')
	    c = 'x';
	break;
    }

    return c;
}
