
/* chpassd.c
 *
 */

#include <des.h>
#include <krb.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/time.h>

#ifdef HAVE_SYS_SIGNAL_H
#include <sys/signal.h>
#endif

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif


#include <stdio.h>
#include <syslog.h>
#include <errno.h>

#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif

#include <z/bool.h>
#include <z/error.h>
#include <z/itemize.h>
#include <z/file.h>

#include <../lib/pw.h>
#include <../lib/proto.h>

#define LOGFILE	"/var/chpass/passwd.log"
#define ACLFILE "/var/chpass/passwd.acl"
#define INFOACLRFILE "/var/chpass/infor.acl"
#define INFOACLWFILE "/var/chpass/infow.acl"

extern char *strcpy();
extern char *strlen();

/*
 * Forward declarations
 */

void boilerplate ();
RETSIGTYPE reaper ();
void dispatch ();
void sendreply ();
bool authority ();
bool authority_change_self ();
int decompose ();
void logchange ();
bool info_authority_read ();
bool info_authority_write ();


/*
 * Global variables
 */

char *			progname = "chpassd";
struct hostent *	host;		/* Used with gethostbyname(3) */
char *			hostname;	/* Name of this machine */
char *			shortname;	/* Shortname of this machine */
struct servent *	serv;		/* Used with getservbyname(3) */
struct sockaddr_in	serv_addr;	/* Servers socket address */
struct sockaddr_in	clnt_addr;	/* Clients socket address */
int			addrlen;
int 			sock;		/* Socket descriptor */  
int  			d;		/* pw-descriptor */
AUTH_DAT		auth;		/* Unpacked data from authenticator */
MSG_DAT			msg;		/* Decoded message */
des_key_schedule	sched;		/* For private communication */
KTEXT_ST		xk;
KTEXT			k = &xk;	/* For decoding authenticator */
struct request		req;		/* Request structure */
struct reply		rep;		/* Reply structure */
char			recvbuf[4096];	/* Buffer for recieving requests */
char			sendbuf[4096];	/* Buffer for sending replies */
int			authlen;	/* The length of the authenticator */
int			reqlen;		/* The length of the request */
FILE *			logfp;		/* Log file */

/*
 * Library routine declarations
 */

extern char *index ();
extern char *rindex ();
extern char *malloc ();
extern char *strdup ();

/* For sendreply () */
#define ENCRYPTION_ON	255
#define ENCRYPTION_OFF	0


int
main (argc, argv)
    int argc;
    char *argv[];
{
    int        	res;		/* General purpose result code */
    char *	tmp;

    
    /*
     * Set program name - for error messages
     */

    if ((progname = rindex (argv[0], '/')) == NULL)
        progname = argv[0];
    else
        progname++;
    argv[0] = progname;

    
    /*
     * Handle command line arguments 
     */

    if (argc > 1 && strcmp ("-d", argv[1]) == 0)
	pw_debug = TRUE;
    

    /*
     * Ensure that effective user is root
     */

    if (!pw_debug && geteuid () != 0)
	my_abort ("must be run by root");


    /*
     * Get service and bind to socket
     */

    hostname = malloc (MAXHOSTNAMELEN + 1);
    res = gethostname (hostname, MAXHOSTNAMELEN);
    if (res == -1)
	my_sysabort ("gethostname(2) fails");
    host = gethostbyname (hostname);
    if (host == NULL)
	my_abort ("gethostbyname(3) fails for %s", hostname);

    tmp = index (hostname, '.');
    if (tmp == NULL)
	shortname = hostname;
    else
    {
	char *src, *dst;

	shortname = malloc (tmp - hostname + 1);
	src = hostname;
	dst = shortname;
	while (*src != '.')
	    *dst++ = *src++;
	*dst = '\0';
    }

	
    
    serv = getservbyname ("chpass", "udp");
    if (serv == NULL)
	my_abort ("can't get service entry for chpass/udp");

    bzero ((char *) &serv_addr, sizeof (serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_port = serv->s_port;
    bcopy (host->h_addr, (char *) &serv_addr.sin_addr, host->h_length);

    sock = socket (AF_INET, SOCK_DGRAM, 0);
    if (sock == -1)
	my_sysabort ("socket(2) fails");

    res = bind (sock, (struct sockaddr *) &serv_addr, sizeof (serv_addr));
    if (res == -1)
	my_sysabort ("bind(2) fails");


    /*
     * Open pw-channel.
     */

#ifdef NONIS
    d = pw_open (PASSWDPATH, PW_NDBM, 0);
#else
    d = pw_open (NULL, PW_NDBM, 0);
#endif
    if (d == -1)
	pw_abort ();

    
    /*
     * Open logfile
     */

    logfp = fopen (LOGFILE, "a");
    if (logfp == NULL)
	my_sysabort ("can't open logfile (%s)", LOGFILE);


    /*
     * Fork off a child that becomes the daemon
     */

    if (!pw_debug)
    {
	int pid;
	int ndesc;
	int d;
    
	/*
	 * Ignore terminal signals
	 */

	signal (SIGTTOU, SIG_IGN);
	signal (SIGTTIN, SIG_IGN);
	signal (SIGTSTP, SIG_IGN);
	

	pid = fork ();
	if (pid == -1)
	    my_sysabort ("couldn't fork(2)");
	
	if (pid != 0)
	    exit (0);			/* Parent just exits */

	/*
	 * Release from process group
	 */

	if (setpgrp (0, getpid ()) == -1)
	    my_sysabort ("couldn't setpgrp(2)");

	
	/*
	 * Release daemon from controlling terminal
	 */

	d = open ("/dev/tty", O_RDWR);
	if (d >= 0)
	{
	    ioctl (d, TIOCNOTTY, 0);
	    close (d);
	}

	/*
	 * Don't make "Device busy"
	 */

	chdir ("/tmp");


	/*
	 * Don't inherit strange umask
	 */

	umask (0);

    
	/*
	 * Redirect stdin, stdout and stderr to /dev/null
	 */

	(void) close (0);
	(void) close (1);
	(void) close (2);
	open ("/dev/null", O_RDONLY);		/* stdin */
	open ("/dev/null", O_WRONLY);		/* stdout */
	dup2 (1, 2);				/* stderr */
    }
    

    /*
     * Take care of dying children
     */

    (void) signal (SIGCHLD, reaper);


    /*
     * Go for it
     */

    boilerplate ();

    /*NOTREACHED*/
}

void
boilerplate ()
{
    int		res;

    while (TRUE)
    {

	addrlen = sizeof (clnt_addr);
	bzero (&clnt_addr, addrlen);

	/* Get request */

	res = recvfrom (sock, recvbuf, 4096, 0, (struct sockaddr *)&clnt_addr, &addrlen);
	if (res == -1)
	{
	    do_syslog (LOG_ERR, "recvfrom(2) fails: %s", strerror(errno));
	    continue;
	}
	/* 
	 * Check if this is a an authenticated call - only needed for
         * operations that change the database.
         */

	authlen = (recvbuf[0] << 8) + recvbuf[1];
	if (authlen == 0) { /* No authenticator */
	    msg.app_data = (unsigned char *) (recvbuf + 2);
	    msg.app_length = res-2;
	} else {
	  
	  /* Extract authenticator */
	  if(authlen > sizeof(k->dat)) {
	    do_syslog (LOG_WARNING, "chpassd: WARNING HACKING : Someone is trying to buffer overflow us\n");
	    
	    continue;
	  }
	  
	  reqlen = res - 2 - authlen;
	  bcopy (recvbuf + 2, k->dat, authlen);
	  k->length = authlen;
	  
	  
	  /* Check authenticator */
	  
	    res = krb_rd_req (k, "chpass", shortname, 0L, &auth, "");
	    if (res != KSUCCESS)
	      {
		do_syslog (LOG_WARNING, "%s", krb_err_txt[res]);
		rep.pw_errno = PW_EAUTHENTICATOR;
		rep.xerrno.krberr = res;
		rep.error = errno;
		rep.id = 0;
		sendreply (ENCRYPTION_OFF);
		continue;
	    }
	    des_key_sched (&auth.session, sched);
	    
	    
	    /* Extract request */
	    
#ifdef PRIVATE
	    res = krb_rd_priv (recvbuf + 2 + authlen, reqlen,
			       sched, &auth.session,
			       &clnt_addr, &serv_addr, &msg);
#else
	    res = krb_rd_safe (recvbuf + 2 + authlen, reqlen, &auth.session,
			       &clnt_addr, &serv_addr, &msg);
#endif
	    if (res != KSUCCESS)
	    {
		do_syslog (LOG_WARNING, "%s", krb_err_txt[res]);
		rep.pw_errno = PW_EREQUEST;
		rep.xerrno.krberr = res;
		rep.error = errno;
		rep.id = 0;
		sendreply (ENCRYPTION_OFF);
		continue;
	    }
	}
	dispatch ();
    }
}

void
sendreply (encryption)
    int encryption;
{
    int res=0;
    int len=0;
    char compbuf[4096];
    int complen=0;

    compose_reply(&rep, compbuf, &complen);
    if(complen == 0) 
    {
	do_syslog (LOG_ERR, "can't compose reply message");
        rep.pw_errno = PW_EREQUEST;
        sendreply (ENCRYPTION_OFF);
	return;
    }

    len +=1;
    sendbuf[0] = encryption;
    if (encryption != 0)
    {
#ifdef PRIVATE
	res = krb_mk_priv (compbuf, sendbuf + len, complen,
			   sched, &auth.session, &serv_addr, &clnt_addr);
#else
	res = krb_mk_safe (compbuf, sendbuf + len, complen,
			   &auth.session, &serv_addr, &clnt_addr);
#endif
	if (res == -1)
	{
	    do_syslog (LOG_ERR, "can't encode reply message");
	    logchange ();
	    return;
	}
	len+= res;
    }
    else
    {
	bcopy (compbuf, sendbuf + len, complen);
	len+= complen;
    }
    
    res = sendto (sock, sendbuf, len , 0, (struct sockaddr *)&clnt_addr, sizeof (clnt_addr));
    if (res == -1)
	do_syslog (LOG_ERR, "can't send reply message; sendto(2) fails: %s",
		   strerror(errno));
    logchange ();
}

RETSIGTYPE
reaper ()
{
    while (wait3 (0, WNOHANG, (struct rusage *) 0) > 0)
        ;
}

void
dispatch ()
{
    int  	res;
    char *	line;
    int len;

    len = msg.app_length;
    decompose_request(&req, msg.app_data, &len);
    if(len == 0) {
	do_syslog(LOG_ERR, "Couldn't decompose request");
        rep.pw_errno = PW_EREQUEST;
        sendreply (ENCRYPTION_OFF);
	return;
    }

    rep.id = req.id;

    switch (req.type)
    {


    /*
     * NULL just tells that we are alive
     */

    case REQ_NULL:
	rep.pw_errno = 0;
	sendreply (ENCRYPTION_OFF);
	return;


    /*
     * UPDATE - request to update passwd-file and NIS/YP map
     */

    case REQ_UPDATE:
	req.arg.update.old[PW_BUFLEN - 1] = '\0';
	req.arg.update.new[PW_BUFLEN - 1] = '\0';
	res = authority_update ();
	if (res != PW_EOK)
	{
	    rep.pw_errno = res;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	pw_update (d, req.arg.update.old, req.arg.update.new);
	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;

    /*
     * DELETE - delete an old user
     */

    case REQ_DELETE:
	req.arg.delete.old[PW_BUFLEN - 1] = '\0';
	if (authlen == 0 || !authority_special ())
	{
	    rep.pw_errno = PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	pw_delete (d, req.arg.delete.old);
	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;

    /*
     * CREATE - create a new user
     */

    case REQ_CREATE:
	req.arg.create.new[PW_BUFLEN - 1] = '\0';
	if (authlen == 0 || !authority_special ())
	{
	    rep.pw_errno = PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	pw_create (d, req.arg.create.new);
	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;
	

    /*
     * FIRSTUID - return first free uid that is not less than the argument.
     */

    case REQ_FIRSTUID:
	rep.res.uid = pw_firstuid (d, req.arg.firstuid.uid);
	rep.pw_errno = pw_errno;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;


    /*
     * NEXTUID - return next in-use uid that is not less than the argument.
     */

    case REQ_NEXTUID:
	rep.res.uid = pw_nextuid (d, req.arg.nextuid.uid);
	line = pw_prefetched (d);
	strcpy (rep.res.line, line);
	free (line);
	rep.pw_errno = pw_errno;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;


   /*
    * GETPWUID - return the matching  passwd(5)-entry for the uid
    */

    case REQ_GETPWUID:
	line = pw_getpwuid (d, req.arg.getpwuid.uid);
	rep.pw_errno = pw_errno;
	rep.error = errno;
	if (pw_errno == PW_EOK)
	    bcopy (line, rep.res.line, strlen (line) + 1);
	sendreply (ENCRYPTION_OFF);
	return;


   /*
    * GETPWNAM - return the matching  passwd(5)-entry for the name
    */

    case REQ_GETPWNAM:
	req.arg.getpwnam.name[PW_BUFLEN - 1] = '\0';
	line = pw_getpwnam (d, req.arg.getpwnam.name);
       	rep.pw_errno = pw_errno;
	rep.error = errno;
	if (pw_errno == PW_EOK)
	    bcopy (line, rep.res.line, strlen (line) + 1);
	sendreply (ENCRYPTION_OFF);
	return;


    /*
     * PNRCREATE - request to CREATE pnr
     */

    case REQ_PNRCREATE:
	req.arg.pnrcreate.pnr[PNRLEN - 1] = '\0';
	req.arg.pnrcreate.name[NAMELEN - 1] = '\0';
	
	if (!info_authority_write ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	pnr_create (d, req.arg.pnrcreate.pnr, req.arg.pnrcreate.name);
	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;

    /*
     * PNRDELETE - request to DELETE pnr 
     */

    case REQ_PNRDELETE:
	req.arg.pnrdelete.name[NAMELEN - 1] = '\0';
	if (!info_authority_write ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	pnr_delete (d, req.arg.pnrdelete.name);
	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;

    /*
     * COMMENTCHANGE - request change a comment
     */

    case REQ_COMMENTCHANGE:
	req.arg.commentchange.pnr[PNRLEN - 1] = '\0';
	req.arg.commentchange.comment[COMMENTLEN - 1] = '\0';
	if (!info_authority_write ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	comment_change (d, req.arg.commentchange.pnr, req.arg.commentchange.comment);

	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;

    case REQ_COMMENTDELETE:
	req.arg.commentdelete.pnr[PNRLEN - 1] = '\0';
	if (!info_authority_write ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	comment_delete (d, req.arg.commentdelete.pnr);

	rep.pw_errno = pw_errno;
	rep.xerrno.syntaxerr = pw_syntaxerr;
	rep.error = errno;
	sendreply (ENCRYPTION_OFF);
	return;

   /*
    * INFOGETBYPNR - return infobypnr
    */

    case REQ_INFOGETBYPNR:
	req.arg.infogetbypnr.pnr[PNRLEN - 1] = '\0';
	if (!info_authority_read ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	line = info_getbypnr (d, req.arg.infogetbypnr.pnr);
       	rep.pw_errno = pw_errno;
	rep.error = errno;
	if (pw_errno == PW_EOK)
	    bcopy (line, rep.res.line, strlen (line) + 1);
	sendreply (ENCRYPTION_OFF);
	free(line); /* Borde vll vara dr? Men varfr har inte alla andra det? */
	return;

   /*
    * NAMESGETBYPNR - return namesbypnr
    */

    case REQ_NAMESGETBYPNR:
	req.arg.namesgetbypnr.pnr[PNRLEN - 1] = '\0';
	if (!info_authority_read ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	line = names_getbypnr (d, req.arg.namesgetbypnr.pnr);
       	rep.pw_errno = pw_errno;
	rep.error = errno;
	if (pw_errno == PW_EOK)
	    bcopy (line, rep.res.line, strlen (line) + 1);
	sendreply (ENCRYPTION_OFF);
	free(line); /* Borde vll vara dr? Men varfr har inte alla andra det? */
	return;

   /*
    * COMMENTGETBYPNR - return commentbypnr
    */

    case REQ_COMMENTGETBYPNR:
	req.arg.commentgetbypnr.pnr[PNRLEN - 1] = '\0';
	if (!info_authority_read ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	line = comment_getbypnr (d, req.arg.commentgetbypnr.pnr);
       	rep.pw_errno = pw_errno;
	rep.error = errno;
	if (pw_errno == PW_EOK)
	    bcopy (line, rep.res.line, strlen (line) + 1);
	sendreply (ENCRYPTION_OFF);
	free(line); /* Borde vll vara dr? Men varfr har inte alla andra det? */
	return;

   /*
    * PNRGETBYNAME - return pnrbyname
    */

    case REQ_PNRGETBYNAME:
	req.arg.pnrgetbyname.name[NAMELEN - 1] = '\0';
	if (!info_authority_read ())
	{
	    rep.pw_errno =  PW_EAUTHORITY;
	    sendreply (ENCRYPTION_OFF);
	    return;
	}
	line = pnr_getbyname (d, req.arg.pnrgetbyname.name);
       	rep.pw_errno = pw_errno;
	rep.error = errno;
	if (pw_errno == PW_EOK)
	    bcopy (line, rep.res.line, strlen (line) + 1);
	sendreply (ENCRYPTION_OFF);
	free(line); /* Borde vll vara dr? Men varfr har inte alla andra det? */
	return;

    /*
     * If we got a request for an unexisting procedure
     */

    default:
	rep.pw_errno = PW_EIMPL;
	sendreply (ENCRYPTION_OFF);
	return;
    }
}

bool
authority_special ()
{
    static char principal[1024];

    strcpy (principal, auth.pname);

    if(*auth.pinst != '\0') {
    strcat (principal, ".");
    strcat (principal, auth.pinst);
    }

    strcat (principal, "@");
    strcat (principal, auth.prealm);

    return acl_exact_match (ACLFILE, principal);
}

bool
info_authority_read ()
{
    static char principal[1024];

    strcpy (principal, auth.pname);

    if(*auth.pinst != '\0') {
    strcat (principal, ".");
    strcat (principal, auth.pinst);
    }

    strcat (principal, "@");
    strcat (principal, auth.prealm);

    return acl_exact_match (INFOACLRFILE, principal);
}

bool
info_authority_write ()
{
    static char principal[1024];

    strcpy (principal, auth.pname);

    if(*auth.pinst != '\0') {
    strcat (principal, ".");
    strcat (principal, auth.pinst);
    }

    strcat (principal, "@");
    strcat (principal, auth.prealm);

    return acl_exact_match (INFOACLWFILE, principal);
}

bool
authority_update ()
{
    int res;
    int i;
    static char localrealm[REALM_SZ];
    pw_data *old;
    pw_data *new;


    if (authlen == 0)
	return PW_EAUTHORITY;

    if (authority_special ())
	return PW_EOK;

    bzero (localrealm, REALM_SZ);
    krb_get_lrealm (localrealm, 1);
    if (strcmp (localrealm, auth.prealm) != 0)
	return PW_EAUTHORITY;

    old = pw_string_to_data (req.arg.update.old);
    if (old == NULL)
	return PW_EMEMORY;

    if (strcmp (old->pwv[PW_NAME], auth.pname) != 0)
    {
	pw_free_data (old);
	return PW_EAUTHORITY;
    }

    new = pw_string_to_data (req.arg.update.new);
    if (new == NULL)
    {
	pw_free_data (old);
	return PW_EMEMORY;
    }

    if (strcmp (old->pwv[PW_NAME], auth.pname) != 0)
    {
	pw_free_data (old);
	pw_free_data (new);
	return PW_EAUTHORITY;
    }

    for (i = 0; i < PW_NFIELDS; i++)
	if (i != PW_GECOS && i != PW_SHELL)
	    if (strcmp (old->pwv[i], new->pwv[i]) != 0)
	    {
		pw_free_data (old);
		pw_free_data (new);
		return PW_EAUTHORITY;
	    }

    if (strcmp (old->pwv[PW_SHELL], new->pwv[PW_SHELL]) != 0)
	if (!pw_check_shell (new->pwv[PW_SHELL], PW_SHELLFILE))
	{
	    pw_free_data (old);
	    pw_free_data (new);
	    return PW_EAUTHORITY;
	}
	
    for (i = 0; i < GC_NFIELDS; i++)
	if (strcmp (old->gcv[i], new->gcv[i]) == 0)
	    continue;
	else if (!gc_user_editable (i))
	{
	    pw_free_data (old);
	    pw_free_data (new);
	    return PW_EAUTHORITY;
	}
	else if (gc_syntax_field (new->gcv[i], PW_SYNTAX_NORMAL, i) != 0)
	{
	    pw_free_data (old);
	    pw_free_data (new);
	    return PW_EAUTHORITY;
	}

    pw_free_data (old);
    pw_free_data (new);

    return PW_EOK;
}


void
logchange ()
{
    struct timeval tval;

    if (rep.pw_errno != PW_EOK)
	return;

    switch (req.type)
    {

    case REQ_UPDATE:
	gettimeofday (&tval, NULL);
	fprintf (logfp, "UPDATE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\n%s\n", req.arg.update.old, req.arg.update.new);
	fflush (logfp);
	return;

    case REQ_DELETE:
	gettimeofday (&tval, NULL);
	fprintf (logfp, "DELETE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\n", req.arg.delete.old);
	fflush (logfp);
	return;

    case REQ_CREATE:
	gettimeofday (&tval, NULL);
	fprintf (logfp, "CREATE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\n", req.arg.create.new);
	fflush (logfp);
	return;
    case REQ_PNRCREATE:
      	gettimeofday (&tval, NULL);
	fprintf (logfp, "PNRCREATE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\t%s\n", req.arg.pnrcreate.pnr, req.arg.pnrcreate.name);
	fflush (logfp);
	return;

    case REQ_PNRDELETE:
	gettimeofday (&tval, NULL);
	fprintf (logfp, "PNRDELETE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\n", req.arg.pnrdelete.name);
	fflush (logfp);
	return;
    case REQ_COMMENTCHANGE:
	gettimeofday (&tval, NULL);
	fprintf (logfp, "COMMENTCHANGE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\t%s\n", req.arg.commentchange.pnr,req.arg.commentchange.comment);
	fflush (logfp);
	return;
    case REQ_COMMENTDELETE:
	gettimeofday (&tval, NULL);
	fprintf (logfp, "COMMENTDELETE %s.%s@%s %s",
		 auth.pname, auth.pinst, auth.prealm,
		 ctime (&tval.tv_sec));
	fprintf (logfp, "%s\n", req.arg.commentchange.pnr);
	fflush (logfp);
	return;
	
    default:
	return;

    }
}
