
/* file.c
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include "bool.h"
#include "file.h"
#include "str.h"

/* eachline
 *
 * The given procedure is called once for each line in the file with the given
 * pathname. The procedure should return FALSE when it discovers a syntax
 * error which it can't cope with. The procedure is called with the a string
 * (the input line) as the first argument an integer (the line number) as the
 * second.
 *
 * eachline returns:
 *
 *	FILE_EREAD if an unexpected error occurs
 *	FILE_EOPEN if the file couldn't be opened
 *      FILE_EOK (zero) on success
 *	any positive integer is the line number of a line with a syntax error
 */
int
eachline (path, proc)
    char *path;			/* Pathname of file */
    bool (*proc) ();		/* Procedure to call */
{
    FILE *fp;			/* File handle */
    char buf[FILE_BUFLEN];	/* Buffer for lines */
    int lineno;			/* Line number */
    int ok;			/* Result code from procedure */
    
    
    fp = fopen (path, "r");
    if (fp == NULL)
	return FILE_EOPEN;
    
    lineno = 0;
    while (fgets (buf, FILE_BUFLEN, fp) != NULL)
    {
	stripline (buf);
	ok = (*proc) (buf, ++lineno);
	if (!ok)
	{
	    (void) fclose (fp);
	    return lineno;
	}
    }
    
    if (ferror (fp))
    {
	(void) fclose (fp);
	return FILE_EREAD;
    }
    
    return FILE_EOK;
}


/* mapline
 *
 * The given function is called once for each line in the input file. The
 * result of the function is then written to the outfile - unless the result
 * is NULL (then nothing is written - filtering).
 *
 * mapline returns:
 *
 *	FILE_EWRITE if an unexpected error occurs with the outfile
 *      FILE_ECREAT if the output file could not be created
 *	FILE_EREAD if an unexpected error occurs with the infile
 *	FILE_EOPEN if the input file couldn't be opened
 *      FILE_EOK (zero) on success
 */
int
mapline (inpath, outpath, func)
    char *inpath;			/* Pathname of input file */
    char *outpath;			/* Pathname of output file */
    char *(*func) ();			/* Function to call */
{
    FILE *ifp;				/* File handle for input */
    FILE *ofp;				/* File handle for output */
    static char buf[FILE_BUFLEN];	/* Buffer for lines */
    char *result;			/* Result from function */
    
    ifp = fopen (inpath, "r");
    if (ifp == NULL)
	return FILE_EOPEN;
    
    ofp = fopen (outpath, "w");
    if (ofp == NULL)
	return FILE_ECREAT;
    
    while (fgets (buf, FILE_BUFLEN, ifp) != NULL)
    {
	stripline (buf);
	result = (*func) (buf);
	if (result != NULL)
	{
	    if (fputs (result, ofp) == EOF || fputc ('\n', ofp) == EOF)
	    {
		(void) fclose (ifp);
		(void) fclose (ofp);
		(void) unlink (outpath);
		return FILE_EWRITE;
	    }
	}
    }
    
    if (ferror (ifp))
    {
	(void) fclose (ifp);
	(void) fclose (ofp);
	(void) unlink (outpath);
	return FILE_EREAD;
    }

    (void) fclose (ifp);
    if (fclose (ofp) == EOF)
	return FILE_EWRITE;

    return FILE_EOK;
}
