
/* pwmkhomedirs
 *
 * This program creates home directories for users. A global spool directory
 * is scanned for files. These files have the same name as newly created
 * users. Home directories are only created for users which should have their
 * home directories on the local host.
 */

#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <strings.h>
#include <sys/param.h>
#include <z/bool.h>
#include <z/error.h>
#include <z/itemize.h>
#include <pw.h>


#define SPOOLDIR "/misc/etc/spool/homedirs"
#define SKELDIR "/usr/local/etc/nadaskel"

#define CP	"/bin/cp"
#define LN	"/bin/ln"
#define TAR	"/usr/local/gnubin/tar"
#define CHOWN	"/etc/chown"
#define RM	"/bin/rm"

#define OP_UID	2123

void mkhomedir ();

char *progname = "pwmkhomedirs";
bool batch;

int
main (argc, argv)
    int argc;
    char *argv[];
{
    int			res;
    int			d;
    DIR	*		dir;
    struct dirent *	file;
    int 		i;

    
    /*
     * Assert that we have root privileges.
     */
    
    res = setreuid (0, 0);
    if (res == -1)
	abort ("Need root privileges");
    
    
    /*
     * Open a channel to the chpass server.
     */
    
    d = pw_open (PW_SERVER, PW_REMOTE, 0);
    if (d == -1)
	pw_abort ();
    
    if (argc == 1)
    {
	batch = TRUE;
	
	/*
	 * Call the function mkhomedir for each of the files found
	 * in the spool directory
	 */
	
	dir = opendir (SPOOLDIR);
	if (dir == NULL)
	    sysabort ("Can't open spool directory (%s)", SPOOLDIR);
	
	file = readdir (dir);	/* Skip "." */
	file = readdir (dir);	/* Skip ".." */
	file = readdir (dir);
	while (file != NULL)
	{
	    mkhomedir (d, file->d_name);
	    errno = 0;
	    file = readdir (dir);
	}
	
	if (errno != 0)
	    sysabort ("Error reading spool directory (%s)", SPOOLDIR);
    }
    else
    {
	batch = FALSE;

	/*
	 * Call the function mkhomedir for each of the names given on the
	 * command line
	 */

	for (i = 1; i < argc; i++)
	    mkhomedir (d, argv[i]);
    }
    
    exit (0);
}

void
mkhomedir (d, name)
    int d;
    char *name;
{
    int  res;				/* General purpose result code */
    char *line;				/* Response from chpass-server */
    char **pwv;				/* Itemization of 'line' */
    char **pathv;			/* Itemization of some pathnames */
    int uid;				/* The users uid */
    static homedir[MAXPATHLEN];		/* Users home directory */
    static cat[MAXPATHLEN];		/* Category */
    static primary_cat[MAXPATHLEN];	/* 'cat' without optional hostpart */
    char *dash;				/* For calculating 'primary_cat' */
    static catdir[MAXPATHLEN];		/* Category directory */
    static infodir[MAXPATHLEN];		/* Skeleton directory for category */
    static char cwd[MAXPATHLEN];	/* Result of getwd(3) in 'catdir' */
    static char first[MAXPATHLEN];	/* First component of 'cwd' */
    static char second[MAXPATHLEN];	/* Second component of 'cwd' */
    static char syscmd[1024];		/* For system(3) */



    /*
     * Ask chpass-server for data about the user
     */

    line = pw_getpwnam (d, name);
    if (line == NULL)
    {
	error ("Can't find user \"%s\" in servers database", name);
	if (pw_errno != PW_ENOTFOUND)
	    pw_error ();
	free (line);
	return;
    }


    /*
     * Extract home directory and uid.
     */

    pwv = itemize (line, ":");
    free (line);
    if (pwv == NULL)
	abort ("Not enough core");
    if (pw_syntax (pwv) != 0)
    {
	error ("Syntax error in users passwd-data");
	itemfree (pwv);
	return;
    }
    strcpy (homedir, pwv[PW_DIR]);
    uid = atoi (pwv[PW_UID]);
    
    itemfree (pwv);
    

    /*
     * Extract category and check home directory "syntax" 
     */

    pathv = itemize (homedir, "/");
    if (pathv == NULL)
	abort ("Not enough core");

    if (pathv[0] == NULL || pathv[0][0] != '\0'
    ||  pathv[1] == NULL || strcmp (pathv[1], "home") != 0
    ||  pathv[2] == NULL
    ||  pathv[3] == NULL || strcmp (pathv[3], name) != 0
    ||  pathv[4] != NULL)
    {
	error ("Strange directory name for user \"%s\": %s",
	       name, homedir);
	itemfree (pathv);
	return;
    }

    strcpy (cat, pathv[2]);
    sprintf (catdir, "/home/%s", cat);
    itemfree (pathv);

    
    /*
     * Find out if the category is on this host
     *
     * This is done by doing chdir(2) to the category directory and then
     * doing getwd(3) and finally checking if the first pathname component
     * is "auto", if so we are on a automounted partition so the directory
     * is *not* on this host. If the first component is "homeXXX" (where XXX
     * can be anything) the directory *is* on this host.
     *
     * It would be better to do a yp_match in "home.amd" to check the category.
     * That wouldn't require any mounts to be done, and it would not require
     * that the automounter is operational.
     */

    res = chdir (catdir);
    if (res == -1)
    {
	syserr ("Can't chdir(2) to \"%s\"", catdir);
	return;
    }

    if (getwd (cwd) == 0)
    {
	error ("Can't getwd(3): %s", cwd);
	return;
    }

    pathv = itemize (cwd, "/");
    if (pathv == NULL)
	abort ("Not enough core");

    if (pathv[0] == NULL || pathv[0][0] != '\0'
    ||  pathv[1] == NULL)
    {
	error ("Strange directory for category %s: %s", cat, cwd);
	itemfree (pathv);
	return;
    }

    strcpy (first, pathv[1]);
    strcpy (second, pathv[2]);
    itemfree (pathv);

    if (strcmp (first, "auto") == 0)
    {
	if (!batch)
	{
	    error ("Home directory for %s is not on this host", name);
	    error ("Run this program on %s to create home directory for %s",
		   second, name);
	}
	return;
    }
    else if (strncmp (first, "home", 4) != 0)
    {
	error ("Strange directory for category %s: %s", cat, cwd);
	return;
    }


    /*
     * Create the users home directory
     */

    umask(000);

    if (mkdir (homedir, (mode_t) 0755) != 0)
    {
        syserr ("Can't create directory %s", homedir);
	return;
     }
        

    /*
     * Copy files from skeleton directory
     */

    sprintf (syscmd, "%s %s/.[a-z]* %s", CP, SKELDIR, homedir);
    res = system (syscmd);
    if (res != 0)
    {
	error ("%s: Can't copy NADA skeleton files", name);
	goto cleanup;
    }

    sprintf (syscmd, "cd %s ; %s -s /usr/local/etc/nadaskel/README",
	     homedir, LN);
    res = system (syscmd);
    if (res != 0)
    {
	error ("%s: Can't make link to README file", name);
	goto cleanup;
    }


    /*
     * Calculate the primary category. The value in 'cat' may have a
     * hostname-suffix appended to it.
     */

    strcpy (primary_cat, cat);
    dash = rindex (primary_cat, '-');
    if (dash != NULL)
	*dash = '\0';
    sprintf (infodir, "/misc/info/%s/.skel", primary_cat);


    /* 
     * Copy files from /misc/info/<category>/.skel if such a
     * directory exists
     */

    res = access (infodir, F_OK);
    if (res == 0)
    {
	sprintf (syscmd, "cd %s; %s cf - . | (cd %s ;  %s xpfB - )",
		 infodir, TAR, homedir, TAR);
	res = system (syscmd);
	if (res != 0)
	{
	    error ("%s: Can't copy %s-specific skeleton files",
		   name, primary_cat);
	    goto cleanup;
	}
    }
    else if (errno != ENOENT)
    {
	syserr ("Can't access %s", infodir);
	goto cleanup;
    }


    /*
     * Make the user own his/her files
     */

    sprintf (syscmd, "%s -R %d.0 %s", CHOWN, uid, homedir);
    res = system (syscmd);
    if (res != 0)
    {
	error ("%s: Can't make user own skeleton files", name);
	goto cleanup;
    }

    
    /*
     * Remove file in spool directory
     */

    if (batch)
    {
	res = setreuid (-1, OP_UID);
	if (res == -1)
	    sysabort ("Can't change effective uid to %d", OP_UID);
	
	res = chdir (SPOOLDIR);
	if (res == -1)
	    sysabort ("Can't chdir(2) to spool directory (%s)", SPOOLDIR);
	
	res = unlink (name);
	if (res == -1)
	    sysabort ("%s: Can't unlink(2) file in spool directory", name);
	
	res = setreuid (-1, 0);
	if (res == -1)
	    sysabort ("Can't change effective uid back to root");
    }

    if (isatty (1))
	printf ("Created home directory for user \"%s\"\n", name); 

    return;
    
 cleanup:
    sprintf (syscmd, "/bin/rm -rf %s", homedir);
    (void) system (syscmd);
    return;

}
