
/* pwcheck
 *
 */

#include <stdio.h>
#include <gdbm.h>
#include <z/bool.h>
#include <z/error.h>
#include <z/date.h>
#include <z/file.h>
#include <z/itemize.h>
#include <pw.h>

char *progname = "pwcheck";

#define NORMAL 0
#define WARN 1
#define EXPIR 2
#define ARCH 3
#define UNKNOWN 99

#define TODAY		1
#define TOMORROW	2
#define NEXTWEEK	4

struct info
{
    char name[9];
    int creday;
    int expday;
    struct {
	char now;
	char today;
	char tomorrow;
	char nextweek;
    } state;
    unsigned char category;
};
typedef struct info info;

info itab[0x10000];

bool show = FALSE;
bool verbose = FALSE;
bool doit = FALSE;

int
warndays (total)
    int total;
{
    if (total < 30)
	return 0;

    if (total > 12 * 30)
	return 30;

    return (total * 7) / (3 * 30);
}

int
state (creday, today, expday)
    int creday;
    int today;
    int expday;
{
    int wday;

    wday = expday - warndays (expday - creday);
    
    if (today < wday)
	return NORMAL;

    if (today < expday)
	return WARN;

    if (today < expday + 30)
	return EXPIR;

    return ARCH;
}

int
adminfo2state (adminfo)
    char *adminfo;
{
    if (*adminfo == '\0')
	return NORMAL;
	
    if (strcmp (adminfo, "warn") == 0)
	return WARN;

    if (strcmp (adminfo, "expir") == 0)
	return EXPIR;

    if (strcmp (adminfo, "arch") == 0)
	return ARCH;

    return UNKNOWN;
}

char *
state2adminfo (state)
    int state;
{
    switch (state)
    {
    case NORMAL:
	return "";

    case WARN:
	return "warn";

    case EXPIR:
	return "expir";

    case ARCH:
	return "arch";

    default:
	return "unknown";
    }
}

char *
state2text (state)
    int state;
{
    char *adminfo;

    adminfo = state2adminfo (state);
    if (strlen (adminfo) == 0)
	return "normal";
    else
	return adminfo;
}

bool
popedo (name)
    char *name;
{
    char buf[9];
    char *p;
    int  n;
    char no[4];
    extern char *index ();

    strcpy (buf, name);
    p = index (buf, '-');
    if (p == NULL)
	return FALSE;
    *p++ = '\0';
    n = atoi (p);
    if (n < 0 || n > 999)
	return FALSE;
    sprintf (no, "%03d", n);
    if (strcmp (no, p) != 0)
	return FALSE;

    return TRUE;
}

int
icmp (i1, i2)
    info *i1;
    info *i2;
{
    return strcmp (i1->name, i2->name);
}
    
int
main(argc, argv)
    int argc;
    char **argv;
{
    int argi;
    int res;
    GDBM_FILE *db;
    datum key;
    datum val;
    pw_data *pwd;
    char line[PW_BUFLEN];
    char *old;
    char *new;
    int n;
    int i;
    int today;
    int ntoday;
    int ntomorrow;
    int nnextweek;
    int d;
    char *domain;
    char path[256];

    argi = 1;
    while (argi < argc)
    {
	if (strcmp (argv[argi], "-verbose") == 0)
	    verbose = TRUE;
	else if (strcmp (argv[argi], "-show") == 0)
	    show = TRUE;
	else if (strcmp (argv[argi], "-doit") == 0)
	    doit = TRUE;
	argi++;
    }

    today = date2int (datetoday ());

    if (yp_get_default_domain (&domain) != 0)
	my_abort ("Can't get NIS domain");
    sprintf (path, "/var/yp/%s/passwd.byname", domain);

    db = gdbm_open (path, 0, GDBM_READER, 0, NULL);
    if (db == NULL)
	my_sysabort ("Can't open database");

    if (verbose)
	fprintf (stderr, "Scanning database...\n");
    
    n = 0;
    for (key = gdbm_firstkey (db); key.dptr != NULL; key = gdbm_nextkey (db, key))
    {
	if (key.dsize > 2 && strncmp (key.dptr, "YP_", 3) == 0)
	    continue;

	if (verbose)
	    fprintf (stderr, "\015%d", n);

	val = gdbm_fetch (db, key);
	strncpy (line, val.dptr, val.dsize);
	line[val.dsize] = '\0';

	pwd = pw_string_to_data (line);
	if (pwd == NULL)
	    my_abort ("Memory allocation fails");

	res = pw_syntax (pwd, PW_SYNTAX_NORMAL);
	if (res != PW_SYNTAX_EOK)
	{
	    my_error ("Syntax error in entry: %s\n%s",
		   pw_syntax_message (res), line);
	    continue;
	}

	if (popedo (pwd->pwv[PW_NAME]))
	    continue;

	strcpy (itab[n].name, pwd->pwv[PW_NAME]);
	itab[n].creday = date2int (pwd->gcv[GC_CREATIONDATE]);
	itab[n].expday = date2int (pwd->gcv[GC_EXPIRATIONDATE]);
	itab[n].state.now = adminfo2state (pwd->gcv[GC_ADMINFO]);

	pw_free_data (pwd);

	itab[n].state.today =
	    state (itab[n].creday, today, itab[n].expday);
	itab[n].state.tomorrow =
	    state (itab[n].creday, today + 1, itab[n].expday);
	itab[n].state.nextweek =
	    state (itab[n].creday, today + 7, itab[n].expday);

	itab[n].category = 0;
	if (itab[n].state.now != itab[n].state.today)
	    itab[n].category |= TODAY;
	if (itab[n].state.today != itab[n].state.tomorrow)
	    itab[n].category |= TOMORROW;
	if (itab[n].state.tomorrow != itab[n].state.nextweek)
	    itab[n].category |= NEXTWEEK;

	n++;
    }
    if (verbose)
	fprintf (stderr, "\015      \015");

    gdbm_close (db);

    if (verbose)
	fprintf (stderr, "A total of %d accounts in database\n", n);


    if (verbose)
	fprintf (stderr, "Sorting names...\n");
    
    qsort (itab, n, sizeof (info), icmp);


    if (show)
    {
	printf ("Changes to be done today:\n");
	printf ("========================================\n");
    }
    ntoday = 0;
    for (i = 0; i < n; i++)
	if (itab[i].state.now != itab[i].state.today)
	{
	    if (show)
	    {
		printf ("%-10s", itab[i].name);
		printf ("%-6s", state2text (itab[i].state.now));
		printf (" --> ");
		printf ("%s", state2text (itab[i].state.today));
		printf ("\n");
	    }
	    ntoday++;
	}
    
    if (show)
    {
	printf ("\n\nChanges to be done tomorrow:\n");
	printf ("========================================\n");
    }
    ntomorrow = 0;
    for (i = 0; i < n; i++)
	if (itab[i].state.today != itab[i].state.tomorrow)
	{
	    if (show)
	    {
		printf ("%-10s", itab[i].name);
		printf ("%-6s", state2text (itab[i].state.today));
		printf ( "--> ");
		printf ("%s", state2text (itab[i].state.tomorrow));
		printf ("\n");
	    }
	    ntomorrow++;
	}
    
    if (show)
    {
	printf ("\n\nChanges to be done next week:\n");
	printf ("========================================\n");
    }
    nnextweek = 0;
    for (i = 0; i < n; i++)
	if (itab[i].state.tomorrow != itab[i].state.nextweek)
	{
	    if (show)
	    {
		printf ("%-10s", itab[i].name);
		printf ("%-6s", state2text (itab[i].state.tomorrow));
		printf (" --> ");
		printf ("%s", state2text (itab[i].state.nextweek));
		printf ("\n");
	    }
	    nnextweek++;
	}

    fflush (stdout);

    if (verbose)
	fprintf (stderr, "A total of %d changes today\n", ntoday);
    if (verbose)
	fprintf (stderr, "A total of %d changes tomorrow\n", ntomorrow);
    if (verbose)
	fprintf (stderr, "A total of %d changes next week\n", nnextweek);


    if (doit)
    {
	if (verbose)
	    fprintf (stderr, "Committing changes...\n");

	d = pw_open (NULL, PW_REMOTE, 0);
	if (d == -1)
	    pw_abort ();
	
	for (i = 0; i < n; i++)
	{
	    if ((itab[i].category & TODAY) == 0)
		continue;

	    old = pw_getpwnam (d, itab[i].name);
	    if (old == NULL)
	    {
		fprintf (stderr, "\n");
		pw_abort ();
	    }

	    pwd = pw_string_to_data (old);
	    if (pwd == NULL)
		my_abort ("Memory allocation fails");

	    res = pw_syntax (pwd, PW_SYNTAX_NORMAL);
	    if (res != PW_SYNTAX_EOK)
	    {
		my_error ("Syntax error in entry: %s\n%s",
		       pw_syntax_message (res), old);
		free (old);
		continue;
	    }

	    if (atoi (pwd->pwv[PW_UID]) < 100)
	    {
		free (old);
		continue;
	    }

	    if (adminfo2state (pwd->gcv[GC_ADMINFO]) != itab[i].state.now)
	    {
		free (old);
		continue;
	    }

	    switch (itab[i].state.today)
	    {
	    case NORMAL:
		if (strcmp (pwd->pwv[PW_SHELL],
			    "/usr/local/bin/shExpired") == 0)
		    pwd->pwv[PW_SHELL] = "/bin/tcsh";
		break;
		
	    case WARN:
		break;
		
	    case EXPIR:
		if (strcmp (pwd->pwv[PW_SHELL],
			    "/usr/local/bin/shZAPPED") != 0)
		    pwd->pwv[PW_SHELL] = "/usr/local/bin/shExpired";
		break;
		
	    case ARCH:
		if (strcmp (pwd->pwv[PW_SHELL],
			    "/usr/local/bin/shZAPPED") != 0)
		    pwd->pwv[PW_SHELL] = "/usr/local/bin/shExpired";
		break;
		
	    case UNKNOWN:
	    default:
		free (old);
		continue;
	    }

	    pwd->gcv[GC_ADMINFO] = state2adminfo (itab[i].state.today);

	    new = pw_data_to_string (pwd);
	    
	    if (verbose)
		fprintf (stderr, "\015%-8s", itab[i].name);
	    
	    pw_update (d, old, new);
	    if (pw_errno != PW_EOK)
		pw_abort ();

	    free (old);
	    free (new);
	}

	if (verbose)
	    fprintf (stderr, "\015      \015");

	pw_close (d);
    }

    
    exit (0);
}
