/*
 *  CINAG - Chess Is Not A Game
 * 
 *  Copyright (c) 2001, 2002 PAYEMENT Arnaud
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "interface.h"

#include <GL/gl.h>
#include <GL/glu.h>

#include <iostream.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <SDL/SDL_thread.h>

#include "bitboard.h"
#include "board.h"
#include "ini_file.h"
#include "search.h"
#include "analyse.h"
#include "genmove.h"
#include "db.h"
#include "transposition.h"
#include "killer.h"
#include "debug.h"

#define MULTITASK

#define SELECTBUFSIZE 128
#define CASE_SIZE 1.8
#define DECAL 3

// used to give information to thread
typedef struct {
  board_t *board;
  bool color;
  SDL_mutex *mut;
  move_t *move;
  int *progress;
} threaddata_t;


// the thread function
int threadfunc(void *buf)
{
  move_t move;
  threaddata_t *data;
  data = (threaddata_t *)buf;
  move = SE_Search(data->board,data->color);
  if(SDL_mutexP(data->mut)==-1){
    warning(Could not lock mutex);
  }
  *data->move = move;
  *data->progress = -1;
  if(SDL_mutexV(data->mut)==-1){
    warning(Could not unlock mutex);
  }
  return true;
}

move_t search(board_t *board,bool color,interface_t *inter)
{
    SDL_Thread *thread;
    threaddata_t *data;
    move_t move;
    data = new threaddata_t;
    data->color = color;
    data->board = board;
    data->progress = new int;
    *data->progress = 0;
    data->move = new move_t;
    data->mut = SDL_CreateMutex();
    thread = SDL_CreateThread((threadfunc), (void *)data); 
    for(;;)
    {
	inter->updateEvent();
	if(SDL_mutexP(data->mut)==-1){
	    warning(Could not lock mutex);
	}
	if (*data->progress == -1) break;
	if(SDL_mutexV(data->mut)==-1){
	    warning(Could not unlock mutex);
	}
	inter->updateEvent();
	*inter << board;
    }
    SDL_KillThread(thread);
    SDL_DestroyMutex(data->mut);
    delete data->progress;
    move = *data->move;
    delete data->move;
    delete data;
    return move;
}

void interface_t::start()
{
    ini_file_t conf,theme;
    char *tmp;
    int tmpi = 0;
    bool color = 0;
    int height,width;    


    int iboard[8][8]={{ 4, 2, 3, 5, 6, 3, 2, 4},
		      { 1, 1, 1, 1, 1, 1, 1, 1},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      {-1,-1,-1,-1,-1,-1,-1,-1},
		      {-4,-2,-3,-5,-6,-3,-2,-4}};
/*/
    int iboard[8][8]={{ 0, 0, 0, 0, 0, 0, 0, 0},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      { 0, 0, 0, 0, 0, 0, 6, 1},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      { 1, 1, 1, 0, 0, 0,-6,-1},
		      { 0, 0, 0, 0, 0, 0, 0, 0},
		      {-1,-1,-1, 0, 0, 0, 0, 0},
		      { 0, 0, 0, 0, 0, 0, 0, 0}};
/*/
    *this << "loading ... \n";
    *this << "Board\n";
    board = BO_Convert(iboard);
    
    *this << "GenMove\n";
    GN_Init();
    *this << "DataBase\n";
    DB_Init();
    *this << "Transposition\n";
    
    // reading conf.ini
    conf.read("conf.ini");
    width = conf.getintvalue("INTERFACE","width");
    height = conf.getintvalue("INTERFACE","height");

    tmp = conf.getcharvalue("INTERFACE", "fullscreen");
    if(strcmp(tmp, "true") == 0) window.create(width,height,true);
    else window.create(width,height,false);
    window.setCaption("Cinag - 1.1.4 - PAYEMENT Arnaud - sauron@webmails.com");

    tmp = conf.getcharvalue("INTERFACE","theme");
    theme.read(tmp);
    
    *this << "3d Model\n";
    tmp = theme.getcharvalue("OBJECTS","dialog");
    dialog.load(tmp);
    tmp = theme.getcharvalue("OBJECTS","rook");
    rook.load(tmp);
    tmp = theme.getcharvalue("OBJECTS","bishop");
    bishop.load(tmp);
    tmp = theme.getcharvalue("OBJECTS","queen");
    queen.load(tmp);
    tmp = theme.getcharvalue("OBJECTS","king");
    king.load(tmp);
    tmp = theme.getcharvalue("OBJECTS","knight");
    knight.load(tmp);
    tmp = theme.getcharvalue("OBJECTS","pawn");
    pawn.load(tmp);
    
    *this << "Texture\n";
    tmp = theme.getcharvalue("TEXTURES", "piece_texture");
    if(strcmp(tmp, "true") == 0) {
	tmpi = conf.getintvalue("INTERFACE", "piece_texture_quality");
	pieceTexture = true;
	tmp = theme.getcharvalue("TEXTURES","white");
	whiteText.load(tmp,tmpi,true);
	tmp = theme.getcharvalue("TEXTURES","black");
	blackText.load(tmp,tmpi,true);
    }
    tmp = theme.getcharvalue("TEXTURES", "board_texture");
    if(strcmp(tmp, "true") == 0) {
	tmpi = conf.getintvalue("INTERFACE", "board_texture_quality");
	boardTexture = true;
	tmp = theme.getcharvalue("TEXTURES","white_board");
	whiteBoardText.load(tmp,tmpi,false);
	tmp = theme.getcharvalue("TEXTURES","black_board");
	blackBoardText.load(tmp,tmpi,false);
    }
    tmp = theme.getcharvalue("TEXTURES","white_win");
    whiteWin.load(tmp,2,false);
    tmp = theme.getcharvalue("TEXTURES","black_win");
    blackWin.load(tmp,2,false);
    
    *this << "OpenGL\n";
    tmp = conf.getcharvalue("INTERFACE", "shade_model");
    if(strcmp(tmp, "smooth") == 0) glShadeModel(GL_SMOOTH);
    if(strcmp(tmp, "flat") == 0) glShadeModel(GL_FLAT);
    
    tmp = conf.getcharvalue("INTERFACE", "reflection");
    if(strcmp(tmp, "true") == 0) reflection = true;
  
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
#ifndef WIN32
    tmp = conf.getcharvalue("INTERFACE", "separate_specular_color");
    if(strcmp(tmp, "true") == 0) glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL,GL_SEPARATE_SPECULAR_COLOR);
#endif
    
    clearRed = (float)theme.getintvalue("COLORS","clear_r");
    clearGreen = (float)theme.getintvalue("COLORS","clear_g");
    clearBlue = (float)theme.getintvalue("COLORS","clear_b");
    clearRed /= 255;
    clearGreen /= 255;
    clearBlue /= 255;
    glClearColor(clearRed,clearGreen,clearBlue,1.0);
    glViewport(0,0,width-1,height-1);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(60.0,1.3,1.0,1024.0);
    
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
    float light_pos[4] = { 0.0,0.0,0.0,0.0};
    float light_amb[4] = { 1.0,1.0,1.0,1.0};
    float light_dif[4] = { 1.0,1.0,1.0,1.0};
    float light_spe[4] = { 1.0,1.0,1.0,1.0};
    
    glLightfv(GL_LIGHT0, GL_POSITION,light_pos);
    glLightfv(GL_LIGHT0, GL_AMBIENT,light_amb);
    glLightfv(GL_LIGHT0, GL_DIFFUSE,light_dif);
    glLightfv(GL_LIGHT0, GL_SPECULAR,light_spe);
    
    *this << "Game Started\n";
    
    sdlMove.start = NOPOS;
    sdlMove.stop = NOPOS;
    
    while(((rotaV > 0.2)||(rotaV < -0.2)))
    {
	rotaV = (float)(rotaV / 1.2);
	rotaH = (float)(rotaH / 1.2);
	*this << board;
    }
    while(((rotaH > 0.2)||(rotaH < -0.2)))
    {
	rotaH = (float)(rotaH / 1.2);
	*this << board;
    }
 
    tmp = conf.getcharvalue("GAME", "usercolor");
    if(strcmp(tmp, "white") == 0){
	userColor = WHITE;
	color = BLACK;
    }
    if(strcmp(tmp, "black") == 0){
	color = WHITE;
	userColor = BLACK;
    }
    *this << board;
    if(strcmp(tmp, "demo") == 0){
	move_t move;
	for(;;){
#ifdef MULTITASK
	    move = search(board,WHITE,this);
#else
	    move = SE_Search(board,WHITE);
#endif
	    BO_Move(board,move);
	    sdlMove = move;
	    this->AddToGame(move);
	    if (board->data[0][5] == 0) DB_AddGame(game,GameNbMove,WHITE);
	    *this << move;
	    *this << " - ";
	    *this << board;
#ifdef MULTITASK
	    move = search(board,BLACK,this);
#else
	    move = SE_Search(board,BLACK);
#endif
	    BO_Move(board,move);
	    sdlMove = move;
	    this->AddToGame(move);
	    if (board->data[1][5] == 0) DB_AddGame(game,GameNbMove,BLACK);
	    *this << move;
	    *this << "\n";
	    *this << board;
	}
    }
    this->loop(color);
}

void interface_t::loop(bool color)
{
    move_t move;
    if (color == WHITE)
	for(;;){
#ifdef MULTITASK
	    move = search(board,WHITE,this);
#else
	    move = SE_Search(board,WHITE);
#endif
	    BO_Move(board,move);
	    sdlMove = move;
	    this->AddToGame(move);
	    if (board->data[0][5] == 0) DB_AddGame(game,GameNbMove,WHITE);
	    *this << move;
	    *this << " - ";
	    *this << board;
	    move = getmove(board,BLACK);
	    BO_Move(board,move);
	    this->AddToGame(move);
	    if (board->data[1][5] == 0) DB_AddGame(game,GameNbMove,BLACK);
	    *this << board;
	    *this << move;
	    *this << "\n";
	}
    if (color == BLACK)
	for(;;){
	    *this << board;
	    move = getmove(board,WHITE);
	    BO_Move(board,move);
	    this->AddToGame(move);
	    if (board->data[0][5] == 0) DB_AddGame(game,GameNbMove,WHITE);
	    *this << board;
	    *this << move;
	    *this << " - ";
#ifdef MULTITASK
	    move = search(board,BLACK,this);
#else
	    move = SE_Search(board,BLACK);
#endif
	    BO_Move(board,move);
	    sdlMove = move;
	    this->AddToGame(move);
	    if (board->data[1][5] == 0) DB_AddGame(game,GameNbMove,BLACK);
	    *this << move;
	    *this << "\n";
	}
}

const interface_t& interface_t::operator<< (board_t *buffer)
{
    SDL_Event event;
    bool c = BLACK;
    int num = 0,let = 0,piece = 0;
    int board[8][8];
    int I,J;
    
    BO_Convert(buffer,board);
    
    static GLfloat board_amb[] = {1.0,1.0,1.0,0.7};
    static GLfloat board_sel[] = {0.0,0.0,1.0,0.75};
    static GLfloat board_spe[] = {0.0,0.0,0.0,1.0};
    static GLfloat board_shi[] = {0.0};
    static GLfloat piece_amb[] = {1.0,1.0,1.0,1.0};
    static GLfloat piece_spe[] = {1.0,1.0,1.0,1.0};
    static GLfloat piece_shi[] = {20.0};
    static GLfloat out_amb[] = {clearRed,clearGreen,clearBlue,1.0};
    static GLfloat out_spe[] = {0.0,0.0,0.0,1.0};
    static GLfloat out_shi[] = {0.0};
    
    glClear(GL_COLOR_BUFFER_BIT);
    glClear(GL_DEPTH_BUFFER_BIT);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslatef(0.0,0.0,-37.0);
    glRotatef(50.0,1.0,0.0,0.0);
    glRotatef(rotaH,1.0,0.0,0.0);
    glRotatef(rotaV,0.0,1.0,0.0);

    if (renderMode == GL_SELECT) glInitNames();	

    if (renderMode == GL_RENDER) {
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, out_amb);
	glMaterialfv(GL_FRONT, GL_SPECULAR, out_spe);
	glMaterialfv(GL_FRONT, GL_SHININESS, out_shi);
	glBegin(GL_QUADS);
	glVertex3f(-8*CASE_SIZE,    0,-8*CASE_SIZE);
	glVertex3f(-8*CASE_SIZE,    0, 8*CASE_SIZE);
	glVertex3f(-8*CASE_SIZE-10, 0, 8*CASE_SIZE+10);
	glVertex3f(-8*CASE_SIZE-10, 0,-8*CASE_SIZE-10);
	glVertex3f( 8*CASE_SIZE,    0,-8*CASE_SIZE);
	glVertex3f( 8*CASE_SIZE,    0, 8*CASE_SIZE);
	glVertex3f( 8*CASE_SIZE+10, 0, 8*CASE_SIZE+10);
	glVertex3f( 8*CASE_SIZE+10, 0,-8*CASE_SIZE-10);
	glVertex3f(-8*CASE_SIZE,    0,-8*CASE_SIZE);
	glVertex3f( 8*CASE_SIZE,    0,-8*CASE_SIZE);
	glVertex3f( 8*CASE_SIZE+10, 0,-8*CASE_SIZE-10);
	glVertex3f(-8*CASE_SIZE-10, 0,-8*CASE_SIZE-10);
	glVertex3f(-8*CASE_SIZE,    0, 8*CASE_SIZE);
	glVertex3f( 8*CASE_SIZE,    0, 8*CASE_SIZE);
	glVertex3f( 8*CASE_SIZE+10, 0, 8*CASE_SIZE+10);
	glVertex3f(-8*CASE_SIZE-10, 0, 8*CASE_SIZE+10);
	glEnd();

	if (reflection == true) {
	    if (pieceTexture) {
		glEnable(GL_TEXTURE_2D);
		glEnable(GL_TEXTURE_GEN_S);
		glEnable(GL_TEXTURE_GEN_T);
	    }	    
	    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, piece_amb);
	    glMaterialfv(GL_FRONT, GL_SPECULAR, piece_spe);
	    glMaterialfv(GL_FRONT, GL_SHININESS, piece_shi);
	    for(J=0;J<8;J++){
		c = !c;
		for(I=0;I<8;I++)
		{
		    let = 7-let;
		    num = 7-num;
		    if( userColor == BLACK ) num =  7-I;
		    else num = I;
		    if( userColor == WHITE ) let = 7-J;
		    else let = J;
		    glTranslatef((I-3.49)*CASE_SIZE*2,0,(J-3.49)*CASE_SIZE*2);

			piece = board[let][num];
		    glRotatef(180,1,0,0);
		    if(piece != 0)
		    {
			switch(piece)
			{
			    case -6 : blackText.use();king.draw(); break;
			    case -5 : blackText.use();queen.draw(); break;
			    case -4 : blackText.use();rook.draw(); break;
			    case -3 : blackText.use();bishop.draw(); break;
			    case -2 : blackText.use();knight.draw(); break;
			    case -1 : blackText.use();pawn.draw(); break;
			    case  6 : whiteText.use();king.draw(); break;
			    case  5 : whiteText.use();queen.draw(); break;
			    case  4 : whiteText.use();rook.draw(); break;
			    case  3 : whiteText.use();bishop.draw(); break;
			    case  2 : whiteText.use();knight.draw(); break;
			    case  1 : whiteText.use();pawn.draw(); break;
			}
		    }
		    glRotatef(180,1,0,0);
		    glTranslatef((3.49-I)*CASE_SIZE*2,0,(3.49-J)*CASE_SIZE*2);
		    c = !c;
		}
	    }
	    if (pieceTexture) {
		glDisable(GL_TEXTURE_GEN_S);
		glDisable(GL_TEXTURE_GEN_T);
		glDisable(GL_TEXTURE_2D);
	    }
	}
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, board_amb);
	glMaterialfv(GL_FRONT, GL_SPECULAR, board_spe);
	glMaterialfv(GL_FRONT, GL_SHININESS, board_shi);
	if (boardTexture) glEnable(GL_TEXTURE_2D);
	if (reflection) glEnable(GL_BLEND);
    }
    for(J=0;J<8;J++){
	c = !c;
	for(I=0;I<8;I++)
	{
	    if( userColor == BLACK ) num =  7-I;
	    else num = I;
	    if( userColor == WHITE ) let = 7-J;
	    else let = J;
	    
	    glTranslatef((I-3.49)*CASE_SIZE*2,0,(J-3.49)*CASE_SIZE*2);	
	    if (renderMode == GL_RENDER) {
		if ((num+1 == PS_GetX(sdlMove.start))&&(let+1 == PS_GetY(sdlMove.start))) {	
		    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, board_sel);
		}
		if ((num+1 == PS_GetX(sdlMove.stop))&&(let+1 == PS_GetY(sdlMove.stop))) {	
		    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, board_sel);
		}
		if(c == WHITE) whiteBoardText.use();
		if(c == BLACK) blackBoardText.use();
	    }
	    if (renderMode == GL_SELECT) glPushName(I*8+J);
	    glBegin(GL_QUADS);
	    glVertex3f(CASE_SIZE,0,CASE_SIZE);
	    glTexCoord2f(1.0,1.0);
	    glVertex3f(CASE_SIZE,0,-CASE_SIZE);
	    glTexCoord2f(1.0,0.0);
	    glVertex3f(-CASE_SIZE,0,-CASE_SIZE);
	    glTexCoord2f(0.0,0.0);
	    glVertex3f(-CASE_SIZE,0,CASE_SIZE);
	    glTexCoord2f(0.0,1.0);
	    glEnd();
	    glTranslatef((3.49-I)*CASE_SIZE*2,0,(3.49-J)*CASE_SIZE*2);
	    c = !c;
	    if (renderMode == GL_RENDER) {
		if ((num+1 == PS_GetX(sdlMove.start))&&(let+1 == PS_GetY(sdlMove.start))) {	
		    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, board_amb);
		}
		if ((num+1 == PS_GetX(sdlMove.stop))&&(let+1 == PS_GetY(sdlMove.stop))) {	
		    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, board_amb);
		}
	    }
	}
    }
    if (renderMode == GL_RENDER) {
	if (reflection) glDisable(GL_BLEND);	
	if (boardTexture) glDisable(GL_TEXTURE_2D);
	if (pieceTexture) {
	    glEnable(GL_TEXTURE_2D);
	    glEnable(GL_TEXTURE_GEN_S);
	    glEnable(GL_TEXTURE_GEN_T);
	}
	glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, piece_amb);
	glMaterialfv(GL_FRONT, GL_SPECULAR, piece_spe);
	glMaterialfv(GL_FRONT, GL_SHININESS, piece_shi);
	for(J=0;J<8;J++){
	    c = !c;
	    for(I=0;I<8;I++)
	    {
		if( userColor == BLACK ) num =  7-I;
		else num = I;
		if( userColor == WHITE ) let = 7-J;
		else let = J;
		glTranslatef((I-3.49)*CASE_SIZE*2,0,(J-3.49)*CASE_SIZE*2);
		piece = board[let][num];
		if(piece != 0)
		{
		    switch(piece)
		    {
			case -6 : blackText.use();king.draw(); break;
			case -5 : blackText.use();queen.draw(); break;
			case -4 : blackText.use();rook.draw(); break;
			case -3 : blackText.use();bishop.draw(); break;
			case -2 : blackText.use();knight.draw(); break;
			case -1 : blackText.use();pawn.draw(); break;
			    
			case  6 : whiteText.use();king.draw(); break;
			case  5 : whiteText.use();queen.draw(); break;
			case  4 : whiteText.use();rook.draw(); break;
			case  3 : whiteText.use();bishop.draw(); break;
			case  2 : whiteText.use();knight.draw(); break;
			case  1 : whiteText.use();pawn.draw(); break;
		    }
		}
		glTranslatef((3.49-I)*CASE_SIZE*2,0,(3.49-J)*CASE_SIZE*2);
		c = !c;
	    }
	}
	if (pieceTexture) {
	    glDisable(GL_TEXTURE_GEN_S);
	    glDisable(GL_TEXTURE_GEN_T);
	    glDisable(GL_TEXTURE_2D);
	}	
	if (buffer->data[0][5]==0){
	    glEnable(GL_TEXTURE_2D);
	    blackWin.use();
	    dialog.draw();
	    while(true) 
		while(SDL_PollEvent(&event))
		{
		    window.redraw();
		    switch(event.type)
		    {
			case SDL_QUIT :
			    end();
			    break;
			case SDL_KEYUP :
			    end();
			    break;
			case SDL_MOUSEBUTTONDOWN :
			    end();
			    break;
		    }
		}
	    glDisable(GL_TEXTURE_2D);
	}
	if (buffer->data[1][5]==0){
	    glEnable(GL_TEXTURE_2D);
	    whiteWin.use();
	    dialog.draw();
	    while(true) 
		while(SDL_PollEvent(&event))
		{
		    window.redraw();
		    switch(event.type)
		    {
			case SDL_QUIT :
			    end();
			    break;
			case SDL_KEYUP :
			    end();
			    break;
			case SDL_MOUSEBUTTONDOWN :
			    end();
			break;
		    }
		}
	    glDisable(GL_TEXTURE_2D);
	}
	window.redraw();
    }
    return *this;
}

const interface_t& interface_t::operator<< (char *buffer)
{
  printf("%s",buffer);
  return *this;
}


const interface_t& interface_t::operator<< (int buffer)
{
  printf("%i",buffer);
  return *this;
}

const interface_t& interface_t::operator<< (double buffer)
{
  printf("%f",buffer);
  return *this;
}

const interface_t& interface_t::operator<< (float buffer)
{
  printf("%f",buffer);
  return *this;
}

const interface_t& interface_t::operator<< (long int buffer)
{
  printf("%i",(int)buffer);
  return *this;
}

const interface_t& interface_t::operator<< (move_t buffer)
{
  printf("%c%i",(char)(PS_GetX(buffer.start)+96),(int)PS_GetY(buffer.start));
  printf("%c%i",(char)(PS_GetX(buffer.stop)+96),(int)PS_GetY(buffer.stop));
  return *this;
}

move_t interface_t::getmove(board_t *board,bool color)
{
  SDL_Event event;
  double mouse_x, mouse_y;
  int nbresult;
  move_t result[MAX_NB_RESULT];
  GLint hits;
  GLint viewport[4];
  GLuint selectBuf[SELECTBUFSIZE];
  GLuint select;
  int tmp_x,tmp_y;
  
  for(;;) {
      while(SDL_PollEvent(&event))
      {
	  switch(event.type)
	  {
	      case SDL_QUIT :
		  end();
		  break;   
	      case SDL_KEYUP :
		  if(event.key.keysym.sym == SDLK_ESCAPE)
		  {
		      end();
		  }
		  break;
	      case SDL_MOUSEBUTTONDOWN :
		  mouse_x = event.button.x;
		  mouse_y = event.button.y;
	    
		  if ((sdlMove.start != NOPOS)&&(sdlMove.stop != NOPOS)) {
		      sdlMove.start = NOPOS;
		      sdlMove.stop = NOPOS;
		  }
		  glGetIntegerv(GL_VIEWPORT,viewport);
		  glSelectBuffer(SELECTBUFSIZE,selectBuf);
		  glRenderMode(GL_SELECT);
		  renderMode = GL_SELECT;
		  glMatrixMode(GL_PROJECTION);
		  glPushMatrix();
		  glLoadIdentity();
		  gluPickMatrix(mouse_x,viewport[3]-mouse_y,1.0,1.0,viewport);
		  gluPerspective(60.0,1.3,1.0,1024.0);
		  *this << board;
		  glMatrixMode(GL_PROJECTION);
		  glPopMatrix();
		  hits = glRenderMode(GL_RENDER);
		  renderMode = GL_RENDER;
		  select = NOPOS;
		  if (hits>0) {
		      select = selectBuf[hits-1];
		      select = selectBuf[0];
		      select = select - 1;
		      tmp_x = PS_GetX(select);
		      tmp_y = PS_GetY(select);
		      if (userColor == WHITE) {
			  tmp_y = 8 - tmp_y; 
			  tmp_x = tmp_x - 1;
		      } else {
			  tmp_x = 8 - tmp_x;
			  tmp_y = tmp_y - 1;
		      }
		      select = (tmp_y)*8+(tmp_x);
		  }
		  
		  if( sdlMove.start != NOPOS )
		  {
		      sdlMove.stop = select;
		      if( sdlMove.start == sdlMove.stop ) sdlMove.start = NOPOS;
		      if ((sdlMove.start == 4)&&(sdlMove.stop == 6)){
			  sdlMove.start = WSROCK;
			  sdlMove.stop  = WSROCK;
		      }
		      if ((sdlMove.start == 4)&&(sdlMove.stop == 2)){
			  sdlMove.start = WBROCK;
			  sdlMove.stop  = WBROCK;
		      }
		      if ((sdlMove.start == 60)&&(sdlMove.stop == 62)){
			  sdlMove.start = BSROCK;
			  sdlMove.stop  = BSROCK;
		      }
		      if ((sdlMove.start == 60)&&(sdlMove.stop == 58)){
			  sdlMove.start = BBROCK;
			  sdlMove.stop  = BBROCK;
		      }
		      
		      nbresult = GN_Generate(board,color,result);
		      for(int I=0;I<nbresult;I++){
			  if ((result[I].start==sdlMove.start)&&(result[I].stop==sdlMove.stop)) return sdlMove;
		      }
		      sdlMove.start = NOPOS;
		      sdlMove.stop = NOPOS;
		  }
		  
		  if( sdlMove.start == NOPOS )
		  {
		      sdlMove.start = select;
		  }
	  }
      }
      Uint8 *keystate = SDL_GetKeyState(NULL);	   
      if ((keystate[SDLK_RIGHT])&&(rotaV<180)) rotaV+=DECAL;
      if ((keystate[SDLK_LEFT])&&(rotaV>-180)) rotaV-=DECAL;
      if ((keystate[SDLK_DOWN])&&(rotaH<60)) rotaH+=DECAL;
      if ((keystate[SDLK_UP])&&(rotaH>-30)) rotaH-=DECAL;
      *this << board;
  }
}

void interface_t::clearprogress()
{
  isDialog = false;
  *this << board;
  window.redraw();
}

void interface_t::end()
{
  window.close();
  SDL_Quit();
  exit(0);
}

void interface_t::AddToGame(move_t move)
{
  int I;
  move_t *buffer = new move_t[GameNbMove+1];
  for(I=0;I<GameNbMove;I++) buffer[I] = game[I];
  buffer[GameNbMove] = move;
  GameNbMove++;
  game = new move_t[GameNbMove];
  for(I=0;I<GameNbMove;I++) game[I] = buffer[I];
  delete buffer;
}

// take all the event and process them
void interface_t::updateEvent()
{
  SDL_Event event;
  while(SDL_PollEvent(&event)) {
    switch(event.type)
      {
      case SDL_QUIT :
	end();
      case SDL_KEYDOWN :
	if(event.key.keysym.sym == SDLK_ESCAPE)
	  {
	    end();
	  }
	break;
      }
  }
  Uint8 *keystate = SDL_GetKeyState(NULL);
  if ((keystate[SDLK_RIGHT])&&(rotaV<180)) rotaV+=DECAL;
  if ((keystate[SDLK_LEFT])&&(rotaV>-180)) rotaV-=DECAL;
  if ((keystate[SDLK_DOWN])&&(rotaH<60)) rotaH+=DECAL;
  if ((keystate[SDLK_UP])&&(rotaH>-30)) rotaH-=DECAL;	   
  *this << board;
}
