/*
 *  CINAG - Chess Is Not A Game
 * 
 *  Copyright (c) 2001, 2002 PAYEMENT Arnaud
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <time.h>
#include <stdlib.h>
#include <iostream.h>
#include <stdio.h>
#include <string.h>

#include "search.h"
#include "analyse.h"
#include "genmove.h"
#include "analyse.h"
#include "board.h"
#include "ini_file.h"
#include "db.h"
#include "transposition.h"
#include "killer.h"
#include "debug.h"

#define MIN(a, b)  (((a) < (b)) ? (a) : (b))
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))

// configuration
bool quiescent;
char quiet_depth;
char fixed_depth;
int  threshold;
int  max_threshold;

//#define DEBUG

typedef struct {
    slint_t result;
    char depth;
} result_t;

int iabs(int I)
{
    if (I>0) return I;
    else return -I;
}

void ident_depth(int depth)
{
    for(int I=0;I<depth*4;I++) printf(" ");
}

void print(move_t move)
{
    printf("%c%i",(char)(PS_GetX(move.start)+96),(int)PS_GetY(move.start));
    printf("%c%i",(char)(PS_GetX(move.stop)+96),(int)PS_GetY(move.stop));
}

void SE_Sort(result_t *num,move_t *move,char nb,bool type)
{
    result_t Iswap;
    move_t Mswap;
    for(char i=0;i<nb;i++)
	for(char index=0;index<nb-1;index++)
	{
	    if (((num[index+1].result>num[index].result)&&(type==WHITE))||((num[index+1].result == num[index].result)&&(num[index+1].depth>num[index].depth))){
		Iswap = num[index];
		num[index] = num[index+1];
		num[index+1] = Iswap;
		Mswap = move[index];
		move[index] = move[index+1];
		move[index+1] = Mswap;
	    }
	    if (((num[index+1].result<num[index].result)&&(type==BLACK))||((num[index+1].result == num[index].result)&&(num[index+1].depth>num[index].depth))){
		Iswap = num[index];
		num[index] = num[index+1];
		num[index+1] = Iswap;
		Mswap = move[index];
		move[index] = move[index+1];
		move[index+1] = Mswap;
	    }
	}
}

void SE_SortInt(int *num,move_t *move,char nb,bool type)
{
    int Iswap;
    move_t Mswap;
    for(char i=0;i<nb;i++)
	for(char index=0;index<nb-1;index++)
	{
	    if ((num[index+1]>num[index])&&(type==WHITE)){
		Iswap = num[index];
		num[index] = num[index+1];
		num[index+1] = Iswap;
		Mswap = move[index];
		move[index] = move[index+1];
		move[index+1] = Mswap;
	    }
	    if ((num[index+1]<num[index])&&(type==BLACK)){
		Iswap = num[index];
		num[index] = num[index+1];
		num[index+1] = Iswap;
		Mswap = move[index];
		move[index] = move[index+1];
		move[index+1] = Mswap;
	    }
	}
}

void SE_KillerModify(int nbmove,move_t *move,int depth)
{
    int I,J;
    move_t killer_move;

    if (depth <= -quiet_depth) return;

    for(I=4;I>=0;I--) {
	killer_move = KL_GetKiller(depth,I);
	for(J=0;J<nbmove;J++)
	{
	    if ((move[J].start == killer_move.start)&&(move[J].stop == killer_move.stop))
	    {
		    move[J].start = move[I].start;
		    move[J].stop = move[I].stop;
		    move[I].start = killer_move.start;
		    move[I].stop = killer_move.stop;
	    }
	}
    }
}

result_t SE_AlphaBeta(board_t *board,bool color,char depth,int alpha,int beta,int old_note)
{
  board_t *board_tmp;
  move_t move[MAX_NB_RESULT];
  move_t best_move;
  int alpha_up,beta_up;     
  int nbmove;
  int I;
  transpo_t trans;
  hashcode_t code1 = HC_CreateHashCode(board,hashcode1);
  hashcode_t code2 = HC_CreateHashCode(board,hashcode2);
  int note = AN_Analyse(board);
  result_t best_result,up_result;

  if (iabs(note) == MATE_VALUE) {
      best_result.result = note;
      best_result.depth = depth;
      return best_result;
  }

// transposition 
  trans = TP_FindResult(code1,color);
  if ((trans.depth > -127)&&(trans.depth >= depth)&&(trans.code2 == code2))
  {
      if (((trans.type == hash_beta)&&(trans.value >= beta))||((trans.type == hash_alpha)&&(trans.value <= alpha))||(trans.type == hash_exact)) 
      {
	  best_result.result = trans.value;
	  best_result.depth = trans.depth;
	  return best_result;
      }
      if (trans.type == hash_alpha) {
	  alpha = MAX(alpha,trans.value);
      }
      if (trans.type == hash_beta) {
	  beta = MIN(beta,trans.value);
      }
  }

// checking depth 
  if ((depth <= 0)&&(((quiescent==false)||(iabs(note-old_note)<threshold)||((fixed_depth-depth)>quiet_depth))||(iabs(note-old_note)<max_threshold))) {
      best_result.result = AN_Analyse(board);
      best_result.depth = depth;
// hard work now !!
  } else if (color == WHITE) {
      best_result.result = -MATE_VALUE;
      best_result.depth = -127;
      alpha_up = alpha;
      beta_up = beta;
      nbmove = GN_Generate(board,color,move);
      SE_KillerModify(nbmove,move,depth);
      for(I=0;I<nbmove;I++)
      {
	  board_tmp = BO_Copy(board);
	  BO_Move(board_tmp,move[I]);
#ifdef DEBUG
	  ident_depth(depth);
	  print(move[I]);
	  printf("\n");
	  ident_depth(depth);
	  printf("SE_AlphaBeta(board_tmp,%i,%i,%i,%i,%i)\n",!color,depth-1,alpha_up,beta_up,note);
#endif
	  up_result = SE_AlphaBeta(board_tmp,!color,depth-1,alpha_up,beta_up,note);
#ifdef DEBUG
	  ident_depth(depth);
	  printf("up_result = %i;\n",(int)(up_result.result));
#endif
	  delete board_tmp;
	  if (up_result.result > best_result.result)
	  {
	      best_result.result = up_result.result;
	      best_result.depth = up_result.depth;
	      best_move = move[I];
	  }
	  alpha_up = MAX(alpha_up,best_result.result);
	  if (best_result.result >= beta) break;
      }    
      KL_SetKiller(best_move,depth);
  } else {
      best_result.result = MATE_VALUE;
      best_result.depth = -127;
      alpha_up = alpha;
      beta_up = beta;
      nbmove = GN_Generate(board,color,move);
      SE_KillerModify(nbmove,move,depth);
      for(I=0;I<nbmove;I++)
      {
	  board_tmp = BO_Copy(board);
	  BO_Move(board_tmp,move[I]);
#ifdef DEBUG
	  ident_depth(depth);
	  print(move[I]);
	  printf("\n");
	  ident_depth(depth);
	  printf("SE_AlphaBeta(board_tmp,%i,%i,%i,%i,%i)\n",!color,depth-1,alpha_up,beta_up,note);
#endif
	  up_result = SE_AlphaBeta(board_tmp,!color,depth-1,alpha_up,beta_up,note);
#ifdef DEBUG
	  ident_depth(depth);
	  printf("up_result = %i;\n",(int)(up_result.result));
#endif
	  delete board_tmp;
	  if (up_result.result < best_result.result)
	  {
	      best_result.result = up_result.result;
	      best_result.depth = up_result.depth;
	      best_move = move[I];
	  }
	  beta_up = MIN(beta_up,best_result.result);
	  if (best_result.result <= alpha) break;
      }
      KL_SetKiller(best_move,depth);
  }

  if (depth >= 0) {
      trans.code2 = code2;
      if (best_result.result <= alpha) trans.type = hash_alpha;
      if (best_result.result >= beta) trans.type = hash_beta;
      if ((best_result.result > alpha)&&(best_result.result < beta)) trans.type = hash_exact;
      trans.depth = depth;
      trans.value = best_result.result;
      TP_AddResult(code1,trans,color);
  }
  return best_result;
}


move_t SE_MinMax(board_t *board,bool color,int depth)
{
    move_t move[MAX_NB_RESULT];
    result_t result[MAX_NB_RESULT];
    board_t *board_tmp;
    int I;
    move_t error;
    error.start = NOMOVE;
    error.stop = NOMOVE;

    KL_Init(fixed_depth,quiet_depth);

    int nbmove = GN_Generate(board,color,move);
    int note = AN_Analyse(board);

    for(I=0;I<MAX_NB_RESULT;I++)
    { 
	result[I].result = 0;
	result[I].depth = -127;
    }
    for(I=0;I<nbmove;I++)
    {	
	board_tmp = BO_Copy(board);
	BO_Move(board_tmp,move[I]);
	result[I] = SE_AlphaBeta(board_tmp,!color,depth,-MATE_VALUE,MATE_VALUE,note);
	delete board_tmp;
    }
    SE_Sort(result,move,nbmove,color);

    KL_Free();

    for(I=0;I<nbmove;I++) {
	if (move[I].start == NOMOVE) continue;
	return move[I];
    }
    return error;
}


result_t SE_MTD(board_t *board,bool color,int depth,int guess)
{
    int beta;
    result_t result;
    result.result = guess;
    result.depth = -127;
    int upperbound = MATE_VALUE;
    int lowerbound = -MATE_VALUE;
    int I=0;
    int note = AN_Analyse(board);
    for(;lowerbound<upperbound;)
    {
	if (result.result == lowerbound) beta = result.result+1;
	else beta = result.result;
	result = SE_AlphaBeta(board,color,depth,beta-1,beta,note);
	if (result.result<beta) upperbound = result.result;
	else lowerbound = result.result;
	I++;
    }
#ifdef DEBUG
    printf("number of test : %i\n",I);
#endif
    return result;
}


move_t SE_IterativeDeepening(board_t *board,bool color,int depth)
{
    move_t move[MAX_NB_RESULT];
    result_t result[MAX_NB_RESULT];
    result_t old[MAX_NB_RESULT];
    board_t *board_tmp;
    int I,J,K;
    move_t error;
    int prev;
    error.start = NOMOVE;
    error.stop = NOMOVE;

    KL_Init(fixed_depth,quiet_depth);

    int nbmove = GN_Generate(board,color,move);
    for(I=0;I<MAX_NB_RESULT;I++)
    {
	result[I].result = 0;
	result[I].depth = -127;
	old[I].result = 0;
	old[I].depth = -127;
    }
    printf("\n");
    for(J=0;J<=depth;J++) {
	printf("depth %i\n",J);
	for(I=0;I<(nbmove);I++)
	{	
	    board_tmp = BO_Copy(board);
	    BO_Move(board_tmp,move[I]);
#ifdef DEBUG
	    printf("(b) ");
	    print(move[I]);
	    printf("\n");
#endif
	    prev = 0;
	    for(K=0;K<I;K++) prev += result[K].result-old[K].result;
	    if (I>0) prev = prev/I;      
#ifdef DEBUG
	    printf(" prev : %i ",(int)(prev+old[I].result));
#endif
	    result[I] = SE_MTD(board_tmp,!color,J,prev+old[I].result);
#ifdef DEBUG
	    printf(" result : %i\n",(int)(result[I].result));
#else
	    printf(".");
#endif
	    fflush(stdout);
	    delete board_tmp;
	}
	SE_Sort(result,move,nbmove,color);
	printf("\n");
	print(move[0]);
	printf("\n");
	for(K=0;K<nbmove;K++)
	{
	    old[K].result = result[K].result;
	    old[K].depth = result[K].depth;
	}
    }

    KL_Free();

    for(I=0;I<nbmove;I++) {
	if (move[I].start == NOMOVE) continue;
	return move[I];
    }
    return error;
}


move_t SE_DB(board_t *board,bool color)
{
    int I;
    int64 J;
    int64 nb_db;
    move_t move[MAX_NB_RESULT];
    int result[MAX_NB_RESULT];
    move_t db_move;
    move_t error;
    error.start = NOMOVE;
    error.stop = NOMOVE;
    
    int nbmove = GN_Generate(board,color,move);
    for(I=0;I<MAX_NB_RESULT;I++) result[I] = 0;

    if (color == WHITE) {
	nb_db = DB_FindMove(board,WHITE);
	for(I=0;I<nbmove;I++) {
	    for(J=0;J<nb_db;J++) {
		db_move=DB_GetMove(J);
		if ((move[I].start == db_move.start)&&(move[I].stop == db_move.stop)){
		    result[I] += 1;
		}
	    }
	}
    }
    if (color == BLACK) {
	nb_db = DB_FindMove(board,BLACK);
	for(I=0;I<nbmove;I++) {
	    for(J=0;J<nb_db;J++) {
		db_move=DB_GetMove(J);
		if ((move[I].start == db_move.start)&&(move[I].stop == db_move.stop)){
		    result[I] -= 1;
		}
	    }
	}
    }
  SE_SortInt(result,move,nbmove,color);
  for(I=0;I<nbmove;I++) {
      if (move[I].start == NOMOVE) continue;
      return move[I];
  }
  warning(database ./chess/chess.db have strange value !!!);
  return error;
}


move_t SE_Search(board_t *board,bool color)
{
    ini_file_t conf;
    int64 nb_db;
    move_t move;
    char *buff;

    move.start = NOMOVE;
    move.stop = NOMOVE;

    conf.read("conf.ini");
    buff = conf.getcharvalue("QUIESCENT","enable");
    if (!strcmp("true",buff)) quiescent = true;
    else quiescent = false;
    threshold = conf.getintvalue("QUIESCENT","threshold");
    max_threshold = conf.getintvalue("QUIESCENT","max_threshold");
    quiet_depth = conf.getintvalue("QUIESCENT","quiet_depth");
    fixed_depth = conf.getintvalue("ALGO", "depth");
    buff = conf.getcharvalue("ALGO","algo");

    TP_Init();
    nb_db = DB_FindMove(board,color);
    if (nb_db < 1) {
	if (!strcmp("mtd(f)",buff)) move =  SE_IterativeDeepening(board,color,fixed_depth);
	else if (!strcmp("minmax",buff)) move = SE_MinMax(board,color,fixed_depth);
	else error(conf.ini : unknown algo !!!);
    } else {
	move = SE_DB(board,color);
    }
    TP_Free();
    return move;
}
