/*
 * SourceSafe/CVS conversion server
 * cserver.c
 * Implementation of the server functions 
 *
 * (c) Benjamin Loo - 2001
 *
 * 
 * You may distribute under the terms of the GNU General Public License as
 * specified in the README file that comes with the VSSExtractor/Cserver
 * source distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details. 
 *
 * Visual SourceSafe is a registered trademark from Microsoft Corporation.
 * 
 */

#include "cvs.h"
#include "cserver.h"
#include "file.h"
#include "users.h"
#include "servlog.h"
#include <fcntl.h>

/*
 * constants for the commands understood by the server
 */
static const char * commands[] = {
	"BEGIN",
	"BYE",
	"PING",
	"CONFIG",
	"PUT",
	"DIR",
	"FLABEL",
	"PLABEL",
	"DELETE",
	"RECOVER",
	"MOVE",
	"USERS"
};

/*
 * number of valid commands
 */	
static const int ncommand = 12;

/*
 *  secure send function 
 *
 *  returns 0 when send is successful, -1 when it fails
 */
int my_send(int sock, char * buffer, int len) {
	int count = 0;
	int i;

	while(count<len) {
		i = send(sock, &buffer[count], len-count, 0);
		if (i<=0) return -1;
		else count += i;
	}
	return 0;
}

/*
 * parse a line from the config file
 * format "[variable_name] variable_value\n"
 *
 * returns 0 if the line was correctly parsed, -1 otherwise
 * allocates *var & *content strings
 */
int parseConfigline(char * line, char ** var, char ** content) {
	int i;
	int begin;
	int end;
	int len;
	int len2;

	if (line == NULL || line[0] == '#') {
		return -1;
	}
	else {
		len = strlen(line);
		for(i = 0; i<len && line[i] != '['; i++);
		if (i == len) return -1;
		begin = i;
		for(i = begin + 1; i<len && line[i] != ']'; i++);
		if (i == len) return -1;
		end = i;
		len2 = end - begin;
		*var = (char *) malloc(sizeof(char) * len2);
		strncpy(*var, &line[begin+1], len2-1);
		(*var)[len2 - 1] = 0;

		for(i = end + 1; i<len && line[i] == ' '; i++);
		begin = i;
		for(i = len - 1; i>begin && (line[i] == '\n' || line[i] == ' '); i--);
		end = i;
		len2 = end - begin + 2;
		*content = (char*) malloc(sizeof(char) * len2);
		strncpy(*content, &line[begin], len2 - 1);
		(*content)[len2-1] = 0;
	}

	return 0;
}
			
/*
 * read the server parameters from the config file
 * (cserver.conf)
 */
void readConfig(char * file) {
	FILE * f;
	char * var;
	char * value;
	char buffer[512];

	/* setting default values */
	port = 4555;
	dataport = 4556;
	cvs_path = "/cvs";
	temp_dir = "/tmp";
	logfilename = "logfile";

	/* reading config file */
	if ((f = fopen(file, "r")) == NULL) {
		printf("Can't open config file !\n");
		return;
	}
	while (! feof(f) ) {
		fgets(buffer, 512, f);
		if (parseConfigline(buffer, &var, &value) == 0) {
			if (strcasecmp(var, "commandport") == 0) {
				port = atoi(value);
				free(value);
			}
			else if (strcasecmp(var, "dataport") == 0) {
				dataport = atoi(value);
				free(value);
			}
			else if (strcasecmp(var, "cvsroot") == 0) cvs_path = value;
			else if (strcasecmp(var, "tempdir") == 0) temp_dir = value;
			else if (strcasecmp(var, "logfile") == 0) logfilename = value;
			else if (value != NULL) free(value);

			if (var != NULL) free(var);
		}
	}

	fclose(f);
}
				
			

/* 
 * initialize the server (socket & global variables)
 */
void server_init() {
	struct sockaddr_in s;

	/* Creates control socket */
	s.sin_family = AF_INET;
	s.sin_port = htons( (unsigned short) port);
	s.sin_addr.s_addr = INADDR_ANY;

	if ((serversocket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		fprintf(stderr, "Unable to create socket\n");
		exit(-1);
	}
	if (bind(serversocket, (struct sockaddr *) &s, sizeof(struct sockaddr)) == -1) {
		fprintf(stderr, "Unable to bind socket\n");
		exit(-1);
	}
	if (listen(serversocket, 5) != 0) {
		fprintf(stderr, "Unable to listen to socket\n");
		exit(-1);
	}
	
	printf("Conversion server running on port %d...\n", port);
	
	/* Creates data socket */
	s.sin_port = htons( (unsigned short) dataport);
	s.sin_addr.s_addr = INADDR_ANY;

	if ((dataserversocket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		fprintf(stderr, "Unable to create data socket\n");
		exit(-1);
	}
	if (bind(dataserversocket, (struct sockaddr *) &s, sizeof(struct sockaddr)) == -1) {
		fprintf(stderr, "Unable to bind data socket\n");
		exit(-1);
	}
	if (listen(dataserversocket, 5) != 0) {
		fprintf(stderr, "Unable to listen to data socket\n");
		exit(-1);
	}

	printf("Data port is %d...\n", dataport);
	
	logMessage("Server running", "");
}

/*
 * main loop for the server, listening to the control socket and 
 * building the CVS base
 */
void server_run() {
	struct sockaddr_in s;
	int length;
	int cmd;

	current_dir = NULL;
	userlist = NULL;
	usermode = NULL;
	repository_files = NULL;
	lastline = NULL;	
	writemode = 0;
	allowbinaries = 0;

	startLogging(logfilename);
	
	printf("Ready.\n");
	length = sizeof(struct sockaddr);
	sock = accept(serversocket, (struct sockaddr*) &s, &length);

	if (sock == -1) {
		fprintf(stderr, "Can't receive connection\n");
		exit(-1);
	}

	logMessage("Connection", "Successfully received client connection");
	
	fcntl(sock, F_SETFL, O_NONBLOCK);
	
	cmd = CMD_INVALID;
	while (cmd != CMD_BEGIN) {
		cmd = getCommand();
		if (cmd == CMD_PING) reply("PONG\n");
		else if (cmd != CMD_BEGIN) {
			reply("ERROR Invalid Command\n");
			return;
		}
	}
	reply("OK\n");

	while (cmd != CMD_BYE) {
		cmd = getCommand();
		switch (cmd) {
		case CMD_PING: reply("PONG\n");
			break;
		case CMD_PUT: getFile();
			break;
		case CMD_BEGIN: reply("ERROR Can't redo begin\n");
			break;
		case CMD_CONFIG: getConfig();
			break;
		case CMD_DIR: getDir();
			break;
		case CMD_FLABEL: getFLabel();
			break;
		case CMD_PLABEL: getPLabel();
			break;
		case CMD_DELETE: getDelete();
			break;
		case CMD_RECOVER: getRecover();
			break;
		case CMD_MOVE: getMove();
			break;
		case CMD_USERS: getUsers();
			break;	
		case CMD_INVALID: reply("ERROR Invalid command\n");
			break;
		default:
		}
	}
	
	reply("OK\n");

	deleteList(repository_files);
	close(sock);

	logMessage("Disconnection", "Client successfully disconnected from server");
	endLogging();

	cleanTempdir();
}

/*
 * gets a line (up to 4096 characters, ended with a '\n') on the control socket
 *
 * returns a new string containing the line, or NULL if nothing was read
 */
char * getLine() {
	int nfds;
	int selected;
	fd_set readfds;
	char buffer[4096];
	int nbytes = 0;
	int i = 0;
	char * cmd;
	struct timeval timeout;
	
	while (i == nbytes) {
	
		nfds = sock+1;
		FD_ZERO(&readfds);
		FD_SET(sock, &readfds);
		timeout.tv_sec = 20;
		timeout.tv_usec = 0;

		if ((selected = select(nfds, &readfds, NULL, NULL, &timeout)) == -1) {
			fprintf(stderr, "Can't do select\n");
			exit(-1);
		}

		if (selected == 0) {
			return NULL;
		}
		
		if ((nbytes = recv(sock, buffer, 4096, MSG_PEEK)) <= 0) {
			fprintf(stderr, "Broken connection\n");
			exit(-1);
		}

		for(i = 0; i<nbytes && buffer[i] != '\n'; i++); 
	}

	cmd = (char *) malloc((i+1) * sizeof(char));
	if (recv(sock, cmd, i+1, 0) != i+1) {
		fprintf(stderr, "Transmission error\n");
		exit(-1);
	}
	cmd[i] = 0;
	if (cmd[i-1] == '\r') cmd[i-1] = 0;

	return cmd;
}


/*
 * Gets the next command from the control socket
 * 
 * Returns a command number as defined in cserver.h, as an int
 */
int getCommand() {
	char * cmd; 
	int ret;
	int i;

	if (lastline != NULL) cmd = lastline;
	else {
		cmd = NULL;
		while (cmd == NULL) cmd = getLine();
	}
	lastline = NULL;

	i = 0;
	while (i<ncommand && strcmp(cmd, commands[i]) != 0) i++;
	if (i == ncommand) ret = CMD_INVALID; else ret = i;

	free(cmd);
	return ret;
}

/*
 * Sends a reply message to the client, via the control socket
 */
void reply(char * msg) {
	fd_set writefds;
	struct timeval timeout;
	int n;
	int s;
	int len;

	FD_ZERO(&writefds);
	FD_SET(sock, &writefds);
	timeout.tv_sec = 10;
	timeout.tv_usec = 0;

	if ((n = select(sock+1, NULL, &writefds, NULL, &timeout)) == -1) {
		fprintf(stderr, "Can't do select\n");
		exit(-1);
	}

	if (n == 0) {
		fprintf(stderr, "Lost connection\n");
		exit(-1);
	}

	len = strlen(msg);
	s = my_send(sock, msg, len);
	if (s != 0) {
		fprintf(stderr, "Client/server communication error\n");
		exit(-1);
	}

}

/*
 * Sets a temporary file for storing received data
 */
void initTempfile(struct FileHeader * h) {
	int handle;
		
	clearHeader(h);
	h->tempfile = (char *) malloc(sizeof(char) * (strlen(temp_dir) + 12));
	strcpy(h->tempfile, temp_dir);
	strcat(h->tempfile, "/CvSXXXXXX");
	handle = mkstemp(h->tempfile);
	close(handle);

}
		
/*
 * Receives a complete file (a file header and file contents) from the client
 */
void getFile() {
	struct FileHeader h;
	char * line;
	int transmit;
	struct FileList * f;
	RCSNode * RCShandle;
	int newfile = 0;
	char logbuffer[256];
	time_t thischange;
	char * version;
	
	/* initializing */	
	initTempfile(&h);

	/* getting file header */
	line = getLine();
	
	if (line == NULL || strcmp(line, "HEADER") != 0) {
		lastline = line;
		reply("ERROR invalid PUT content\n");
		return;
	}
	free(line);
	printf("Receiving file header...\n");

	if (getHeader(&h) == -1) {
		reply("ERROR Can't get header\n");
		return;
	}
	reply("OK\n");

	/*receiving file content */
	transmit = getFilecontent(h);

	/* If transmission OK, updates the CVS repository */
	if (transmit == 0)  {
		/* is it a new file? */
		if (findinList(h.filename, repository_files) == NULL) newfile = 1;
		f = checkaddList(h.filename, h.timestamp, h.type);
		if (newfile == 1) setRCSEntry(f, h);

		/* check if the timestamp is valid */
		thischange = makeGMTime(h.timestamp);
	
		if (f->lasttime < thischange) {
			/* the commit is valid, proceed */
		
			/* increases the RCS version number */
			version = updateVersion(f->RCSversion);
			free(f->RCSversion);
			f->RCSversion = version;
				
			/* sets the timestamp for the file */
			setFiletime(h.tempfile, h.timestamp);
	
			/* adds the new version to the repository */
			RCShandle = RCS_parsercsfile(f->RCSfile);
			RCS_mycheckin(RCShandle, h.tempfile, h.comment, f->RCSversion, h.author, (RCS_FLAGS_MODTIME | RCS_FLAGS_QUIET));
			freercsnode(&RCShandle);

			/* remembers the last timestamp */
			f->lasttime = thischange;
			
			/* logs the file transfer */
			snprintf(logbuffer, 256, "Received file %s, RCS version %s", h.filename, f->RCSversion);
			logMessage("Received file", logbuffer);
		}
		else {
			fprintf(stderr, "This out-of-date version of file %s was rejected by the server\n\n", h.filename);
			snprintf(logbuffer, 256, "Rejected out-of-date version of file %s, stamped %s", h.filename, h.timestamp);
			logMessage("Rejected file", logbuffer);
}
	}
	else {
		snprintf(logbuffer, 256, "Could not get file %s, RCS version %s : network error", h.filename, f->RCSversion);
		logMessage("Transfer error", logbuffer);
	}
	
}
	
	
/*
 * Opens a data socket connection with the client and receives file contents
 *
 * returns 0 when reception is successful, -1 when an error occurs
 */
int getFilecontent(struct FileHeader h) {
	struct sockaddr_in s;
	int length = sizeof(struct sockaddr);
	char * line;
	char buffer[RECVBUFSIZE + 1];
	fd_set readfds;
	struct timeval timeout;
	int n;
	int nb;
	int count;
	FILE * f;
	int fd;

	/* opening data socket and local temporary file */
	printf("Establishing data connection...");
	datasock = accept(dataserversocket, (struct sockaddr *) &s, &length);
	if (datasock == -1) {
		fprintf(stderr, "Can't receive connection\n");
		exit(-1);
	}
	printf("done\n");
	
	if ((f = fopen(h.tempfile, "w")) == NULL) {
		fprintf(stderr, "Can't write to file\n");
		exit(-1);
	}
	fd = fileno(f);

	line = getLine();
	if (line == NULL || strcmp(line, "CONTENT") != 0) {
		lastline = line;
		reply("ERROR invalid PUT content\n");
		return -1;
	}
	free(line);
	printf("Receiving file content...\n");

	/* Receiving file content */
	count = 0;
	while (1) {
		if (h.filesize == 0) break;
		FD_ZERO(&readfds);
		FD_SET(datasock, &readfds);
		timeout.tv_sec = 20;
		timeout.tv_usec = 0;
		
		if ((n = select(datasock+1, &readfds, NULL, NULL, &timeout)) == -1) {
			printf("Select error\n");
			reply("ERROR Can't receive file\n");
			break;
		}   

		if (n == 0) {
			printf("Timeout\n");
			reply("ERROR Request timed out\n");
			break;
		}
		
		nb = read(datasock, buffer, RECVBUFSIZE);
		if (nb > 0) {
			if (count + nb > h.filesize) nb = h.filesize - count;	
			count += nb;
			write(fd, buffer, nb);
			if (count == h.filesize) break;
		}
		else {
			printf("Nothing to read!\n");
			reply("ERROR Unexpected end of transmission\n");
			break;
		}
			
	}

	fclose(f);
	close(datasock);
	if (count == h.filesize) {
		/* Transfer was OK, displays information */
		reply("OK\n");
		printf("File transfer complete.\nFile header is :\n");
		printHeader(h);
		printf("Successfully received %d bytes\n\n", count);
		return 0;
	}
	else {
		/* failed to receive file */
		printf("Transfer failed\n");
		return -1;
	}

	return 0;


}

/* 
 * Receives a file header in pseudo-XML format, on the control socket
 * Header starts with line "<Header>\n" and ends with the line "</Header>\n"
 * Header items format is : "<Tag> content </Tag>\n" WITHOUT extra spacing or
 * linefeeds - that's why it's only "pseudo"-XML 
 *
 * returns 0 if header was successfully received, -1 otherwise
 */
int getHeader(struct FileHeader * h) {
	char * line;
	char * file;
	
	line = getLine();
	while (line != NULL && strstr(line, "<Header>") == NULL) {
		free(line);
		line = getLine();
	}

	if (line == NULL) return -1;
	
	free(line);
	line = getLine();
	
	while (line != NULL && strstr(line, "</Header>") == NULL) {
		getHeaderElement(line, h);
		free(line);
		line = getLine();
	}

	if (line == NULL) return -1;

	free(line);

	file = trimFilename(h->filename);
	free(h->filename);
	h->filename = file;
	
	return 0;
}

/*
 * Given a path like /dir/subdir1/subdir2, finds the n-th depth level in this
 * path, and moves to the corresponding directory - if the directory does not
 * exist, creates it.
 * 
 * Return 0 if the directory existed, nonzero if it created it.
 */
int mknsubdir(char * path, int subdir) {
	int count;
	int end;
	int i;
	int len = strlen(path);
	char * mypath;
	int ret = 0;

	count = 0;
	for(i = 0; i<len && count<subdir; i++) {
		if (path[i] == '/') {
			count++;
			end = i;
		}
	}
	
	if (count<subdir) {
		mypath = (char*) malloc(strlen(path) + strlen(cvs_path) + 1);
		strcpy(mypath, cvs_path);
		strcat(mypath, path);
		if (chdir(mypath) == -1) {
			mkdir(mypath, 0777);
			chdir(mypath);
			ret = 1;
		}
		free(mypath);
	}
	else {
		mypath = (char*) malloc(strlen(path) + strlen(cvs_path) + 1);
		strcpy(mypath, cvs_path);
		strncat(mypath, path, i);
		mypath[strlen(cvs_path) + i + 1] = 0;
		if (chdir(mypath) == -1) {
			mkdir(mypath, 0777);
			chdir(mypath);
			ret = 1;
		}
		ret += mknsubdir(path, subdir+1);
	}
	return ret;
}

/*
 * Given a project path, moves to/creates the corresponding directory 
 * in the CVS repository tree.
 *
 * return 0 if the directory existed, >0 if it was created
 */
int tosubdir(char * path) {
	return mknsubdir(path, 2);
}

/*
 * Receives the indication of the current project/repository on the control
 * socket
 */
void getDir() {
	char * d;
	char * dir;
	char logbuffer[256];
	int newdir;
	struct FileList * fl;
	struct FileList * dummy;

	d = getLine();
	if (d == NULL || strlen(d) == 0) {
		reply("ERROR Invalid directory\n");
		printf("Invalid chdir command\n");
		return;
	}	
   	else {
		if (current_dir != NULL) free(current_dir);
		current_dir = (char*) malloc(sizeof(char) * strlen(d));
		dir = strstr(d, "/");
		strcpy(current_dir, dir);
		free(d);
		reply("OK\n");
		printf("Changed current directory to :\n%s\n", current_dir);
		snprintf(logbuffer, 256, "Changed to directory %s", current_dir);
		logMessage("Chdir", logbuffer);
	}

	if (repository_files != NULL) {
		fl = repository_files;
		while(fl != NULL) {
			dummy = fl;
			fl = fl->next;
			dummy->next = NULL;
			if (dummy->filename != NULL) free(dummy->filename);
			if (dummy->RCSfile != NULL) free(dummy->RCSfile);
			if (dummy->RCSversion != NULL) free(dummy->RCSversion);
			free(dummy);
		}
		repository_files = NULL;
	}
	
	newdir = tosubdir(current_dir);
	if (newdir == 0 && writemode == 0) {
		parseDir(current_dir);
	}
}

/*
 * Receives a file label from the control socket 
 */
void getFLabel() {
	struct FileHeader h;
	struct FileList * f;
	RCSNode * RCShandle;
	char logbuffer[256];
	time_t thischange;

	/* receiving the header containing label information */
	clearHeader(&h);
	getHeader(&h);
	reply("OK\n");

	printf("Successfully received file Label\nFLabel header is :\n");
	printHeader(h);
	printf("\n");

	if (h.label == NULL) return;
	
	/* checking for file and updating CVS repository */
	if ((f = checkupdateList(h.filename, h.timestamp)) == NULL) {
		printf("Error : file %s is not in the repository\n", h.filename);
		snprintf(logbuffer, 256, "Could not label file %s : file not found", h.filename);
		logMessage("Label error", logbuffer);
	}
	else {
		thischange = makeGMTime(h.timestamp);

		if (thischange > f->lasttime) {
			RCShandle = RCS_parsercsfile(f->RCSfile);
			RCS_settag(RCShandle, h.label, NULL);
			RCS_rewrite(RCShandle, NULL, NULL);
			freercsnode(&RCShandle);

			f->lasttime = thischange;
		}
	}
}

/*
 * Receives a project label from the control socket
 */
void getPLabel() {
	struct FileHeader h;
	struct FileList * fl;
	RCSNode * RCShandle;
	time_t thischange;

	/* receiving the header containing label information */
	clearHeader(&h);
	getHeader(&h);
	reply("OK\n");

	printf("Successfully received project Label\nPLabel header is :\n");
	printHeader(h);
	printf("\n");
	
	if (h.label == NULL) return;

	thischange = makeGMTime(h.timestamp);
	
	/* setting the label on all the files in the project */
	for (fl = repository_files; fl != NULL; fl = fl->next) {
		if (thischange > fl->lasttime) {
			RCShandle = RCS_parsercsfile(fl->RCSfile);
			RCS_settag(RCShandle, h.label, NULL);
			RCS_rewrite(RCShandle, NULL, NULL);
			freercsnode(&RCShandle);

			fl->lasttime = thischange;
		}
	}

}

/*
 * Receives a delete order from the control socket
 * Rather obsolete : Deletes are interesting only for the files that are to be
 * recovered, and this means moving the file to, then from, the Attic in the CVS
 * repository. Thus Delete/Recover pairs may be ignored.
 */
void getDelete() {
	struct FileHeader h;

	clearHeader(&h);
	getHeader(&h);
	reply("OK\n");

	printf("Successfullly received Delete\nDelete header is :\n");
	printHeader(h);
	printf("\n");
	
	if (checkupdateList(h.filename, h.timestamp) == NULL) {
		printf("Error : file %s is not in the repository\n", h.filename);
	}
	else {
	}
}


/*
 * Receives a recover order from the control socket
 * Rather obsolete, too
 */
void getRecover() {
	struct FileHeader h;

	clearHeader(&h);
	getHeader(&h);
	reply("OK\n");

	printf("Successfullly received Recover\nRecover header is :\n");
	printHeader(h);
	printf("\n");
}

/* 
 * Receives a move order from the control socket
 * Useless : the client send files directly to their final position, so that
 * "Move"s in the file history ma be ignored.
 */
void getMove() {
	struct FileHeader h;

	clearHeader(&h);
	getHeader(&h);
	reply("OK\n");

	printf("Successfullly received Move\nMove header is :\n");
	printHeader(h);
	printf("\n");

	if (checkupdateList(h.filename, h.timestamp) == NULL) {
		printf("Error : file %s is not in the repository\n", h.filename);
	}
	else {
	}

}


/*
 * Gets a userlist from the control socket.
 * The list is in pseudo-XML format.
 * Mode can be 
 *   - "Simple" : all users are mapped to a single CVS user, set as "Default"
 *   - "Mapped" : users are real system/CVS users
 */
void getUsers() {
	struct LList * l;
	char * line;
	int len;
	int i;
	char * begin, * end;
	int arglen;
	char * arg;
	
	if (defaultuser != NULL) free(defaultuser);
	defaultuser = NULL;
	if (usermode != NULL) free(usermode);
	usermode = NULL;
	
	line = getLine();
	while (line != NULL && strstr(line, "<Userlist>") == NULL) {
		free(line);
		line = getLine();
	}

	if (line == NULL) {
		fprintf(stderr, "Error receiving user list\n");
		return;
	}
	
	free(line);
	line = getLine();
	
	while (line != NULL && strstr(line, "</Userlist>") == NULL) {
		len = strlen(line);
		for (i=0; i<len && line[i] != '<'; i++);
		if (i != len) line[i]=' ';
		for (i=len-1; i>=0 && line[i] != '>'; i--);
		if (i != -1) line[i]=' ';

		begin = strstr(line, ">");
		end = strstr(line, "<");
		if (begin == NULL || end == NULL || (arglen = (int) (end - begin) - 1) == 0)		return;
		begin++;
		while (*begin == ' ') begin++;
		end--;
		while(*end == ' ') end--;
		arglen = (int) (end - begin) + 1;

		if (arglen>0) {
		
			arg = (char *) malloc((arglen+1) * sizeof(char));
			strncpy(arg, begin, arglen);
			arg[arglen] = 0;

			if (strstr(line, "Mode") != NULL) usermode = arg;
			else if (strstr(line, "Default") != NULL) defaultuser = arg;
			else if (strstr(line, "User") != NULL) {
				l = (struct LList *) malloc(sizeof(struct LList));
				l->entry = arg;
				l->next = userlist;
				userlist = l;
			}
			else free(arg);
		}
		
		free(line);
		line = getLine();
	}

	if (line == NULL) {
		fprintf(stderr, "Error receiving user list\n");
		return;
	}

	free(line);
	reply("OK\n");
	
	printf("Successfully received user list\n");

	if (strcmp(usermode, "Simple") == 0) {
		printf("Adding entries in CVS user table\n");
		addUsers(userlist);
	}
	else {
		printf("Checking for users in CVS user table\n");
		checkUsers(userlist);
	}
	
	logMessage("Userlist", "Successfully received userlist");
}


/* 
 * Receives conversion parameters from the control socket.
 * Again, the parameters are expected in a pseudo-XML format.
 */
void getConfig() {
	char * line;
	int len;
	int i;
	char * begin, * end;
	int arglen;
	char * arg;
	
	line = getLine();
	while (line != NULL && strstr(line, "<Config>") == NULL) {
		free(line);
		line = getLine();
	}

	if (line == NULL) {
		fprintf(stderr, "Error receiving config\n");
		return;
	}

	free(line);
	line = getLine();
	
	while (line != NULL && strstr(line, "</Config>") == NULL) {
		len = strlen(line);
		for (i=0; i<len && line[i] != '<'; i++);
		if (i != len) line[i]=' ';
		for (i=len-1; i>=0 && line[i] != '>'; i--);
		if (i != -1) line[i]=' ';

		begin = strstr(line, ">");
		end = strstr(line, "<");
		if (begin == NULL || end == NULL || (arglen = (int) (end - begin) - 1) == 0) return;
		begin++;
		while (*begin == ' ') begin++;
		end--;
		while(*end == ' ') end--;
		arglen = (int) (end - begin) + 1;
		
		if (arglen > 0) {
			arg = (char *) malloc((arglen+1) * sizeof(char));
			strncpy(arg, begin, arglen);
			arg[arglen] = 0;
		
			if (strstr(line, "WriteMode") != NULL) {
				if (strcmp(arg, "Rewrite") == 0) writemode = 1;
				else writemode = 0;
			}
			else if (strstr(line, "AllowBinaries") != NULL) {
				if (strcmp(arg, "No") == 0) allowbinaries = 1;
				else allowbinaries = 0;
			}
		
			free(arg);
		}

		free(line);
		line = getLine();
	}

	if (line == NULL) {
		fprintf(stderr, "Error receiving config\n");
		return;
	}
	
	free(line);
	reply("OK\n");
	
	printf("Successfully received conversion parameters\n");
	logMessage("Config", "Successfully received conversion parameters");
}
	



		
