/*
 * SourceSafe/CVS conversion server
 * file.c
 * Implementation of file-related functions
 *
 * (c) Benjamin Loo - 2001
 *
 * 
 * You may distribute under the terms of the GNU General Public License as
 * specified in the README file that comes with the VSSExtractor/Cserver
 * source distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details. 
 *
 * Visual SourceSafe is a registered trademark from Microsoft Corporation.
 * 
 */

#include "cvs.h"
#include "file.h"
#include "cserver.h"
#include<string.h>
#include<stdlib.h>
#include<stdio.h>
#include<sys/types.h>
#include<utime.h>
#include<time.h>
#include<dirent.h>

/*
 * Parses a line of the pseudo-XML file header
 * If it matches a real header field, the corresponding fileheader structure
 * is updated, else the line is ignored.
 */
void getHeaderElement(char * line, struct FileHeader * h) {
	int len = strlen(line);
	int i;
	char * begin, * end;
	int arglen;
	char * arg;

	for (i=0; i<len && line[i] != '<'; i++);
	if (i != len) line[i]=' ';
	for (i=len-1; i>=0 && line[i] != '>'; i--);
	if (i != -1) line[i]=' ';

	begin = strstr(line, ">");
	end = strstr(line, "<");
	if (begin == NULL || end == NULL || (arglen = (int) (end - begin) - 1) == 0)		return;
	begin++;
	while (*begin == ' ') begin++;
	end--;
	while(*end == ' ') end--;
	arglen = (int) (end - begin) + 1;
	if (arglen <= 0) return;

	arg = (char *) malloc((arglen+1) * sizeof(char));
	strncpy(arg, begin, arglen);
	arg[arglen] = 0;

	if (strstr(line, "Filename") != NULL) h->filename = arg;
	else if (strstr(line, "Project") != NULL) h->project = arg;
	else if (strstr(line, "Author") != NULL) h->author = arg;
	else if (strstr(line, "Timestamp") != NULL) h->timestamp = arg;
	else if (strstr(line, "Version") != NULL) h->RCSversion = arg;
	else if (strstr(line, "Comment") != NULL) h->comment = arg;
	else if (strstr(line, "Label") != NULL) h->label = arg;
	else if (strstr(line, "Filetype") != NULL) {
		h->type = atoi(arg);
		free(arg);
	}
	else if (strstr(line, "Size") != NULL) {
		h->filesize = atoi(arg);
		free(arg);
	}
	else free(arg);
}

/*
 * Initializes an empty file header
 */
void clearHeader(struct FileHeader * h) {
	h->filename = NULL;
	h->tempfile = NULL;
	h->project = NULL;
	h->author = NULL;
	h->timestamp = NULL;
	h->comment = NULL;
	h->RCSversion = NULL;
	h->label = NULL;
	h->type = 0;
	h->filesize = 0;
}

/*
 * Frees the strings in a fileheader
 */
void deleteHeader(struct FileHeader * h) {
	if (h->filename) free(h->filename);
	if (h->tempfile) free(h->tempfile);
	if (h->comment) free(h->comment);
	if (h->author) free(h->author);
	if (h->timestamp) free(h->timestamp);
	if (h->label) free(h->label);
	if (h->RCSversion) free(h->RCSversion);
	if (h->project) free(h->project);
}

/*
 * Displays the contents of a fileheader
 */
void printHeader(struct FileHeader h) {
	printf("Filename    : %s\n", h.filename);
	printf("Project     : %s\n", h.project);
	printf("Author      : %s\n", h.author);
	printf("Timestamp   : %s\n", h.timestamp);
	printf("Comment     : %s\n", h.comment);
	printf("RCS version : %s\n", h.RCSversion);
	printf("Label       : %s\n", h.label);
	printf("File type   : %d\n", h.type);
	printf("File size   : %ld\n", h.filesize);
}

/*
 * Finds a given file in a linked list of files
 *
 * returns a pointer to the FileList element when found, else NULL
 */
struct FileList * findinList(char * filename, struct FileList * f) {
	struct FileList * fl = f;

	for (fl = f; fl != NULL && strcmp(fl->filename, filename) != 0; fl = fl->next);
	return fl;
}


/*
 * malloc's and initializes a new Filelist element
 *
 * returns a pointer to the new element
 */
struct FileList * newList(char * filename, int type) {
	struct FileList * f;
	char * version;

	f = (struct FileList *) malloc(sizeof(struct FileList));
	version = (char *) malloc(sizeof(char) * 4);

	strcpy(version, "1.0");
	f->filename = filename;
	f->RCSversion = version;
	f->RCSfile = NULL;
	f->lasttime = 0;
	f->type = type;
	f->next = NULL;

	return f;
}


/*
 * free's a filelist element
 */
void deleteList(struct FileList * f) {
	if (f == NULL) return;

	if (f->next != NULL) deleteList(f->next);
	free(f->filename);
	free(f->RCSversion);
	free(f->RCSfile);
	free(f);

	return;
}

/*
 * Increases of 1 a RCS version number.
 * It can be either a main trunk or a branch version number
 *
 * returns a newly allocated string containing the new version number
 */
char * updateVersion(char * oldversion) {
	char * version;
	char * tail;
	int i;
	int len;
	int lastdot=0;
	int minor;

	len = strlen(oldversion);
	for (i = 0; i<len; i++) if (oldversion [i] == '.') lastdot = i;
	version = (char *) malloc(sizeof(char) * (len + 2));
	strncpy(version, oldversion, lastdot+1);

	tail = &oldversion[lastdot+1];
	minor = atoi(tail);
	(void) snprintf(&version[lastdot+1], (len - lastdot) + 1, "%d", minor+1);
	
	return version;
}

/*
 * Given a filename, generates the complete RCS filename for it (CVS root + 
 * current repository + filename + ",v")
 *
 * returns a new string containing the filename
 */
char * makeRCSfilename(char * file) {
	char * f;

	f = (char *) malloc( sizeof(char) * (strlen(cvs_path) + strlen(current_dir) + strlen(file) + 4) );
	strcpy(f, cvs_path);
	strcat(f, current_dir);
	strcat(f, "/");
	strcat(f, file);
	strcat(f, ",v");

	return f;
}


/*
 * Looks for a file in the repository file list.
 * If the file is not present, creates an entry for it.
 *
 * returns a pointer to the corresponding FileList element
 */
struct FileList * checkaddList(char * filename, char * timestamp, int type) {
	struct FileList * f;
	char * version;
	
	if ((f = findinList(filename, repository_files)) == NULL) {
		f = newList(filename, type);
		f->lasttime = 0;
		f->RCSfile = makeRCSfilename(filename);
		f->type = type;
		f->next = repository_files;
		repository_files = f;
	}

	return f;
}


/*
 * Looks for a file in the repository file list.
 * It present, updates its timestamp.
 *
 * returns a pointer to the matching element, or NULL if not found
 */
struct FileList * checkupdateList(char * filename, char * timestamp) {
	struct FileList * f;

	if ((f = findinList(filename, repository_files)) != NULL) {
		f->lasttime = makeTime(timestamp);
	}
	
	return f;
}
	
/*
 * Displays current repository file list.
 */
void printList(struct FileList * f) {
	struct FileList * fl;

	printf("Current filelist is :\n");
	for (fl = f; fl != NULL; fl = fl->next) {
		printf("File : %s, Current version : %s\n", fl->filename, fl->RCSversion);
	}
	printf("\n");

}
	

/*
 * Removes head and queue whitespaces from a filename (if any).
 * 
 * returns a new string containing the trimmed filename
 */
char * trimFilename(char * filename) {
	char * c;
	int i, j;
	int len = strlen(filename);

	for (i = 0; i<len && (filename[i] == ' ' || filename[i] == '\t'); i++);
	for (j = len-1; j>0 && (filename[j] == ' ' || filename[j] == '\t'); j--);
	
	c = (char *) malloc( sizeof(char) * (j - i + 2) );
	strncpy(c, &filename[i], j-i+1);
	c[j-i+1] = 0;

	return c;
}


/*
 * Finds the next dot '.' character in a string, from character i.
 *
 * returns the index of next '.' in string c from beginning, as an int
 */
int nextDot(char * c, int i) {
	int j;
	for (j = i; j<strlen(c) && c[j] != '.'; j++);
	return j;
}


/*
 * Turns a RCS timestamp "YY[YY].MM.DD.HH.MM.SS" into a time value in seconds.
 *
 * returns the corresponding time as a time_t value 
 */
time_t makeTime(char * timestamp) {
	struct tm newtime;
	time_t t;
	char * ptr;
	int nextdot;
	char temp[6];
	int len;

	nextdot = nextDot(timestamp, 0);
	if (nextdot == 2) {
		strncpy(temp, timestamp, 2);
		temp[2] = 0;
		newtime.tm_year = atoi(temp);
	}
	else {
		strncpy(temp, timestamp, 4);
		temp[4] = 0;
		newtime.tm_year = atoi(temp) - 1900;
	}
	
	ptr = &timestamp[nextdot+1];
	nextdot = nextDot(timestamp, nextdot + 1);
	strncpy(temp, ptr, 2);
	temp[2] = 0;
	newtime.tm_mon = atoi(temp) - 1;
	
	ptr = &timestamp[nextdot+1];
	nextdot = nextDot(timestamp, nextdot + 1);
	strncpy(temp, ptr, 2);
	temp[2] = 0;
	newtime.tm_mday = atoi(temp);

	ptr = &timestamp[nextdot+1];
	nextdot = nextDot(timestamp, nextdot + 1);
	strncpy(temp, ptr, 2);
	temp[2] = 0;
	newtime.tm_hour = atoi(temp);
	
	ptr = &timestamp[nextdot+1];
	nextdot = nextDot(timestamp, nextdot + 1);
	strncpy(temp, ptr, 2);
	temp[2] = 0;
	newtime.tm_min = atoi(temp);

	ptr = &timestamp[nextdot+1];
	strncpy(temp, ptr, 2);
	temp[2] = 0;
	newtime.tm_sec = atoi(temp);

	t = mktime(&newtime);
	return t;
}

/*
 * Makes GMT time from a timestamp
 *
 * returns time as a time_t value
 */
time_t makeGMTime(char * timestamp) {
	time_t local;
	struct tm * split;
	local = makeTime(timestamp);
	split = gmtime(&local);
	return mktime(split);
}


/*
 * Given a timestamp, sets the access and modification times of a file
 * accorting to these date and time.
 */
void setFiletime(char * filename, char * timestamp) {
	time_t filetime;
	struct utimbuf tb;

	filetime = makeTime(timestamp);
	tb.actime = filetime;
	tb.modtime = filetime;
	utime(filename, &tb);
	
}	

/*
 * when adding an entry in the filelist, set links to the corresponding RCS file
 * creating/reseting the file if necessary
 */
void setRCSEntry(struct FileList * f, struct FileHeader h) {
	FILE * handle;
	int exist;
	char * typestr;
	RCSNode * RCShandle;

	if ((handle = fopen(f->RCSfile, "r")) == NULL) exist = 0;
	else {
		fclose(handle);
		exist = 1;
	}

	if (exist == 0 || writemode == 1) {
		unlink(f->RCSfile);
		if (allowbinaries == 0 ) {
			if (f->type == 0) typestr = NULL; else typestr = "b";
		}
		else typestr = NULL;
		add_rcs_file(h.comment, f->RCSfile, h.tempfile, NULL, typestr, NULL, NULL, 0, NULL, NULL, 0, NULL);
	}
	else {
		RCShandle = RCS_parsercsfile(f->RCSfile);
		if (f->RCSversion != NULL) free(f->RCSversion);
		f->RCSversion = RCS_head(RCShandle);
		f->lasttime = RCS_getrevtime(RCShandle, f->RCSversion, NULL, 0);
		
		freercsnode(&RCShandle);
	}
}	
		

/*
 * when accessing an existing directory in CVS repository, parses
 * the contents of the directory and builds a file list for this
 * directory
 */
void parseDir(char * dir) {
	char * fulldir;
	int len;
	DIR * dirhandle;
	struct dirent * nextfile;
	char * file;
	struct FileList * f;
	RCSNode * RCShandle;

	len = strlen(cvs_path) + strlen(dir) + 2;
	fulldir = (char *) malloc(sizeof(char) * len);
	strcpy(fulldir, cvs_path);
	strcat(fulldir, dir);
	
	if ((dirhandle = opendir(fulldir)) == NULL) return;

	while ((nextfile = readdir(dirhandle)) != NULL) {
		if (nextfile->d_name != NULL) {
			len = strlen(nextfile->d_name);
			file = (char *) malloc(sizeof(char) * (len + 1));
			strcpy(file, nextfile->d_name);

			if (strncmp(&file[len-2], ",v", 2) == 0) {
				/*it's a RCS file, proceed */
				f = (struct FileList *) malloc(sizeof(struct FileList));
				f->RCSfile = file;
				f->type = 0;
				f->RCSversion = NULL;
				f->lasttime = 0;
				f->filename = (char *) malloc(sizeof(char) * (len - 1));
				strncpy(f->filename, file, len - 2);
				f->filename[len-2] = 0;
				
				RCShandle = RCS_parsercsfile(f->RCSfile);
				if (f->RCSversion != NULL) free(f->RCSversion);
				f->RCSversion = RCS_head(RCShandle);
				f->lasttime = RCS_getrevtime(RCShandle, f->RCSversion, NULL, 0);
				freercsnode(&RCShandle);

				f->next = repository_files;
				repository_files = f;
			}
			else {
				/* Not an RCS file : ignore */
				free(file);
			}
		}	
	}
	
	closedir(dirhandle);
	free(fulldir);
}
	
/*
 * delete temp files after conversion
 */
void cleanTempdir() {
	DIR * dirhandle;
	struct dirent * nextfile;
	char * fullname;

	if ((dirhandle = opendir(temp_dir)) == NULL) return;

	while ((nextfile = readdir(dirhandle)) != NULL) {
		if (nextfile->d_name != NULL && strncmp(nextfile->d_name, "CvS", 3) == 0) {
			/* It is a temp file, delete it */
			fullname = (char *) malloc( sizeof(char) * (strlen(temp_dir) + strlen(nextfile->d_name) + 2));
			strcpy(fullname, temp_dir);
			strcat(fullname, "/");
			strcat(fullname, nextfile->d_name);
			unlink(fullname);
			free(fullname);
		}
	}

	closedir(dirhandle);
}

			
	

	
