#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "misc_gtk.h"
#include "gdl_ctree.h"

/******************************************************/
/* free memory used by a GDL_CT_ENTRY and its content */
/******************************************************/
void free_gdl_ct_entry(GDL_CT_ENTRY *gce)
{
	if(gce==NULL)
		return;

	switch(gce->ct_type)
	{
		case GDL_ROOT:
										if(gce->c.root.local_filename!=NULL)
											free(gce->c.root.local_filename);
										break;

		case GDL_SEARCH_PATTERN:
										if(gce->c.search_pattern.search_pattern!=NULL)
											free(gce->c.search_pattern.search_pattern);
										break;

		case GDL_ACTIVE_SEGMENT:
										if(gce->c.active_segment.nickname!=NULL)
											free(gce->c.active_segment.nickname);
										if(gce->c.active_segment.remote_filename!=NULL)
											free(gce->c.active_segment.remote_filename);
										if(gce->c.active_segment.status!=NULL)
											free(gce->c.active_segment.status);
										break;

		case GDL_STORED_SEGMENT:
										if(gce->c.stored_segment.stored_filename!=NULL)
											free(gce->c.stored_segment.stored_filename);
										if(gce->c.stored_segment.stored_interval!=NULL)
											free(gce->c.stored_segment.stored_interval);
										break;

		case GDL_RENAME_AT_END:
										if(gce->c.rename_at_end.final_directory!=NULL)
											free(gce->c.rename_at_end.final_directory);
										if(gce->c.rename_at_end.final_filename!=NULL)
											free(gce->c.rename_at_end.final_filename);
										break;

		case GDL_SCRIPT_AT_END:
										if(gce->c.script_at_end.script_to_start!=NULL)
											free(gce->c.script_at_end.script_to_start);
										break;

		default:
										fprintf(stderr,"free_gdl_ct_entry: unknown entry type: %d\n",gce->ct_type);
										break;
	}
	free(gce);
}

/**********************************************/
/* create a new GDL_CT_ENTRY for a root entry */
/**********************************************/
GDL_CT_ENTRY *new_gdl_ct_entry_as_root( unsigned long gdl_id, char *local_filename, unsigned long gdl_size, unsigned long byte_offset, unsigned long received_bytes, time_t start_time, unsigned long last_10sec_speed)
{
	GDL_CT_ENTRY *nw;

	nw=malloc(sizeof(GDL_CT_ENTRY));
	if(nw==NULL)
		return nw;

	nw->ct_type=GDL_ROOT;
	nw->c.root.gdl_id=gdl_id;
	nw->c.root.local_filename=strdup((local_filename!=NULL)?local_filename:"");
	nw->c.root.gdl_size=gdl_size;
	nw->c.root.byte_offset=byte_offset;
	nw->c.root.received_bytes=received_bytes;
	nw->c.root.start_time=start_time;
	nw->c.root.last_10sec_speed=last_10sec_speed;
	return nw;
}

/**************************************************/
/* create a new GDL_CT_ENTRY for a search pattern */
/**************************************************/
GDL_CT_ENTRY *new_gdl_ct_entry_as_search_pattern( unsigned long autoscan_id, unsigned int autoscan_type, char *search_pattern)
{
	GDL_CT_ENTRY *nw;

	nw=malloc(sizeof(GDL_CT_ENTRY));
	if(nw==NULL)
		return nw;

	nw->ct_type=GDL_SEARCH_PATTERN;
	nw->c.search_pattern.autoscan_id=autoscan_id;
	nw->c.search_pattern.autoscan_type=autoscan_type;
	nw->c.search_pattern.search_pattern=strdup((search_pattern!=NULL)?search_pattern:"");
	return nw;
}

/***************************************************/
/* create a new GDL_CT_ENTRY for an active segment */
/***************************************************/
GDL_CT_ENTRY *new_gdl_ct_entry_as_active_segment( char *nickname, char *remote_filename, unsigned long remote_file_size, char *status)
{
	GDL_CT_ENTRY *nw;

	nw=malloc(sizeof(GDL_CT_ENTRY));
	if(nw==NULL)
		return nw;

	nw->ct_type=GDL_ACTIVE_SEGMENT;
	nw->c.active_segment.nickname=strdup((nickname!=NULL)?nickname:"");
	nw->c.active_segment.remote_filename=strdup((remote_filename!=NULL)?remote_filename:"");
	nw->c.active_segment.remote_file_size=remote_file_size;
	nw->c.active_segment.status=strdup((status!=NULL)?status:"");
	return nw;
}

/**************************************************/
/* create a new GDL_CT_ENTRY for a stored segment */
/**************************************************/
GDL_CT_ENTRY *new_gdl_ct_entry_as_stored_segment( char *stored_filename, char *stored_interval)
{
	GDL_CT_ENTRY *nw;

	nw=malloc(sizeof(GDL_CT_ENTRY));
	if(nw==NULL)
		return nw;

	nw->ct_type=GDL_STORED_SEGMENT;
	nw->c.stored_segment.stored_filename=strdup((stored_filename!=NULL)?stored_filename:"");
	nw->c.stored_segment.stored_interval=strdup((stored_interval!=NULL)?stored_interval:"");
	return nw;
}

/*******************************************************/
/* create a new GDL_CT_ENTRY for a rename_at_end entry */
/*******************************************************/
GDL_CT_ENTRY *new_gdl_ct_entry_as_rename_at_end( char *final_directory, char *final_filename)
{
	GDL_CT_ENTRY *nw;

	nw=malloc(sizeof(GDL_CT_ENTRY));
	if(nw==NULL)
		return nw;

	nw->ct_type=GDL_RENAME_AT_END;
	nw->c.rename_at_end.final_directory=strdup((final_directory!=NULL)?final_directory:"");
	nw->c.rename_at_end.final_filename=strdup((final_filename!=NULL)?final_filename:"");
	return nw;
}

/*******************************************************/
/* create a new GDL_CT_ENTRY for a script_at_end entry */
/*******************************************************/
GDL_CT_ENTRY *new_gdl_ct_entry_as_script_at_end( char *script_to_start)
{
	GDL_CT_ENTRY *nw;

	nw=malloc(sizeof(GDL_CT_ENTRY));
	if(nw==NULL)
		return nw;

	nw->ct_type=GDL_SCRIPT_AT_END;
	nw->c.script_at_end.script_to_start=strdup((script_to_start!=NULL)?script_to_start:"");
	return nw;
}

/**************************************************************************************/
/* this function calls the given fnc for each selected entry of the "gdl_ctree" clist */
/**************************************************************************************/
/* fnc is an array of GDL_NUMBER_OF_GDL_CT_TYPE pointers, 1 pointer per GDL_CT_TYPE */
/* a NULL pointer means the entry is not processed                                  */
/************************************************************************************/
void generic_selected_gdl_entry_list_calls(void (*fnc[GDL_NUMBER_OF_GDL_CT_TYPE])(GtkCTree *ctree,GDL_CT_ENTRY *, void *), void *data)
{
   GtkWidget *w;
   GtkCList *clst;
   GtkCListRow *clist_row;
   int row;

   w=get_widget_by_widget_name("gdl_ctree");
   if(w==NULL)
      return;

   clst=GTK_CLIST(w);

   for(row=0;row<clst->rows;row++)
   {
      clist_row=g_list_nth(clst->row_list,row)->data;

      if(clist_row->state==GTK_STATE_SELECTED)
      {
			GDL_CT_ENTRY *gce;

			gce=gtk_clist_get_row_data(clst,row);
			if(gce!=NULL)
			{
				if(fnc[gce->ct_type]!=NULL)
         		(*fnc[gce->ct_type])(GTK_CTREE(w),(GDL_CT_ENTRY*)(clist_row->data),data);
			}
      }
   }
}

