/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * ls_cache.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_ZLIB
#include <zlib.h>
#endif
#include <gtk/gtk.h>

#include "ls_cache.h"
#include "main.h"

/***************************************/
/* load /LS result from the cache file */
/********************************************************************/
/* input: nickname= the nickname of the user having this share list */
/*        ttl_size= variable to set to the size of the shared files.*/
/************************************************************************/
/* output: the content (to free with free() when no more useful or NULL */
/************************************************************************/
char *load_ls_file(char *nickname, guint64 *ttl_size)
{
	FILE *f;
	GString *filename;
	char *ret=NULL;

	filename=g_string_new(dctc_main_dir->str);
	filename=g_string_append(filename,"/ls_cache/");
	filename=g_string_append(filename,nickname);

	f=fopen(filename->str,"rb");
	if(f!=NULL)
	{
		LS_CACHE_HEADER lsch;

		/* read LS cache header */
		if(fread(&lsch,1,sizeof(LS_CACHE_HEADER),f)!=sizeof(LS_CACHE_HEADER))
		{
			fprintf(stderr,"load_ls_file: Truncated LScache file %s: %s\n",filename->str,strerror(errno));
			fclose(f);
			goto abrt;
		}

		/* valid magic ? */
		if(lsch.ls_magic!=LS_CACHE_MAGIC)
		{	
			fprintf(stderr,"load_ls_file: Not a LScache file %s\n",filename->str);
			fclose(f);
			goto abrt;
		}

		switch(lsch.pack_algorithm)
		{
			case LS_CACHE_PACK_NONE:		/* result no compressed */
											ret=malloc(lsch.unpacked_size+1);
											if(ret==NULL)
											{
												fprintf(stderr,"load_ls_file: Out of memory\n");
												fclose(f);
												goto abrt;
											}

											if(fread(ret,1,lsch.unpacked_size,f)!=lsch.unpacked_size)
											{
												fprintf(stderr,"load_ls_file: truncated file\n");
												fclose(f);
												free(ret);
												ret=NULL;
												goto abrt;
											}
											fclose(f);
											ret[lsch.unpacked_size]='\0';
											break;

#ifdef HAVE_ZLIB
			case LS_CACHE_PACK_BZIP:
											{
												char *pack_area;
												unsigned long upack;
												pack_area=malloc(lsch.packed_size);
												if(pack_area==NULL)
												{
													fprintf(stderr,"load_ls_file: Out of memory\n");
													fclose(f);
													goto abrt;
												}
												if(fread(pack_area,1,lsch.packed_size,f)!=lsch.packed_size)
												{
													fprintf(stderr,"load_ls_file: truncated file\n");
													fclose(f);
													free(pack_area);
													goto abrt;
												}
												fclose(f);

												ret=malloc(lsch.unpacked_size+1);
												if(ret==NULL)
												{
													fprintf(stderr,"load_ls_file: Out of memory\n");
													free(pack_area);
													goto abrt;
												}

												upack=lsch.unpacked_size;
												if(uncompress(ret,&upack,pack_area,lsch.packed_size)!=Z_OK)
												{
													fprintf(stderr,"load_ls_file: Fail to uncompress LS cache file\n");
													free(pack_area);
													free(ret);
													ret=NULL;
													goto abrt;
												}

												free(pack_area);
												if(upack!=lsch.unpacked_size)
												{
													fprintf(stderr,"load_ls_file: Invalid uncompress result\n");
													free(ret);
													ret=NULL;
													goto abrt;
												}
												ret[lsch.unpacked_size]='\0';
											}
											break;
#endif

			default:
											fprintf(stderr,"load_ls_file: Unknown encoding scheme\n");
											fclose(f);
											break;
		}
	}
	else
	{
		fprintf(stderr,"load_ls_file: Unable to open LScache file %s: %s\n",filename->str,strerror(errno));
	}

	abrt:
	g_string_free(filename,TRUE);
	return ret;
}

/*****************************************************/
/* open the given file and load the header into lsch */
/**************************************************************/
/* output: =0:ok, no error, the file can be processed         */
/*        !=0:either an invalid file or an unprocessable file */
/**************************************************************/
int get_ls_file_header(char *filename, LS_CACHE_HEADER *lsch)
{
	FILE *f;
	int ret=1;

	f=fopen(filename,"rb");
	if(f!=NULL)
	{
		/* read LS cache header */
		if(fread(lsch,1,sizeof(LS_CACHE_HEADER),f)!=sizeof(LS_CACHE_HEADER))
			goto abrt;

		/* valid magic ? */
		if(lsch->ls_magic!=LS_CACHE_MAGIC)
			goto abrt;

		/* check if the algorithm is supported */
		switch(lsch->pack_algorithm)
		{
			case LS_CACHE_PACK_NONE:		/* result no compressed */
#ifdef HAVE_ZLIB
			case LS_CACHE_PACK_BZIP:
#endif
												ret=0;		/* file ok */
												break;
		}
		abrt:
		fclose(f);
	}

	return ret;
}

