#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <gnome.h>

#include "main.h"
#include "misc_gtk.h"
#include "bookmark.h"
#include "gui_layout.h"

/***************************************************************************************************/
/* Create a string which can be used as a value in gnome_config_set_string from the 3 given string */
/***************************************************************************************************/
/* the 3 strings are joined with a '|' inserted between them. */
/**************************************************************/
static GString *array_bookmark_to_gnome_bookmark_string(char *hubname, char *hub_description, char *hub_address, char *profile_name_to_use, char *flags)
{
	GString *str;
	str=g_string_new("");

	if(hubname!=NULL)
	{
		str=g_string_append(str,hubname);
	}
	str=g_string_append_c(str,'|');
	if(hub_description!=NULL)
	{
		str=g_string_append(str,hub_description);
	}
	str=g_string_append_c(str,'|');
	if(hub_address!=NULL)
	{
		str=g_string_append(str,hub_address);
	}
	str=g_string_append_c(str,'|');
	if(profile_name_to_use!=NULL)
	{
		str=g_string_append(str,profile_name_to_use);
	}
	str=g_string_append_c(str,'|');
	if(flags!=NULL)
	{
		str=g_string_append(str,flags);
	}

	return str;
}

/********************************************/
/* Opposition operation of the previous one */
/**************************************************************/
/* the first 3 strings are mandatory, the others are optional */
/**************************************************************/
static void gnome_bookmark_string_to_array_bookmark(char *gnome_config_string, char **hubname, char **hub_description, char **hub_address, char **profile_name_to_use, char **flags)
{
	gchar **ar;

	ar=g_strsplit(gnome_config_string,"|",0);
	if((ar!=NULL)&&(ar[0]!=NULL)&&(ar[1]!=NULL)&&(ar[2]!=NULL))
	{
		*hubname=strdup(ar[0]);
		*hub_description=strdup(ar[1]);
		*hub_address=strdup(ar[2]);

		*profile_name_to_use=strdup(ar[3]!=NULL ? ar[3] : "");
		*flags=strdup(ar[4]!=NULL ? ar[4] : "");
	}
	else
	{
		*hubname=NULL;
		*hub_description=NULL;
		*hub_address=NULL;
		*profile_name_to_use=NULL;
		*flags=NULL;
	}

	g_strfreev(ar);
	return;
}


/**************************************************************/
/* colorize entry of favorite clist existing in running clist */
/**************************************************************/
void colorize_favorite(GtkCList *fav_clst, GtkCList *running_clst)
{
	int i;
	char *t;
	int fnd;

	gtk_clist_freeze(fav_clst);
	for(i=0;i<fav_clst->rows;i++)
	{
		fnd=0;
		gtk_clist_get_text(fav_clst,i,2,&t);
		if(t!=NULL)
		{
			char *u;
			int j;

			for(j=0;j<running_clst->rows;j++)
			{
				gtk_clist_get_text(running_clst,j,0,&u);
				if((u!=NULL)&&(!strcmp(t,u)))
				{
					fnd=1;
					break;
				}
			}
		}

		if(fnd)
		{
			gtk_clist_set_background(fav_clst,i,&green);
		}
		else
		{
			gtk_clist_set_background(fav_clst,i,&white);
		}
	}

	gtk_clist_thaw(fav_clst);
}

/********************************************************************/
/* this function reloads the bookmark into the 'hub_favorite_clist' */
/*************************************************************************************/
/* NOTE: each clist row data is the bookmark entry number (in the gnome config file) */
/*************************************************************************************/
void reload_bookmark(void)
{
	GtkWidget *w;
	GtkCList *clst;
	int nb_entry;

	w=get_widget_by_widget_name("hub_favorite_clist");
	if(w==NULL)
		return;

	clst=GTK_CLIST(w);

	gtk_clist_freeze(clst);

	nb_entry=gnome_config_get_int_with_default("/" PROGNAME "/Bookmark_v1/next_key",0);

	gtk_clist_clear(clst);
	if(nb_entry!=0)
	{
		int i,j;
		GString *str;
		char *pval;
		str=g_string_new("");

		for(i=0;i<nb_entry;i++)
		{
			str=g_string_assign(str,"/" PROGNAME "/Bookmark_v1/");
			g_string_sprintfa(str,"%d",i);

			pval=gnome_config_get_string(str->str);
			if(pval!=NULL)
			{
				char *entry_list[5];

				gnome_bookmark_string_to_array_bookmark(pval, &(entry_list[0]), &(entry_list[1]), &(entry_list[2]), &(entry_list[3]), &(entry_list[4]));
				if(entry_list[0]!=NULL)
				{
					int num_row;
					if(entry_list[3]==NULL)
						entry_list[3]=strdup("");
					if(entry_list[4]==NULL)
						entry_list[4]=strdup("");

					num_row=gtk_clist_append(clst,entry_list);
					gtk_clist_set_row_data(clst,num_row,GINT_TO_POINTER(i));
					for(j=0;j<5;j++)
						free(entry_list[j]);
				}
			}
		}
		g_string_free(str,TRUE);
	}

	gtk_clist_sort(clst);

	colorize_favorite(clst,GTK_CLIST(get_widget_by_widget_name("running_hub_clist")));
	gtk_clist_thaw(clst);
}

/******************************************/
/* add/update a new entry to the bookmark */
/*****************************************************************************/
/* if id_to_use==-1, the function uses a free id, else, the given id is used */
/*****************************************************************************/
void add_entry_to_bookmark(int id_to_use,char *name,char *description, char *address, char *profilename, char *flags, int no_reload)
{
	int my_id;
	GString *str,*val;

	str=g_string_new("/" PROGNAME "/Bookmark_v1/");
	if(id_to_use==-1)
	{
		my_id=gnome_config_get_int_with_default("/" PROGNAME "/Bookmark_v1/next_key",0);
	}
	else
	{
		my_id=id_to_use;
	}
	g_string_sprintfa(str,"%d",my_id);
	val=array_bookmark_to_gnome_bookmark_string(name, description, address, profilename, flags);
	gnome_config_set_string(str->str,val->str);
	g_string_free(str,TRUE);
	g_string_free(val,TRUE);
	if(id_to_use==-1)
	{	/* update the next id if we have used a new one */
		my_id++;
		gnome_config_set_int("/" PROGNAME "/Bookmark_v1/next_key",my_id);
	}
	
	gnome_config_sync ();

	if(no_reload!=1)
		reload_bookmark();
}

/*********************************************************/
/* search if the searched_flag is inside the flag_string */
/*********************************************************/
/* output: 1=yes, 0=no */
/***********************/
int flag_is_set(char *flag_string,char *searched_flag)
{
	int lstr=strlen(flag_string);
	int lwanted=strlen(searched_flag);
	int i;

	lstr-=lwanted;
	
	for(i=0;i<=lstr;i++)
	{
		if(!strncmp(flag_string+i,searched_flag,lwanted))
		{
			/* we may have found our flag */
			if((flag_string[i+lwanted]=='\0')||		/* either a terminal string */
				(flag_string[i+lwanted]==','))		/* or a comma terminated one */
				return 1;
		}
	}

	return 0;
}

/*********************************************/
/* toggle a flag of an entry of the bookmark */
/*********************************************/
void toggle_entry_flag_from_bookmark_by_id(int my_id, char *flag_to_toggle, int no_reload)
{
	GString *str,*val;
	gchar *pval;
	char *entry_list[5];
	gchar **flags;
	int fnd=0;
	int j;
	GString *new_flag_val;

	str=g_string_new("/" PROGNAME "/Bookmark_v1/");
	g_string_sprintfa(str,"%d",my_id);
	pval=gnome_config_get_string(str->str);
	if(pval==NULL)
	{
		/* the entry does not exist */
		g_string_free(str,TRUE);
		return;
	}

	/* retrieve entry to update */
	gnome_bookmark_string_to_array_bookmark(pval, &(entry_list[0]), &(entry_list[1]), &(entry_list[2]), &(entry_list[3]), &(entry_list[4]));
	if(entry_list[0]==NULL)
	{
		/* the entry does not exist */
		g_string_free(str,TRUE);
		return;
	}

	if(entry_list[3]==NULL)
		entry_list[3]=strdup("");
	if(entry_list[4]==NULL)
		entry_list[4]=strdup("");

	/* now, we have to toggle the wanted flag */
	flags=g_strsplit(entry_list[4],",",0);
	new_flag_val=g_string_new("");
	j=0;
	while(flags[j]!=NULL)
	{
		if(!strcmp(flag_to_toggle,flags[j]))
			fnd=1;
		else
		{
			if(new_flag_val->len!=0)
				new_flag_val=g_string_append_c(new_flag_val,',');
			new_flag_val=g_string_append(new_flag_val,flags[j]);
		}
		j++;
	}
	if(!fnd)
	{
		if(new_flag_val->len!=0)
			new_flag_val=g_string_append_c(new_flag_val,',');
		new_flag_val=g_string_append(new_flag_val,flag_to_toggle);
	}
	g_strfreev(flags);

	/* and save updated entry */
	val=array_bookmark_to_gnome_bookmark_string(entry_list[0], entry_list[1], entry_list[2], entry_list[3], new_flag_val->str);
	gnome_config_set_string(str->str,val->str);

	g_string_free(new_flag_val,TRUE);
	g_string_free(str,TRUE);
	g_string_free(val,TRUE);
	for(j=0;j<5;j++)
		free(entry_list[j]);
	
	gnome_config_sync();

	if(no_reload!=1)
		reload_bookmark();
}

/***********************************************/
/* set the profile of an entry of the bookmark */
/***********************************************/
void set_entry_profile_from_bookmark_by_id(int my_id, char *profile_to_use, int no_reload)
{
	GString *str,*val;
	gchar *pval;
	char *entry_list[5];
	int j;

	str=g_string_new("/" PROGNAME "/Bookmark_v1/");
	g_string_sprintfa(str,"%d",my_id);
	pval=gnome_config_get_string(str->str);
	if(pval==NULL)
	{
		/* the entry does not exist */
		g_string_free(str,TRUE);
		return;
	}

	/* retrieve entry to update */
	gnome_bookmark_string_to_array_bookmark(pval, &(entry_list[0]), &(entry_list[1]), &(entry_list[2]), &(entry_list[3]), &(entry_list[4]));
	if(entry_list[0]==NULL)
	{
		/* the entry does not exist */
		g_string_free(str,TRUE);
		return;
	}

	if(entry_list[3]!=NULL)
		free(entry_list[3]);
	entry_list[3]=strdup(profile_to_use);
	if(entry_list[4]==NULL)
		entry_list[4]=strdup("");

	/* and save updated entry */
	val=array_bookmark_to_gnome_bookmark_string(entry_list[0], entry_list[1], entry_list[2], entry_list[3], entry_list[4]);
	gnome_config_set_string(str->str,val->str);

	g_string_free(str,TRUE);
	g_string_free(val,TRUE);
	for(j=0;j<5;j++)
		free(entry_list[j]);
	
	gnome_config_sync();

	if(no_reload!=1)
		reload_bookmark();
}



/************************************/
/* delete a entry from the bookmark */
/************************************/
void delete_entry_from_bookmark_by_id(unsigned int bm_id, int no_reload)
{
	GString *str;

	str=g_string_new("/" PROGNAME "/Bookmark_v1/");
	g_string_sprintfa(str,"%d",bm_id);
	gnome_config_clean_key(str->str);
	gnome_config_sync ();
	g_string_free(str,TRUE);

	if(no_reload!=1)
		reload_bookmark();
}

/******************************************************************************************************/
/* Convert bookmarks from the old bookmark format (in /Bookmark/Content) into the new bookmark format */
/******************************************************************************************************/
void convert_old_bookmark_to_new_bookmark_format(void)
{
	int nb_entry;
	gchar **entry_list;
	int i;
	char num[32];

	gnome_config_get_vector("/" PROGNAME "/Bookmark/Content",&nb_entry,&entry_list);

	if(nb_entry!=0)
	{
		if(entry_list!=NULL)
		{
			for(i=0;i<nb_entry;i+=3)
			{
				/* entry_list[i] = Hubname */
				/* entry_list[i+1] = Hub description */
				/* entry_list[i+2] = Hub address */
				GString *bm_val;

				bm_val=array_bookmark_to_gnome_bookmark_string(entry_list[i],entry_list[i+1],entry_list[i+2],NULL,NULL);
				
				sprintf(num,"/" PROGNAME "/Bookmark_v1/%d",i/3);
				gnome_config_set_string(num,bm_val->str);
				g_string_free(bm_val,TRUE);
			}
		}
		gnome_config_set_int("/" PROGNAME "/Bookmark_v1/next_key",nb_entry/3);
	}

	gnome_config_sync ();
	
	/* entry list is no more useful */
	if(entry_list!=NULL)
		g_strfreev(entry_list);

	/* and delete the old bookmark */
	gnome_config_clean_section("/" PROGNAME "/Bookmark/");
	gnome_config_sync ();
}

