/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * network.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <string.h>
#include <net/if.h>
#include <net/route.h>
#include <netdb.h>
#include <ctype.h>
#include <errno.h>
#include <glib.h>

static struct sockaddr_in INET_ZERO = { AF_INET };

/****************************************************************/
/* the following parameters are used to handle SOCKS proxy (v4) */
/****************************************************************/
char *socks_ip=NULL;						/* hostname or IP of the socks proxy */
unsigned short socks_port=1080;		/* port number of the socks proxy */
char *socks_name=NULL;					/* userID to use on the socks proxy (or NULL if none required) */
char *socks_pass=NULL;					/* password to use on the socks proxy (or NULL if none required) */
												/* only valid when socks_name!=NULL AND SOCKSv5 enabled */
int use_socks_v5=FALSE;					/* prefer SOCKSv5 instead of v4 */

/****************************************************************************************/
/* perform the socket connection. If socks_ip is NULL, a direct connection is performed */
/* if socks_ip!=NULL, the function tries to establish a connection thru a SOCKS proxy   */
/* using the given information.                                                         */
/****************************************************************************************/
/* WARNING: if the given socks is in non-blocking mode, the connection will fail */
/*********************************************************************************/
static int do_connect_opt_socks(int desc,struct sockaddr_in nom)
{
	int ret=-1;
	GByteArray *cmd=NULL;

	if(socks_ip==NULL)
	{	/* SOCKS proxy disabled */
		ret=connect(desc,(void*)&nom,sizeof(nom));
	}
	else
	{	/* SOCKS proxy enabled */
		if(use_socks_v5==FALSE)
		{	/* SOCKS proxy v4 */
			struct sockaddr_in socks_cnx;
			struct hostent *socks_host;
			unsigned char val[9];
		
			printf("connecting thru SOCKSv4.\n");

			if((socks_host=gethostbyname(socks_ip))==NULL)
				goto eofunc;
			memcpy(&socks_cnx.sin_addr,socks_host->h_addr,socks_host->h_length);
			socks_cnx.sin_family=AF_INET;
			socks_cnx.sin_port=htons(socks_port);
	
			ret=connect(desc,(void*)&socks_cnx,sizeof(socks_cnx));
			if(ret==-1)
			{
				perror("socks connect");
				goto eofunc;
			}

			cmd=g_byte_array_new();
			val[0]=4;		/* socks version 4 */
			val[1]=1;		/* CONNECT */
			val[2]=ntohs(nom.sin_port)>>8;
			val[3]=ntohs(nom.sin_port)&255;
			val[4]=ntohl(nom.sin_addr.s_addr)>>24;
			val[5]=ntohl(nom.sin_addr.s_addr)>>16;
			val[6]=ntohl(nom.sin_addr.s_addr)>>8;
			val[7]=ntohl(nom.sin_addr.s_addr);
			cmd=g_byte_array_append(cmd,val,8);
			if(socks_name!=NULL)
			{
				cmd=g_byte_array_append(cmd,socks_name,strlen(socks_name));
			}
			val[0]='\0';
			cmd=g_byte_array_append(cmd,val,1);
	
			if(send(desc,cmd->data,cmd->len,MSG_NOSIGNAL|MSG_DONTWAIT)!=cmd->len)
			{
				perror("socks send CONNECT");
				ret=-1;
				goto eofunc;
			}
	
			if(recv(desc,val,8,MSG_WAITALL|MSG_NOSIGNAL)!=8)
			{
				perror("socks receive CONNECT result");
				ret=-1;
				goto eofunc;
			}
	
			if(val[0]!=0)		/*  reply code version 0 ? */
			{
				fprintf(stderr,"socks CONNECT result: unknown reply code version (%d)",(int)(val[0]));
				ret=-1;
				goto eofunc;
			}
	
#if 0
			printf("%X\n",val[1]);
#endif
			if(val[1]!=90)
			{
				fprintf(stderr,"socks CONNECT result: access denied or failed (%d)",(int)(val[1]));
				ret=-1;
				goto eofunc;
			}
			ret=0;
#if 0
			printf("SOCKSv4 connected\n");
#endif
		}
		else
		{	/* SOCKS proxy v5 */
			struct sockaddr_in socks_cnx;
			struct hostent *socks_host;
			unsigned char val[512];
		
			printf("connecting thru SOCKSv4.\n");

			if((socks_host=gethostbyname(socks_ip))==NULL)
				goto eofunc;
			memcpy(&socks_cnx.sin_addr,socks_host->h_addr,socks_host->h_length);
			socks_cnx.sin_family=AF_INET;
			socks_cnx.sin_port=htons(socks_port);
	
			ret=connect(desc,(void*)&socks_cnx,sizeof(socks_cnx));
			if(ret==-1)
			{
				perror("socks connect");
				goto eofunc;
			}

			/* perform initial negociation */
			cmd=g_byte_array_new();
			val[0]=5;		/* socks version 5 */
			val[1]=2;		/* we support 2 methods */
			val[2]=0;		/* method 0: no login required */
			val[3]=2;		/* method 2: connection with login/password */
			cmd=g_byte_array_append(cmd,val,4);
			if(send(desc,cmd->data,cmd->len,MSG_NOSIGNAL|MSG_DONTWAIT)!=cmd->len)
			{
				perror("socks initial negociation");
				ret=-1;
				goto eofunc;
			}
	
			if(recv(desc,val,2,MSG_WAITALL|MSG_NOSIGNAL)!=2)
			{
				perror("socks receive initial negociation result");
				ret=-1;
				goto eofunc;
			}

			if(val[0]!=5)
			{
				fprintf(stderr,"socks initial negociation result: unknown reply code version (%d)",(int)(val[0]));
				ret=-1;
				goto eofunc;
			}
	
			switch(val[1])
			{
				case 0:			/* no login required */
							break;

				case 2:			/* connection with login/password */
							/* create a string: 0x05, login len, login, password len, password */
							cmd=g_byte_array_set_size(cmd,0);		/* clear the temp buffer */
							val[0]=5;
							cmd=g_byte_array_append(cmd,val,1);
							if(socks_name!=NULL)
							{
								val[1]=strlen(socks_name);
								cmd=g_byte_array_append(cmd,val+1,1);	/* no login */
								cmd=g_byte_array_append(cmd,socks_name,strlen(socks_name));

								if(socks_pass!=NULL)
								{
									val[2]=strlen(socks_pass);
									cmd=g_byte_array_append(cmd,val+2,1);	/* no login */
									cmd=g_byte_array_append(cmd,socks_pass,strlen(socks_pass));
								}
								else
								{
									val[2]=0;
									cmd=g_byte_array_append(cmd,val+2,1);	/* no password */
								}
							}
							else
							{
								val[1]=0;
								cmd=g_byte_array_append(cmd,val+1,1);	/* no login */
								val[2]=0;
								cmd=g_byte_array_append(cmd,val+2,1);	/* no password */
							}

							if(send(desc,cmd->data,cmd->len,MSG_NOSIGNAL|MSG_DONTWAIT)!=cmd->len)
							{
								perror("socks send login/password");
								ret=-1;
								goto eofunc;
							}

							if(recv(desc,val,2,MSG_WAITALL|MSG_NOSIGNAL)!=2)
							{
								perror("socks login/password stage");
								ret=-1;
								goto eofunc;
							}

							if(val[0]!=5)
							{
								fprintf(stderr,"socks login/password stage: unknown reply code version (%d)",(int)(val[0]));
								ret=-1;
								goto eofunc;
							}

							if(val[1]!=0)
							{
								fprintf(stderr,"socks login/password stage result: unknown return code (%d)",(int)(val[1]));
								ret=-1;
								goto eofunc;
							}
							break;

				default:
							fprintf(stderr,"socks initial negociation result: unknown authentification method (%d)",(int)(val[1]));
							ret=-1;
							goto eofunc;
			}

			/* authentification done, time to establish connection */
			cmd=g_byte_array_set_size(cmd,0);		/* clear the temp buffer */
			val[0]=5;		/* socks version 5 */
			val[1]=1;		/* CONNECT */
			val[2]=0;		/* reserved */
			val[3]=1;		/* IPv4 address */
			val[4]=ntohl(nom.sin_addr.s_addr)>>24;
			val[5]=ntohl(nom.sin_addr.s_addr)>>16;
			val[6]=ntohl(nom.sin_addr.s_addr)>>8;
			val[7]=ntohl(nom.sin_addr.s_addr);
			val[8]=ntohs(nom.sin_port)>>8;
			val[9]=ntohs(nom.sin_port)&255;
			cmd=g_byte_array_append(cmd,val,10);
	
			if(send(desc,cmd->data,cmd->len,MSG_NOSIGNAL|MSG_DONTWAIT)!=cmd->len)
			{
				perror("socks send CONNECT");
				ret=-1;
				goto eofunc;
			}
	
			if(recv(desc,val,10,MSG_WAITALL|MSG_NOSIGNAL)!=10)
			{
				perror("socks receive CONNECT result");
				ret=-1;
				goto eofunc;
			}

			if(val[0]!=5)
			{
				fprintf(stderr,"socks CONNECT: unknown reply code version (%d)",(int)(val[0]));
				ret=-1;
				goto eofunc;
			}

			if(val[1]!=0)
			{
				fprintf(stderr,"socks CONNECT result: unknown return code (%d)",(int)(val[1]));
				ret=-1;
				goto eofunc;
			}
		
			ret=0;
#if 1
			printf("SOCKSv5 connected\n");
#endif
		}
	}
	eofunc:
	if(cmd!=NULL)
		g_byte_array_free(cmd,TRUE);
		
	return ret;
}

/*************************************************/
/* open a TCP connection with the requested port */
/*************************************************/
/* en entree: port= port demande     */
/* en sortie: numero de socket ou -1 */
/*************************************/
int _x_tcp (int port)
{
	int f, len;
	struct sockaddr_in me ;
	struct sockaddr_in s_in;
	int zz=1;
  
	me = s_in = INET_ZERO;
  
	me.sin_port = htons((unsigned short) port);
	me.sin_family = AF_INET;
  
	if((f=socket(AF_INET,SOCK_STREAM,0)) == -1) return(-1);
  
	if(setsockopt(f,SOL_SOCKET,SO_REUSEADDR,(char *)&zz,sizeof(zz)) < 0 ||
		bind(f,(struct sockaddr *) &me,(len = sizeof(me))) < 0)
	{
		perror("setsock or bind");
		close(f);
		return(-1);
	}

	return(f);
}

/*************************************************/
/* open a UDP connection with the requested port */
/************************************************************/
/* input: port= wanted port number (0=find an unused one)   */
/*        if port!=0 and port_upper_range > port and <65536 */
/*        the function tries to bind on a free port in the  */
/*        range [port:port_upper_range].                    */
/*        if port_upper_range==0, only "port" is used       */
/* output: socket descriptor or -1                          */
/*        if port!=0 and port_upper_range=0, a returned     */
/*        value of -1 probably means the port is busy.      */
/*        if port_upper_range!=0, a returned value of -1    */
/*        probably means all ports on the given range are   */
/*        busy.                                             */
/************************************************************/
int _x_udp (int port, int port_upper_range)
{
	int f, len, zz;
	struct sockaddr_in me ;
	struct sockaddr_in s_in;

	if((f=socket(AF_INET,SOCK_DGRAM,0)) == -1) return(-1);

	zz=1;

	if(setsockopt(f,SOL_SOCKET,SO_REUSEADDR,(char *)&zz,sizeof(zz)) < 0)
	{
		perror("_x_udp");
		close(f);
		return(-1);
	}

	/* try to bind on the wanted port number */
	retry_bind:
	me = s_in = INET_ZERO;
	me.sin_port = htons((unsigned short) port);
	me.sin_family = AF_INET;
	if(bind(f,(struct sockaddr *) &me,(len = sizeof(me))) < 0 )
	{
		if(port_upper_range!=0)
		{
			port++;
			if(port<=port_upper_range)
				goto retry_bind;
		}
		perror("_x_udp");
		close(f);
		return(-1);
	}

	if(getsockname(f,(struct sockaddr *)&s_in,&len) < 0)
	{
		perror("_x_udp");
		close(f);
		return(-1);
	}
	return(f);
}	 

/***************************************/
/* conversion nom d'hote -> adresse ip */
/***************************************/
/* en sortie: -1= fail, 0=ok */
/*****************************/
int str_to_inaddr (const char *str, struct in_addr *iap)
{
#ifdef DONT_HAVE_INET_ATON
	*(int *)iap = inet_addr(str);
	if (*(int *)iap == -1)
#else
	if (!inet_aton(str, iap))
#endif
	{
		struct hostent *he;

		if ((he = gethostbyname(str)))
		{
			if ((unsigned)he->h_length > sizeof *iap)
				he->h_length = sizeof *iap;

			memcpy(iap, he->h_addr, he->h_length);
		}
		else
		{
			return -1;
		}
	}

	return 0;
}

/*****************************************************/
/* routine fixant le flag non bloquant sur un socket */
/*****************************************************/
void set_non_bloquant_sock(int socket_fd)
{
	long retval;

	retval=fcntl(socket_fd,F_GETFL);
	retval|= O_NONBLOCK;

	fcntl(socket_fd,F_SETFL,retval);
}


/*************************************************/
/* routine fixant le flag bloquant sur un socket */
/*************************************************/
void set_bloquant_sock(int socket_fd)
{
	long retval;

	retval=fcntl(socket_fd,F_GETFL);
	retval&= ~O_NONBLOCK;

	fcntl(socket_fd,F_SETFL,retval);
}

/*************************************************/
/* routine to set TOS on socket                  */
/*************************************************/
void set_tos_sock(int socket_fd, int tos)
{
	if(tos)
		if(setsockopt(socket_fd, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)))
			perror("Unable to set TOS:");
}

/****************************************************/
/* create a socket and connect it to the given host */
/***************************************************************/
/* input: non_block=1 (socket is created with O_NONBLOCK flag) */
/*********************************************************************************************/
/* output: sock_fd or -1,-2 or -3                                                            */
/*         if non_block==1, returned sock_fd must be tested (ready_to_write) before using it */
/*********************************************************************************************/
/* -1=unknown host           */
/* -2=socket creation failed */
/* -3=unable to contact host */
/***********************************************************************************/
/* WARNING: if non_block==1 and a SOCKS proxy is used, the connection will fail... */
/***********************************************************************************/
int create_and_open_sock_on(char *hostname,unsigned short port, int non_block)
{
	struct sockaddr_in addr;
	int s;
	int code;

	addr.sin_family=AF_INET;
	addr.sin_port=htons(port);
	if(str_to_inaddr(hostname,&(addr.sin_addr)))
		return -1;

	if((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
	{
		perror("socket:");
		return -2;
	}

	if(non_block)
		set_non_bloquant_sock(s);

#if 0
	code=connect(s,(struct sockaddr*)&addr,sizeof addr);
#else
	code=do_connect_opt_socks(s,addr);
#endif

	if(non_block==0)
	{
		if(code==-1)
		{
			perror("connect");
			close(s);
			return -3;
		}
	}
	else
	{
		/* socket doesn't block, -1 is allowed if errno==EINPROGRESS */
		if((code==-1)&&(errno!=EINPROGRESS))
		{
			perror("connect");
			close(s);
			return -3;
		}
	}

	return s;
}

/******************************/
/* return the default host IP */
/************************************************************************/
/* the default host IP is the IP of the interface used as default route */
/************************************************************************/
GString *get_default_host_ip(void)
{
	FILE *f;
	char buf[512];

	/* the following script does this: */
	/* 1) extract the interface used by the default route: netstat -rn | grep ^0.0.0.0 | awk '{print $8;} */
	/* 2) obtain the IP of this interface: netstat xxx | fgrep inet | cut -d : -f 2 | awk '{print $1;}' */
	const char *cmd=
#if 0
		"/sbin/ifconfig `netstat -rn | grep ^0.0.0.0 | awk '{print $8;}'` | fgrep inet | cut -d : -f 2 | awk '{print $1;}'";
#endif

#ifndef ROUTE_HAS_GET_DEFAULT
		"/sbin/ifconfig `netstat -rn | awk '/^0.0.0.0/{print$NF}'` | fgrep inet | cut -d : -f 2 | awk '{print $1;}'";
#else
		"/sbin/route -n get default | awk '/local/{print$NF}'";
#endif

	f=popen(cmd,"r");
	if(f==NULL)
	{
		fprintf(stderr,"Unable to create a shell to obtain default IP.\n");
		return NULL;
	}

	if(fgets(buf,sizeof(buf),f)==NULL)
	{
		pclose(f);
		fprintf(stderr,"No default interface found.\n");
		return NULL;
	}

	pclose(f);

	if(strlen(buf)!=0)
	{
		char *t;

		t=buf;
		while( (*t) && ((*t=='.') || (isdigit(*t))) )
			t++;

		*t='\0';
		return g_string_new(buf);
	}
	else
		return NULL;
}

