/******************************************************************************
 *
 * $Id: config_templ.l,v 1.8 2001/01/01 10:15:16 root Exp $
 *
 * Copyright (C) 1997-2003 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 */

%{

/*
 *	includes
 */
#include <stdio.h>
#include <stdlib.h>
//#include <iostream.h>
#include <assert.h>
#include <ctype.h>
#include <stdarg.h>

#include <qfileinfo.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qstack.h>
  
#include "config.h"
#include "version.h"

#include "lang_cfg.h"

#undef Config_getString
#undef Config_getInt
#undef Config_getList
#undef Config_getEnum
#undef Config_getBool

// use in-class definitions
#define Config_getString(val)  getString(__FILE__,__LINE__,val)
#define Config_getInt(val)     getInt(__FILE__,__LINE__,val)
#define Config_getList(val)    getList(__FILE__,__LINE__,val)
#define Config_getEnum(val)    getEnum(__FILE__,__LINE__,val)
#define Config_getBool(val)    getBool(__FILE__,__LINE__,val)
  
void config_err(const char *fmt, ...)
{
  va_list args;
  va_start(args, fmt);
  vfprintf(stderr, fmt, args);
  va_end(args); 
}
void config_warn(const char *fmt, ...)
{
  va_list args;
  va_start(args, fmt);
  vfprintf(stderr, fmt, args);
  va_end(args);
}

#define MAX_INCLUDE_DEPTH 10
#define YY_NEVER_INTERACTIVE 1
#define YY_NO_UNPUT

/* -----------------------------------------------------------------
 */

QCString ConfigOption::convertToComment(const QCString &s)
{
  QCString result;
  if (s.isEmpty()) return result;
  else
  {
    result+="# ";
    QCString tmp=s.stripWhiteSpace();
    char *p=tmp.data();
    char c;
    while ((c=*p++))
    {
      if (c=='\n') result+="\n# ";
      else result+=c;
    }
    result+='\n';
  }
  return result;
}

void ConfigOption::writeBoolValue(QTextStream &t,bool v)
{
  if (v) t << "YES"; else t << "NO";
}

void ConfigOption::writeIntValue(QTextStream &t,int i)
{
  t << i;
}

void ConfigOption::writeStringValue(QTextStream &t,QCString &s)
{
  const char *p=s.data();
  char c;
  bool needsEscaping=FALSE;
  if (p)
  {
    while ((c=*p++)!=0 && !needsEscaping) 
      needsEscaping = (c==' ' || c=='\n' || c=='\t' || c=='"');
    if (needsEscaping)
    { 
      t << "\"";
      p=s.data();
      while (*p)
      {
	if (*p=='"') t << "\\"; // escape quotes
	t << *p++;
      }
      t << "\"";
    }
    else
    {
      t << s;
    }
  }
}

void ConfigOption::writeStringList(QTextStream &t,QStrList &l)
{
  const char *p = l.first();
  bool first=TRUE;
  while (p)
  {
    QCString s=p;
    if (!first) t << "                         ";
    first=FALSE;
    writeStringValue(t,s);
    p = l.next();
    if (p) t << " \\" << endl;
  }
}

/* -----------------------------------------------------------------
 */

Config *Config::m_instance = 0;

void ConfigInt::convertStrToVal() 
{
  if (!m_valueString.isEmpty())
  {
    bool ok;
    int val = m_valueString.toInt(&ok);
    if (!ok || val<m_minVal || val>m_maxVal)
    {
      config_warn("Warning: argument `%s' for option %s is not a valid number in the range [%d..%d]!\n"
                "Using the default: %d!\n",m_valueString.data(),m_name.data(),m_minVal,m_maxVal,m_value);
    }
    m_value=val;
  }
}

void ConfigBool::convertStrToVal()
{
  QCString val = m_valueString.stripWhiteSpace().lower();
  if (!val.isEmpty())
  {
    if (val=="yes" || val=="true" || val=="1") 
    {
      m_value=TRUE;
    }
    else if (val=="no" || val=="false" || val=="0")
    {
      m_value=FALSE;
    }
    else
    {
      config_warn("Warning: argument `%s' for option %s is not a valid boolean value\n"
                "Using the default: %s!\n",m_valueString.data(),m_name.data(),m_value?"YES":"NO");
    }
  }
}

QCString &Config::getString(const char *fileName,int num,const char *name) const
{
  ConfigOption *opt = m_dict->find(name);
  if (opt==0) 
  {
    config_err("%s<%d>: Internal error: Requested unknown option %s!\n",fileName,num,name);
    exit(1);
  }
  else if (opt->kind()!=ConfigOption::O_String)
  {
    config_err("%s<%d>: Internal error: Requested option %s not of string type!\n",fileName,num,name);
    exit(1);
  }
  return *((ConfigString *)opt)->valueRef();
}

QStrList &Config::getList(const char *fileName,int num,const char *name) const
{
  ConfigOption *opt = m_dict->find(name);
  if (opt==0) 
  {
    config_err("%s<%d>: Internal error: Requested unknown option %s!\n",fileName,num,name);
    exit(1);
  }
  else if (opt->kind()!=ConfigOption::O_List)
  {
    config_err("%d<%d>: Internal error: Requested option %s not of list type!\n",fileName,num,name);
    exit(1);
  }
  return *((ConfigList *)opt)->valueRef();
}

QCString &Config::getEnum(const char *fileName,int num,const char *name) const
{
  ConfigOption *opt = m_dict->find(name);
  if (opt==0) 
  {
    config_err("%s<%d>: Internal error: Requested unknown option %s!\n",fileName,num,name);
    exit(1);
  }
  else if (opt->kind()!=ConfigOption::O_Enum)
  {
    config_err("%s<%d>: Internal error: Requested option %s not of enum type!\n",fileName,num,name);
    exit(1);
  }
  return *((ConfigEnum *)opt)->valueRef();
}

int &Config::getInt(const char *fileName,int num,const char *name) const
{
  ConfigOption *opt = m_dict->find(name);
  if (opt==0) 
  {
    config_err("%s<%d>: Internal error: Requested unknown option %s!\n",fileName,num,name);
    exit(1);
  }
  else if (opt->kind()!=ConfigOption::O_Int)
  {
    config_err("%s<%d>: Internal error: Requested option %s not of integer type!\n",fileName,num,name);
    exit(1);
  }
  return *((ConfigInt *)opt)->valueRef();
}

bool &Config::getBool(const char *fileName,int num,const char *name) const
{
  ConfigOption *opt = m_dict->find(name);
  if (opt==0) 
  {
    config_err("%s<%d>: Internal error: Requested unknown option %s!\n",fileName,num,name);
    exit(1);
  }
  else if (opt->kind()!=ConfigOption::O_Bool)
  {
    config_err("%s<%d>: Internal error: Requested option %s not of integer type!\n",fileName,num,name);
    exit(1);
  }
  return *((ConfigBool *)opt)->valueRef();
}

/* -----------------------------------------------------------------
 *
 *	static variables
 */

struct ConfigFileState
{
  int lineNr;
  FILE *filePtr;
  YY_BUFFER_STATE oldState;
  YY_BUFFER_STATE newState;
  QCString fileName;
};  

static const char       *inputString;
static int	         inputPosition;
static int               yyLineNr;
static QCString          yyFileName;
static QCString          tmpString;
static QCString         *s=0;
static bool             *b=0;
static QStrList         *l=0;
static int               lastState;
static QCString          elemStr;
static QCString          includeName;
static QStrList          includePathList;
static QStack<ConfigFileState> includeStack;  
static int               includeDepth;

static QCString     tabSizeString;
static QCString     maxInitLinesString;
static QCString     colsInAlphaIndexString;
static QCString     enumValuesPerLineString;
static QCString     treeViewWidthString;
static QCString     maxDotGraphWidthString;
static QCString     maxDotGraphHeightString;

static Config      *config;

/* -----------------------------------------------------------------
 */
#undef	YY_INPUT
#define	YY_INPUT(buf,result,max_size) result=yyread(buf,max_size);

static int yyread(char *buf,int max_size)
{
    // no file included
    if (includeStack.isEmpty()) 
    {
        int c=0;
	while( c < max_size && inputString[inputPosition] )
	{
	      *buf = inputString[inputPosition++] ;
	      c++; buf++;
  	}
	return c;
    } 
    else 
    {
        //assert(includeStack.current()->newState==YY_CURRENT_BUFFER);
	return fread(buf,1,max_size,includeStack.current()->filePtr);
    }
}


static FILE *tryPath(const char *path,const char *fileName)
{
  QCString absName=(QCString)path+"/"+fileName;
  QFileInfo fi(absName);
  if (fi.exists() && fi.isFile())
  {
    FILE *f=fopen(absName,"r");
    if (!f) config_err("Error: could not open file %s for reading\n",absName.data());
    return f;
  }
  return 0;
}

static void substEnvVarsInStrList(QStrList &sl);
static void substEnvVarsInString(QCString &s);

static FILE *findFile(const char *fileName)
{
  substEnvVarsInStrList(includePathList);
  char *s=includePathList.first();
  while (s) // try each of the include paths
  {
    FILE *f = tryPath(s,fileName);
    if (f) return f;
    s=includePathList.next();
  } 
  // try cwd if includePathList fails
  return tryPath(".",fileName);
}

static void readIncludeFile(const char *incName)
{
  if (includeDepth==MAX_INCLUDE_DEPTH) {
    config_err("Error: maximum include depth (%d) reached, %s is not included. Aborting...\n",
	MAX_INCLUDE_DEPTH,incName);
    exit(1);
  } 

  QCString inc = incName;
  substEnvVarsInString(inc);
  inc = inc.stripWhiteSpace();
  uint incLen = inc.length();
  if (inc.at(0)=='"' && inc.at(incLen-1)=='"') // strip quotes
  {
    inc=inc.mid(1,incLen-2);
  }

  FILE *f;

  //printf("Searching for `%s'\n",incFileName.data());
  if ((f=findFile(inc))) // see if the include file can be found
  {
    // For debugging
#if SHOW_INCLUDES
    for (i=0;i<includeStack.count();i++) msg("  ");
    msg("@INCLUDE = %s: parsing...\n",inc.data());
#endif

    // store the state of the old file 
    ConfigFileState *fs=new ConfigFileState;
    fs->oldState=YY_CURRENT_BUFFER;
    fs->lineNr=yyLineNr;
    fs->fileName=yyFileName;
    fs->filePtr=f;
    // push the state on the stack
    includeStack.push(fs);
    // set the scanner to the include file
    yy_switch_to_buffer(yy_create_buffer(f, YY_BUF_SIZE));
    fs->newState=YY_CURRENT_BUFFER;
    yyFileName=inc;
    includeDepth++;
  } 
  else
  {
    config_err("Error: @INCLUDE = %s: not found!\n",inc.data());
    exit(1);
  }
}


%}

%option noyywrap

%x      Start
%x	SkipComment
%x      SkipInvalid
%x      GetString
%x      GetBool
%x      GetStrList
%x      GetQuotedString
%x      GetEnvVar
%x      Include

%%

<*>\0x0d
<Start,GetString,GetStrList,GetBool,SkipInvalid>"#"	 { BEGIN(SkipComment); }
<Start>[a-z_A-Z][a-z_A-Z0-9]*[ \t]*"="	 { QCString cmd=yytext;
                                           cmd=cmd.left(cmd.length()-1).stripWhiteSpace(); 
					   ConfigOption *option = config->get(cmd);
					   if (option==0) // oops not known
					   {
					     config_err("Warning: ignoring unsupported tag `%s' at line %d, file %s\n",
						 yytext,yyLineNr,yyFileName.data()); 
					     BEGIN(SkipInvalid);
					   }
					   else // known tag
					   {
					     switch(option->kind())
					     {
					       case ConfigOption::O_Info:
						 // shouldn't get here!
					         BEGIN(SkipInvalid);
						 break;
					       case ConfigOption::O_List:
						 l = ((ConfigList *)option)->valueRef();
					         l->clear();
						 elemStr="";
					         BEGIN(GetStrList);
					         break;
					       case ConfigOption::O_Enum:
						 s = ((ConfigEnum *)option)->valueRef();
					         s->resize(0);
					         BEGIN(GetString);
					         break;
					       case ConfigOption::O_String:
						 s = ((ConfigString *)option)->valueRef();
					         s->resize(0);
					         BEGIN(GetString);
					         break;
					       case ConfigOption::O_Int:
						 s = ((ConfigInt *)option)->valueStringRef();
					         s->resize(0);
					         BEGIN(GetString);
					         break;
					       case ConfigOption::O_Bool:
						 s = ((ConfigBool *)option)->valueStringRef();
					         s->resize(0);
					         BEGIN(GetString);
						 break;
					     }
					   }
					}
<Start>[a-z_A-Z][a-z_A-Z0-9]*[ \t]*"+="	{ QCString cmd=yytext;
                                          cmd=cmd.left(cmd.length()-2).stripWhiteSpace(); 
					  ConfigOption *option = config->get(cmd);
					  if (option==0) // oops not known
					  {
					    config_err("Warning: ignoring unsupported tag `%s' at line %d, file %s\n",
						yytext,yyLineNr,yyFileName.data()); 
					    BEGIN(SkipInvalid);
					  }
					  else // known tag
					  {
					    switch(option->kind())
					    {
					      case ConfigOption::O_Info:
					        // shouldn't get here!
					        BEGIN(SkipInvalid);
						break;
					      case ConfigOption::O_List:
					        l = ((ConfigList *)option)->valueRef();
						elemStr="";
					        BEGIN(GetStrList);
					        break;
					      case ConfigOption::O_Enum:
					      case ConfigOption::O_String:
					      case ConfigOption::O_Int:
					      case ConfigOption::O_Bool:
					        config_err("Warning: operator += not supported for `%s'. Ignoring line at line %d, file %s\n",
						    yytext,yyLineNr,yyFileName.data()); 
					        BEGIN(SkipInvalid);
						break;
					     }
					   }
					}
<Start>"@INCLUDE_PATH"[ \t]*"=" 	{ BEGIN(GetStrList); l=&includePathList; l->clear(); elemStr=""; }
  /* include a config file */
<Start>"@INCLUDE"[ \t]*"="     		{ BEGIN(Include);}
<Include>([^ \"\t\r\n]+)|("\""[^\n\"]+"\"") { 
  					  readIncludeFile(yytext); 
  					  BEGIN(Start);
					}
<<EOF>>					{
                                          //printf("End of include file\n");
					  //printf("Include stack depth=%d\n",g_includeStack.count());
                                          if (includeStack.isEmpty())
					  {
					    //printf("Terminating scanner!\n");
					    yyterminate();
					  }
					  else
					  {
					    ConfigFileState *fs=includeStack.pop();
					    fclose(fs->filePtr);
					    YY_BUFFER_STATE oldBuf = YY_CURRENT_BUFFER;
					    yy_switch_to_buffer( fs->oldState );
					    yy_delete_buffer( oldBuf );
					    yyLineNr=fs->lineNr;
					    yyFileName=fs->fileName;
					    delete fs; fs=0;
                                            includeDepth--;
					  }
  					}

<Start>[a-z_A-Z0-9]+			{ config_err("Warning: ignoring unknown tag `%s' at line %d, file %s\n",yytext,yyLineNr,yyFileName.data()); }
<GetString,GetBool,SkipInvalid>\n	{ yyLineNr++; BEGIN(Start); }
<GetStrList>\n				{ 
  					  yyLineNr++; 
					  if (!elemStr.isEmpty())
					  {
					    //printf("elemStr1=`%s'\n",elemStr.data());
					    l->append(elemStr);
					  }
					  BEGIN(Start); 
					}
<GetStrList>[ \t]+			{
  				          if (!elemStr.isEmpty())
					  {
					    //printf("elemStr2=`%s'\n",elemStr.data());
  					    l->append(elemStr);
					  }
					  elemStr.resize(0);
  					}
<GetString>[^ \"\t\r\n]+		{ (*s)+=yytext; }
<GetString,GetStrList,SkipInvalid>"\""	{ lastState=YY_START;
  					  BEGIN(GetQuotedString); 
                                          tmpString.resize(0); 
					}
<GetQuotedString>"\""|"\n" 		{ 
  					  //printf("Quoted String = `%s'\n",tmpString.data());
  					  if (lastState==GetString)
					    (*s)+=tmpString;
					  else
					    elemStr+=tmpString;
					  if (*yytext=='\n')
					  {
					    config_err("Warning: Missing end quote (\") on line %d, file %s\n",yyLineNr,yyFileName.data());
					    yyLineNr++;
					  }
					  BEGIN(lastState);
  					}
<GetQuotedString>"\\\""			{
  					  tmpString+='"';
  					}
<GetQuotedString>.			{ tmpString+=*yytext; }
<GetBool>[a-zA-Z]+			{ 
  					  QCString bs=yytext; 
  					  bs=bs.upper();
  					  if (bs=="YES" || bs=="1")
					    *b=TRUE;
					  else if (bs=="NO" || bs=="0")
					    *b=FALSE;
					  else 
					  {
					    *b=FALSE; 
					    config_warn("Warning: Invalid value `%s' for "
						 "boolean tag in line %d, file %s; use YES or NO\n",
						 bs.data(),yyLineNr,yyFileName.data());
					  }
					}
<GetStrList>[^ \#\"\t\r\n]+		{
  					  elemStr+=yytext;
  					}
<SkipComment>\n				{ yyLineNr++; BEGIN(Start); }
<SkipComment>\\[ \r\t]*\n		{ yyLineNr++; BEGIN(Start); }
<*>\\[ \r\t]*\n				{ yyLineNr++; }
<*>.					
<*>\n					{ yyLineNr++ ; }

%%

/*@ ----------------------------------------------------------------------------
 */

#if 0
static void writeBoolValue(QTextStream &t,bool v)
{
  if (v) t << "YES"; else t << "NO";
}

static void writeIntValue(QTextStream &t,int i)
{
  t << i;
}

static void writeStringValue(QTextStream &t,QCString &s)
{
  const char *p=s.data();
  char c;
  bool hasBlanks=FALSE;
  if (p)
  {
    while ((c=*p++)!=0 && !hasBlanks) hasBlanks = (c==' ' || c=='\n' || c=='\t');
    if (hasBlanks) 
      t << "\"" << s << "\"";
    else
      t << s;
  }
}

static void writeStringList(QTextStream &t,QStrList &l)
{
  const char *p = l.first();
  bool first=TRUE;
  while (p)
  {
    char c;
    const char *s=p;
    bool hasBlanks=FALSE;
    while ((c=*p++)!=0 && !hasBlanks) hasBlanks = (c==' ' || c=='\n' || c=='\t');
    if (!first) t << "                         ";
    first=FALSE;
    if (hasBlanks) t << "\"" << s << "\""; else t << s;
    p = l.next();
    if (p) t << " \\" << endl;
  }
}
#endif

void Config::writeTemplate(QFile *f,bool sl,bool upd)
{
  QTextStream t(f);
  t << "# Doxyfile " << versionString << endl << endl;
  if (!sl)
  {
    t << "# This file describes the settings to be used by the documentation system\n";
    t << "# doxygen (www.doxygen.org) for a project\n";
    t << "#\n";
    t << "# All text after a hash (#) is considered a comment and will be ignored\n";
    t << "# The format is:\n";
    t << "#       TAG = value [value, ...]\n";
    t << "# For lists items can also be appended using:\n";
    t << "#       TAG += value [value, ...]\n";
    t << "# Values that contain spaces should be placed between quotes (\" \")\n";
  }
  ConfigOption *option = m_options->first();
  while (option)
  {
    option->writeTemplate(t,sl,upd);
    option = m_options->next();
  }
}

void Config::convertStrToVal()
{
  ConfigOption *option = m_options->first();
  while (option)
  {
    option->convertStrToVal();
    option = m_options->next();
  }
}

static void substEnvVarsInString(QCString &s)
{
  static QRegExp re("\\$\\([a-z_A-Z0-9]+\\)");
  if (s.isEmpty()) return;
  int p=0;
  int i,l;
  //printf("substEnvVarInString(%s) start\n",s.data());
  while ((i=re.match(s,p,&l))!=-1)
  {
    //printf("Found environment var s.mid(%d,%d)=`%s'\n",i+2,l-3,s.mid(i+2,l-3).data());
    QCString env=getenv(s.mid(i+2,l-3));
    substEnvVarsInString(env); // recursively expand variables if needed.
    s = s.left(i)+env+s.right(s.length()-i-l);
    p=i+env.length(); // next time start at the end of the expanded string
  }
  //printf("substEnvVarInString(%s) end\n",s.data());
}

static void substEnvVarsInStrList(QStrList &sl)
{
  char *s = sl.first();
  while (s)
  {
    QCString result(s);
    bool wasQuoted = (result.find(' ')!=-1) || (result.find('\t')!=-1);
    substEnvVarsInString(result);

    if (!wasQuoted) /* as a result of the expansion, a single string
		       may have expanded into a list, which we'll
		       add to sl. If the orginal string already 
		       contained multiple elements no further 
		       splitting is done to allow quoted items with spaces! */
    {
      int l=result.length();
      int i,p=0;
      // skip spaces
      // search for a "word"
      for (i=0;i<l;i++)
      {
	char c=0;
	// skip until start of new word
	while (i<l && ((c=result.at(i))==' ' || c=='\t')) i++; 
	p=i; // p marks the start index of the word
	// skip until end of a word
	while (i<l && ((c=result.at(i))!=' ' && c!='\t' && c!='"')) i++;
	if (i<l) // not at the end of the string
	{
	  if (c=='"') // word within quotes
	  {
	    p=i+1;
	    for (i++;i<l;i++)
	    {
	      c=result.at(i);
	      if (c=='"') // end quote
	      {
		// replace the string in the list and go to the next item.
		sl.insert(sl.at(),result.mid(p,i-p)); // insert new item before current item.
		sl.next();                 // current item is now the old item
		p=i+1;
		break; 
	      }
	      else if (c=='\\') // skip escaped stuff
	      {
		i++;
	      }
	    }
	  }
	  else if (c==' ' || c=='\t') // separator
	  {
	    // replace the string in the list and go to the next item.
	    sl.insert(sl.at(),result.mid(p,i-p)); // insert new item before current item.
	    sl.next();                 // current item is now the old item
	    p=i+1;
	  }
	}
      }
      if (p!=l) // add the leftover as a string
      {
	// replace the string in the list and go to the next item.
	sl.insert(sl.at(),result.right(l-p)); // insert new item before current item.
	sl.next();                 // current item is now the old item
      }

      // remove the old unexpanded string from the list
      i=sl.at();
      sl.remove(); // current item index changes if the last element is removed.
      if (sl.at()==i)     // not last item
	s = sl.current();
      else                // just removed last item
	s = 0;
    }
    else // just goto the next element in the list
    {
      s=sl.next();
    }
  }
}

void ConfigString::substEnvVars()
{
  substEnvVarsInString(m_value);
}

void ConfigList::substEnvVars()
{
  substEnvVarsInStrList(m_value);
}

void ConfigBool::substEnvVars()
{
  substEnvVarsInString(m_valueString);
}

void ConfigInt::substEnvVars()
{
  substEnvVarsInString(m_valueString);
}

void ConfigEnum::substEnvVars()
{
  substEnvVarsInString(m_value);
}

void Config::substituteEnvironmentVars()
{
  ConfigOption *option = m_options->first();
  while (option)
  {
    option->substEnvVars();
    option = m_options->next();
  }
}

void Config::check()
{
  //if (!projectName.isEmpty())
  //{
  //  projectName[0]=toupper(projectName[0]);
  //}

  QCString &warnFormat = Config_getString("WARN_FORMAT");
  if (warnFormat.isEmpty())
  {
    warnFormat="$file:$line $text";
  }
  else
  {
    if (warnFormat.find("$file")==-1)
    {
      config_err("Error: warning format does not contain a $file tag!\n");
      exit(1);
    }
    if (warnFormat.find("$line")==-1)
    {
      config_err("Error: warning format does not contain a $line tag!\n");
      exit(1);
    }
    if (warnFormat.find("$text")==-1)
    {
      config_err("Error: wanring format foes not contain a $text tag!\n");
      exit(1);
    }
  }

  QCString &manExtension = Config_getString("MAN_EXTENSION");
  
  // set default man page extension if non is given by the user
  if (manExtension.isEmpty())
  {
    manExtension=".3";
  }
  
  QCString &paperType = Config_getEnum("PAPER_TYPE");
  paperType=paperType.lower().stripWhiteSpace(); 
  if (paperType.isEmpty())
  {
    paperType = "a4wide";
  }
  if (paperType!="a4" && paperType!="a4wide" && paperType!="letter" && 
      paperType!="legal" && paperType!="executive")
  {
    config_err("Error: Unknown page type specified");
  }
  
  QCString &outputLanguage=Config_getEnum("OUTPUT_LANGUAGE");
  outputLanguage=outputLanguage.stripWhiteSpace();
  if (outputLanguage.isEmpty())
  {
    outputLanguage = "English";
  }

  QCString &htmlFileExtension=Config_getString("HTML_FILE_EXTENSION");
  htmlFileExtension=htmlFileExtension.stripWhiteSpace();
  if (htmlFileExtension.isEmpty())
  {
    htmlFileExtension = ".html";
  }
  
  // expand the relative stripFromPath values
  QStrList &stripFromPath = Config_getList("STRIP_FROM_PATH");
  char *sfp = stripFromPath.first();
  while (sfp)
  {
    register char *p = sfp;
    if (p)
    {
      char c;
      while ((c=*p))
      {
	if (c=='\\') *p='/';
	p++;
      }
    }
    QCString path = sfp;
    if (path.at(0)!='/' && (path.length()<=2 || path.at(1)!=':'))
    {
      QFileInfo fi(path);
      if (fi.exists() && fi.isDir())
      {
	int i = stripFromPath.at();
	stripFromPath.remove();
	if (stripFromPath.at()==i) // did not remove last item
	  stripFromPath.insert(i,fi.absFilePath()+"/");
	else
	  stripFromPath.append(fi.absFilePath()+"/");
      }
    }
    sfp = stripFromPath.next();
  }
  
  
  // Test to see if HTML header is valid
  QCString &headerFile = Config_getString("HTML_HEADER");
  if (!headerFile.isEmpty())
  {
    QFileInfo fi(headerFile);
    if (!fi.exists())
    {
      config_err("Error: tag HTML_HEADER: header file `%s' "
	  "does not exist\n",headerFile.data());
      exit(1);
    }
  }
  // Test to see if HTML footer is valid
  QCString &footerFile = Config_getString("HTML_FOOTER");
  if (!footerFile.isEmpty())
  {
    QFileInfo fi(footerFile);
    if (!fi.exists())
    {
      config_err("Error: tag HTML_FOOTER: footer file `%s' "
	  "does not exist\n",footerFile.data());
      exit(1);
    }
  }
  // Test to see if LaTeX header is valid
  QCString &latexHeaderFile = Config_getString("LATEX_HEADER");
  if (!latexHeaderFile.isEmpty())
  {
    QFileInfo fi(latexHeaderFile);
    if (!fi.exists())
    {
      config_err("Error: tag LATEX_HEADER: header file `%s' "
	  "does not exist\n",latexHeaderFile.data());
      exit(1);
    }
  }
  // check include path
  QStrList &includePath = Config_getList("INCLUDE_PATH");
  char *s=includePath.first();
  while (s)
  {
    QFileInfo fi(s);
    if (!fi.exists()) config_err("Warning: tag INCLUDE_PATH: include path `%s' "
	                  "does not exist\n",s);
    s=includePath.next();
  }

  // check aliases
  QStrList &aliasList = Config_getList("ALIASES");
  s=aliasList.first();
  while (s)
  {
    QRegExp re("[a-z_A-Z][a-z_A-Z0-9]*[ \t]*=");
    QCString alias=s;
    alias=alias.stripWhiteSpace();
    if (alias.find(re)!=0)
    {
      config_err("Illegal alias format `%s'. Use \"name=value\"\n",alias.data());
    }
    s=aliasList.next();
  }

  // check dot image format
  QCString &dotImageFormat=Config_getEnum("DOT_IMAGE_FORMAT");
  dotImageFormat=dotImageFormat.stripWhiteSpace();
  if (dotImageFormat.isEmpty())
  {
    dotImageFormat = "png";
  }
  else if (dotImageFormat!="gif" && dotImageFormat!="png" && dotImageFormat!="jpg")
  {
    config_err("Invalid value for DOT_IMAGE_FORMAT: `%s'. Using the default.\n",dotImageFormat.data());
    dotImageFormat = "png";
  }
  
  
  // check dot path
  QCString &dotPath = Config_getString("DOT_PATH");
  if (!dotPath.isEmpty())
  {
    if (dotPath.find('\\')!=-1)
    {
      if (dotPath.at(dotPath.length()-1)!='\\')
      {
	dotPath+='\\';
      } 
    } 
    else if (dotPath.find('/')!=-1)
    {
      if (dotPath.at(dotPath.length()-1)!='/')
      {
	dotPath+='/';
      } 
    } 
#if defined(_WIN32)
    QFileInfo dp(dotPath+"dot.exe");
#else
    QFileInfo dp(dotPath+"dot");
#endif
    if (!dp.exists() || !dp.isFile())
    {
      config_err("Warning: the dot tool could not be found at %s\n",dotPath.data());
      dotPath="";
    }
    else
    {
      dotPath=dp.dirPath(TRUE)+"/";
#if defined(_WIN32) // convert slashes
      uint i=0,l=dotPath.length();
      for (i=0;i<l;i++) if (dotPath.at(i)=='/') dotPath.at(i)='\\';
#endif
    }
  }
  else // make sure the string is empty but not null!
  {
    dotPath="";
  }
  
  // check input
  QStrList &inputSources=Config_getList("INPUT");
  if (inputSources.count()==0)
  {
    //config_err("Error: tag INPUT: no input files specified after the INPUT tag.\n");
    //exit(1);
    inputSources.append(QDir::currentDirPath());
    //config_warn("Warning: no files after the INPUT tag, defaulting to the current dir\n");
  }
  else
  {
    s=inputSources.first();
    while (s)
    {
      QFileInfo fi(s);
      if (!fi.exists())
      {
	config_err("Error: tag INPUT: input source `%s' does not exist\n",s);
	exit(1);
      }
      s=inputSources.next();
    }
  }

  // add default pattern if needed
  QStrList &filePatternList = Config_getList("FILE_PATTERNS");
  if (filePatternList.isEmpty())
  {
    filePatternList.append("*.c");
    filePatternList.append("*.cc"); 
    filePatternList.append("*.cxx");
    filePatternList.append("*.cpp");
    filePatternList.append("*.c++");
    filePatternList.append("*.java");
    filePatternList.append("*.ii");
    filePatternList.append("*.ixx");
    filePatternList.append("*.ipp");
    filePatternList.append("*.i++");
    filePatternList.append("*.inl");
    filePatternList.append("*.h");
    filePatternList.append("*.hh");
    filePatternList.append("*.hxx");
    filePatternList.append("*.hpp");
    filePatternList.append("*.h++");
    filePatternList.append("*.idl");
    filePatternList.append("*.odl");
  }

  // add default pattern if needed
  QStrList &examplePatternList = Config_getList("EXAMPLE_PATTERNS");
  if (examplePatternList.isEmpty())
  {
    examplePatternList.append("*");
  }

  // add default pattern if needed
  //QStrList &imagePatternList = Config_getList("IMAGE_PATTERNS");
  //if (imagePatternList.isEmpty())
  //{
  //  imagePatternList.append("*");
  //}
  
  // more checks needed if and only if the search engine is enabled.
  if (Config_getBool("SEARCHENGINE"))
  {
    // check cgi name
    QCString &cgiName = Config_getString("CGI_NAME");
    if (cgiName.isEmpty())
    {
      config_err("Error: tag CGI_NAME: no cgi script name after the CGI_NAME tag.\n");
      exit(1);
    }
    // check cgi URL
    QCString &cgiURL = Config_getString("CGI_URL");
    if (cgiURL.isEmpty())
    {
      config_err("Error: tag CGI_URL: no URL to cgi directory specified.\n");
      exit(1);
    }
    else if (cgiURL.left(7)!="http://" && 
	     cgiURL.left(8)!="https://" &&
	     cgiURL.left(4)!="cgi:"
	    )
    {
      config_err("Error: tag CGI_URL: URL to cgi directory is invalid (must "
	  "start with http:// or https://).\n");
      exit(1);
    }
    // check documentation URL
    QCString &docURL = Config_getString("DOC_URL");
    if (docURL.isEmpty())
    {
      docURL = Config_getString("OUTPUT_DIRECTORY").copy().prepend("file://").append("html");
    }
    else if (docURL.left(7)!="http://" && 
	     docURL.left(8)!="https://" &&
	     docURL.left(7)!="file://"
	    )
    {
      config_err("Error: tag DOC_URL: URL to documentation is invalid or "
	  "not absolute.\n"); 
      exit(1);
    }
    // check absolute documentation path
    QCString &docAbsPath = Config_getString("DOC_ABSPATH");
    if (docAbsPath.isEmpty())
    {
      docAbsPath = Config_getString("OUTPUT_DIRECTORY")+"/html"; 
    }
    else if (docAbsPath[0]!='/' && docAbsPath[1]!=':')
    {
      config_err("Error: tag DOC_ABSPATH: path is not absolute!\n");
      exit(1);
    }
    // check path to doxysearch
    QCString &binAbsPath = Config_getString("BIN_ABSPATH");
    if (binAbsPath.isEmpty())
    {
      config_err("Error: tag BIN_ABSPATH: no absolute path to doxysearch "
	  "specified.\n");
      exit(1);
    }
    else if (binAbsPath[0]!='/' && binAbsPath[1]!=':')
    {
      config_err("Error: tag BIN_ABSPATH: path is not absolute!\n");
      exit(1);
    }

    // check perl path
    bool found=FALSE;
    QCString &perlPath = Config_getString("PERL_PATH");
    if (perlPath.isEmpty())
    {
      QFileInfo fi;
      fi.setFile("/usr/bin/perl");
      if (fi.exists()) 
      {
	perlPath="/usr/bin/perl";
        found=TRUE;
      }
      else
      {
	fi.setFile("/usr/local/bin/perl");
	if (fi.exists())
        {
  	  perlPath="/usr/local/bin/perl";
          found=TRUE;
        }
      }
    }
    if (!found)
    {
      QFileInfo fi(perlPath);
      if (!fi.exists())
      {
        config_warn("Warning: tag PERL_PATH: perl interpreter not found at default or"
            "user specified (%s) location\n",
        perlPath.data());
      }
    }
  }

#undef PUTENV
#undef SEP
#if defined(_WIN32) && !defined(__GNUC__) && (__BORLANDC__ < 0x0550)
#define PUTENV _putenv
#define SEP ";"
#else 
#define PUTENV putenv
#define SEP ":"
#endif
  if (Config_getBool("HAVE_DOT")) 
  {
    char *curFontPath = getenv("DOTFONTPATH");
    int l=curFontPath ? strlen(curFontPath)+1 : 0;
    static char *buf = 0;
    buf = (char *)realloc(buf,strlen("DOTFONTPATH=.")+l+1);
    strcpy(buf,"DOTFONTPATH=.");
    if (l>0)
    {
      strcat(buf,SEP);
      strcat(buf,curFontPath);
    }
    PUTENV(buf);
  }

  int &depth = Config_getInt("MAX_DOT_GRAPH_DEPTH");
  if (depth==0)
  {
    depth=1000;
  }
  
  if (Config_getBool("OPTIMIZE_OUTPUT_JAVA") && Config_getBool("INLINE_INFO"))
  {
    // don't show inline info for Java output, since Java has no inline 
    // concept.
    Config_getBool("INLINE_INFO")=FALSE;
  }
  
}

void Config::init()
{
  ConfigOption *option = m_options->first();
  while (option)
  {
    option->init();
    option = m_options->next();
  }
}

void Config::create()
{
  if (m_initialized) return; 
  m_initialized = TRUE;

  ConfigString *cs;
  ConfigEnum   *ce;
  ConfigList   *cl;
  ConfigInt    *ci;
  ConfigBool   *cb;
  
  // option definitions
  //-----------------------------------------------------------------------------------------------
  addInfo("General","General configuration options");
  //-----------------------------------------------------------------------------------------------
  
  
  cs = addString(
                 "PROJECT_NAME",
                 "The PROJECT_NAME tag is a single word (or a sequence of words surrounded \n"
                 "by quotes) that should identify the project. "
                );
  cs = addString(
                 "PROJECT_NUMBER",
                 "The PROJECT_NUMBER tag can be used to enter a project or revision number. \n"
                 "This could be handy for archiving the generated documentation or \n"
                 "if some version control system is used.\n"
                );
  cs = addString(
                 "OUTPUT_DIRECTORY",
                 "The OUTPUT_DIRECTORY tag is used to specify the (relative or absolute) \n"
                 "base path where the generated documentation will be put. \n"
                 "If a relative path is entered, it will be relative to the location \n"
                 "where doxygen was started. If left blank the current directory will be used.\n"
		);
  cs->setWidgetType(ConfigString::Dir);
  ce = addEnum(
                    "OUTPUT_LANGUAGE",
                    "The OUTPUT_LANGUAGE tag is used to specify the language in which all \n"
                    "documentation generated by doxygen is written. Doxygen will use this \n"
                    "information to generate all constant output in the proper language. \n"
                    "The default language is English, other supported languages are: \n"
		    "Brazilian, Catalan, Chinese, Chinese-Traditional, Croatian, Czech, Danish, Dutch, \n"
		    "Finnish, French, German, Greek, Hungarian, Italian, Japanese, Japanese-en \n"
		    "(Japanese with english messages), Korean, Norwegian, Polish, Portuguese, \n"
		    "Romanian, Russian, Serbian, Slovak, Slovene, Spanish, Swedish and Ukrainian.\n",
		    "English"
                   );
#ifdef LANG_BR
  ce->addValue("Brazilian");
#endif
#ifdef LANG_CA
  ce->addValue("Catalan");
#endif
#ifdef LANG_CN
  ce->addValue("Chinese");
#endif
#ifdef LANG_TW
  ce->addValue("Chinese-Traditional");
#endif
#ifdef LANG_HR
  ce->addValue("Croatian");
#endif
#ifdef LANG_CZ
  ce->addValue("Czech");
#endif
#ifdef LANG_DK
  ce->addValue("Danish");
#endif
#ifdef LANG_NL
  ce->addValue("Dutch");
#endif
  ce->addValue("English");
#ifdef LANG_FI
  ce->addValue("Finnish");
#endif
#ifdef LANG_FR
  ce->addValue("French");
#endif
#ifdef LANG_DE
  ce->addValue("German");
#endif
#ifdef LANG_GR
  ce->addValue("Greek");
#endif
#ifdef LANG_HU
  ce->addValue("Hungarian");
#endif
#ifdef LANG_IT
  ce->addValue("Italian");
#endif
#ifdef LANG_JP
  ce->addValue("Japanese");
  ce->addValue("Japanese-en");
#endif
#ifdef LANG_KR
  ce->addValue("Korean");
#endif
#ifdef LANG_NO
  ce->addValue("Norwegian");
#endif
#ifdef LANG_PL
  ce->addValue("Polish");
#endif
#ifdef LANG_PT
  ce->addValue("Portuguese");
#endif
#ifdef LANG_RO
  ce->addValue("Romanian");
#endif
#ifdef LANG_RU
  ce->addValue("Russian");
#endif
#ifdef LANG_SR
  ce->addValue("Serbian");
#endif
#ifdef LANG_SK
  ce->addValue("Slovak");
#endif
#ifdef LANG_SI
  ce->addValue("Slovene");
#endif
#ifdef LANG_ES
  ce->addValue("Spanish");
#endif
#ifdef LANG_SE
  ce->addValue("Swedish");
#endif
#ifdef LANG_UA
  ce->addValue("Ukrainian");
#endif
  cb = addBool(
                    "USE_WINDOWS_ENCODING",
		    "This tag can be used to specify the encoding used in the generated output. \n"
		    "The encoding is not always determined by the language that is chosen, \n"
		    "but also whether or not the output is meant for Windows or non-Windows users. \n"
		    "In case there is a difference, setting the USE_WINDOWS_ENCODING tag to YES \n"
		    "forces the Windows enconding, (this is the default for the Windows binary), \n"
		    "whereas setting the tag to NO uses a Unix-style encoding (the default for the \n"
		    "all platforms other than Windows).\n",
#if defined(_WIN32) || defined(__CYGWIN__)
		    TRUE
#else
		    FALSE
#endif
		 );
  cb = addBool(
                    "EXTRACT_ALL",
                    "If the EXTRACT_ALL tag is set to YES doxygen will assume all entities in \n"
                    "documentation are documented, even if no documentation was available. \n"
                    "Private class members and static file members will be hidden unless \n"
                    "the EXTRACT_PRIVATE and EXTRACT_STATIC tags are set to YES \n",
                    FALSE
                 );
  cb = addBool(
                    "EXTRACT_PRIVATE",
                    "If the EXTRACT_PRIVATE tag is set to YES all private members of a class \n"
                    "will be included in the documentation. \n",
                    FALSE
                 );
  cb = addBool(
                    "EXTRACT_STATIC",
                    "If the EXTRACT_STATIC tag is set to YES all static members of a file \n"
                    "will be included in the documentation. \n",
                    FALSE
                 );
  cb = addBool(
                    "EXTRACT_LOCAL_CLASSES",
                    "If the EXTRACT_LOCAL_CLASSES tag is set to YES classes (and structs) \n"
		    "defined locally in source files will be included in the documentation. \n"
		    "If set to NO only classes defined in header files are included. \n",
                    TRUE
                 );
  cb = addBool(
                    "HIDE_UNDOC_MEMBERS",
                    "If the HIDE_UNDOC_MEMBERS tag is set to YES, Doxygen will hide all \n"
                    "undocumented members of documented classes, files or namespaces. \n"
                    "If set to NO (the default) these members will be included in the \n"
                    "various overviews, but no documentation section is generated. \n"
                    "This option has no effect if EXTRACT_ALL is enabled. \n",
                    FALSE
                 );
  cb = addBool(
                    "HIDE_UNDOC_CLASSES",
                    "If the HIDE_UNDOC_CLASSES tag is set to YES, Doxygen will hide all \n"
                    "undocumented classes that are normally visible in the class hierarchy. \n"
                    "If set to NO (the default) these class will be included in the various \n"
                    "overviews. This option has no effect if EXTRACT_ALL is enabled. \n",
                    FALSE
                 );
  cb = addBool(
                    "HIDE_FRIEND_COMPOUNDS",
                    "If the HIDE_FRIEND_COMPOUNDS tag is set to YES, Doxygen will hide all \n"
                    "friend (class|struct|union) declarations. \n"
                    "If set to NO (the default) these declarations will be included in the \n"
                    "documentation. \n",
                    FALSE
                 );
  cb = addBool(
                    "HIDE_IN_BODY_DOCS",
                    "If the HIDE_IN_BODY_DOCS tag is set to YES, Doxygen will hide any \n"
                    "documentation blocks found inside the body of a function. \n"
                    "If set to NO (the default) these blocks will be appended to the \n"
                    "function's detailed documentation block. \n",
                    FALSE
                 );
  cb = addBool(
                    "BRIEF_MEMBER_DESC",
                    "If the BRIEF_MEMBER_DESC tag is set to YES (the default) Doxygen will \n"
                    "include brief member descriptions after the members that are listed in \n"
                    "the file and class documentation (similar to JavaDoc). \n"
                    "Set to NO to disable this. \n",
                    TRUE
                 );
  cb = addBool(
                    "REPEAT_BRIEF",
                    "If the REPEAT_BRIEF tag is set to YES (the default) Doxygen will prepend \n"
                    "the brief description of a member or function before the detailed description. \n"
                    "Note: if both HIDE_UNDOC_MEMBERS and BRIEF_MEMBER_DESC are set to NO, the \n"
                    "brief descriptions will be completely suppressed. \n",
                    TRUE
                 );
  cb = addBool(
                    "ALWAYS_DETAILED_SEC",
                    "If the ALWAYS_DETAILED_SEC and REPEAT_BRIEF tags are both set to YES then \n"
                    "Doxygen will generate a detailed section even if there is only a brief \n"
                    "description. \n",
                    FALSE
                 );
  cb = addBool(
                    "INLINE_INHERITED_MEMB",
                    "If the INLINE_INHERITED_MEMB tag is set to YES, doxygen will show all inherited \n"
                    "members of a class in the documentation of that class as if those members were \n"
		    "ordinary class members. Constructors, destructors and assignment operators of \n"
		    "the base classes will not be shown. \n",
		    FALSE
                 );
  cb = addBool(
                    "FULL_PATH_NAMES",
                    "If the FULL_PATH_NAMES tag is set to YES then Doxygen will prepend the full \n"
                    "path before files name in the file list and in the header files. If set \n"
                    "to NO the shortest path that makes the file name unique will be used. \n",
                    FALSE
                 );
  cl = addList(
                    "STRIP_FROM_PATH",
                    "If the FULL_PATH_NAMES tag is set to YES then the STRIP_FROM_PATH tag \n"
                    "can be used to strip a user defined part of the path. Stripping is \n"
                    "only done if one of the specified strings matches the left-hand part of \n"
                    "the path. It is allowed to use relative paths in the argument list.\n"
                 );
  cl->addDependency("FULL_PATH_NAMES");
  cb = addBool(
                    "INTERNAL_DOCS",
                    "The INTERNAL_DOCS tag determines if documentation \n"
                    "that is typed after a \\internal command is included. If the tag is set \n"
                    "to NO (the default) then the documentation will be excluded. \n"
                    "Set it to YES to include the internal documentation. \n",
                    FALSE
                 );
  cb = addBool(
                    "CASE_SENSE_NAMES",
                    "If the CASE_SENSE_NAMES tag is set to NO then Doxygen will only generate \n"
                    "file names in lower case letters. If set to YES upper case letters are also \n"
                    "allowed. This is useful if you have classes or files whose names only differ \n"
                    "in case and if your file system supports case sensitive file names. Windows \n"
                    "users are adviced to set this option to NO.\n",
                    TRUE
                 );
  cb = addBool(
                    "SHORT_NAMES",
		    "If the SHORT_NAMES tag is set to YES, doxygen will generate much shorter \n"
		    "(but less readable) file names. This can be useful is your file systems \n"
		    "doesn't support long names like on DOS, Mac, or CD-ROM. \n",
		    FALSE
                 );
  cb = addBool(
                    "HIDE_SCOPE_NAMES",
                    "If the HIDE_SCOPE_NAMES tag is set to NO (the default) then Doxygen \n"
                    "will show members with their full class and namespace scopes in the \n"
                    "documentation. If set to YES the scope will be hidden. \n",
                    FALSE
                 );
  cb = addBool(
                    "VERBATIM_HEADERS",
                    "If the VERBATIM_HEADERS tag is set to YES (the default) then Doxygen \n"
                    "will generate a verbatim copy of the header file for each class for \n"
                    "which an include is specified. Set to NO to disable this. \n",
                    TRUE
                 );
  cb = addBool(
                    "SHOW_INCLUDE_FILES",
                    "If the SHOW_INCLUDE_FILES tag is set to YES (the default) then Doxygen \n"
                    "will put list of the files that are included by a file in the documentation \n"
                    "of that file. \n",
                    TRUE
                 );
  cb = addBool(
                    "JAVADOC_AUTOBRIEF",
                    "If the JAVADOC_AUTOBRIEF tag is set to YES then Doxygen \n"
                    "will interpret the first line (until the first dot) of a JavaDoc-style \n"
                    "comment as the brief description. If set to NO, the JavaDoc \n"
                    "comments  will behave just like the Qt-style comments (thus requiring an \n"
                    "explict @brief command for a brief description. \n",
                    FALSE
                 );
  cb = addBool(
                    "MULTILINE_CPP_IS_BRIEF",
                    "The MULTILINE_CPP_IS_BRIEF tag can be set to YES to make Doxygen \n"
                    "treat a multi-line C++ special comment block (i.e. a block of //! or /// \n"
		    "comments) as a brief description. This used to be the default behaviour. \n"
		    "The new default is to treat a multi-line C++ comment block as a detailed \n"
		    "description. Set this tag to YES if you prefer the old behaviour instead. \n",
                    FALSE
                 );
  cb = addBool(
                    "DETAILS_AT_TOP",
                    "If the DETAILS_AT_TOP tag is set to YES then Doxygen \n"
                    "will output the detailed description near the top, like JavaDoc.\n"
                    "If set to NO, the detailed description appears after the member \n"
                    "documentation. \n",
                    FALSE
                 );
  cb = addBool(
                    "INHERIT_DOCS",
                    "If the INHERIT_DOCS tag is set to YES (the default) then an undocumented \n"
                    "member inherits the documentation from any documented member that it \n"
                    "reimplements. \n",
                    TRUE
                 );
  cb = addBool(
                    "INLINE_INFO",
                    "If the INLINE_INFO tag is set to YES (the default) then a tag [inline] \n"
                    "is inserted in the documentation for inline members. \n",
                    TRUE
                 );
  cb = addBool(
                    "SORT_MEMBER_DOCS",
                    "If the SORT_MEMBER_DOCS tag is set to YES (the default) then doxygen \n"
                    "will sort the (detailed) documentation of file and class members \n"
                    "alphabetically by member name. If set to NO the members will appear in \n"
                    "declaration order. \n",
                    TRUE
                 );
  cb = addBool(
                    "DISTRIBUTE_GROUP_DOC",
                    "If member grouping is used in the documentation and the DISTRIBUTE_GROUP_DOC \n"
                    "tag is set to YES, then doxygen will reuse the documentation of the first \n"
                    "member in the group (if any) for the other members of the group. By default \n"
                    "all members of a group must be documented explicitly.\n",
                    FALSE
                 );
  ci = addInt(
                    "TAB_SIZE",
                    "The TAB_SIZE tag can be used to set the number of spaces in a tab. \n"
                    "Doxygen uses this value to replace tabs by spaces in code fragments. \n",
                    1,16,8
                );
  cb = addBool(
                    "GENERATE_TODOLIST",
                    "The GENERATE_TODOLIST tag can be used to enable (YES) or \n"
                    "disable (NO) the todo list. This list is created by putting \\todo \n"
                    "commands in the documentation.\n",
                    TRUE
                 );
  cb = addBool(
                    "GENERATE_TESTLIST",
                    "The GENERATE_TESTLIST tag can be used to enable (YES) or \n"
                    "disable (NO) the test list. This list is created by putting \\test \n"
                    "commands in the documentation.\n",
                    TRUE
                 );
  cb = addBool(
                    "GENERATE_BUGLIST",
                    "The GENERATE_BUGLIST tag can be used to enable (YES) or \n"
                    "disable (NO) the bug list. This list is created by putting \\bug \n"
                    "commands in the documentation.\n",
                    TRUE
                 );
  cb = addBool(
                    "GENERATE_DEPRECATEDLIST",
                    "The GENERATE_DEPRECATEDLIST tag can be used to enable (YES) or \n"
                    "disable (NO) the deprecated list. This list is created by putting \n"
		    "\\deprecated commands in the documentation.\n",
                    TRUE
	         );
  cl = addList(
                    "ALIASES",
                    "This tag can be used to specify a number of aliases that acts \n"
                    "as commands in the documentation. An alias has the form \"name=value\". \n"
                    "For example adding \"sideeffect=\\par Side Effects:\\n\" will allow you to \n"
                    "put the command \\sideeffect (or @sideeffect) in the documentation, which \n"
                    "will result in a user defined paragraph with heading \"Side Effects:\". \n"
                    "You can put \\n's in the value part of an alias to insert newlines. \n" 
                 );
  cl = addList(
                    "ENABLED_SECTIONS",
                    "The ENABLED_SECTIONS tag can be used to enable conditional \n"
                    "documentation sections, marked by \\if sectionname ... \\endif. \n"
                 );
  ci = addInt(
                    "MAX_INITIALIZER_LINES",
                    "The MAX_INITIALIZER_LINES tag determines the maximum number of lines \n"
                    "the initial value of a variable or define consist of for it to appear in \n"
                    "the documentation. If the initializer consists of more lines than specified \n"
                    "here it will be hidden. Use a value of 0 to hide initializers completely. \n"
                    "The appearance of the initializer of individual variables and defines in the \n"
                    "documentation can be controlled using \\showinitializer or \\hideinitializer \n"
                    "command in the documentation regardless of this setting. \n",
                    0,10000,30
                 );
  cb = addBool(
                    "OPTIMIZE_OUTPUT_FOR_C",
                    "Set the OPTIMIZE_OUTPUT_FOR_C tag to YES if your project consists of C sources \n"
                    "only. Doxygen will then generate output that is more tailored for C. \n"
                    "For instance some of the names that are used will be different. The list \n"
                    "of all members will be omitted, etc. \n",
                    FALSE
                 );
  cb = addBool(
                    "OPTIMIZE_OUTPUT_JAVA",
                    "Set the OPTIMIZE_OUTPUT_JAVA tag to YES if your project consists of Java sources \n"
                    "only. Doxygen will then generate output that is more tailored for Java. \n"
                    "For instance namespaces will be presented as packages, qualified scopes \n"
                    "will look different, etc. \n",
                    FALSE
                 );
  cb = addBool(
                    "SHOW_USED_FILES",
                    "Set the SHOW_USED_FILES tag to NO to disable the list of files generated \n"
                    "at the bottom of the documentation of classes and structs. If set to YES the \n"
                    "list will mention the files that were used to generate the documentation. \n",
                    TRUE
                );
  //-----------------------------------------------------------------------------------------------
  addInfo(  "Messages","configuration options related to warning and progress messages");
  //-----------------------------------------------------------------------------------------------

  cb = addBool(
                    "QUIET",
                    "The QUIET tag can be used to turn on/off the messages that are generated \n"
                    "by doxygen. Possible values are YES and NO. If left blank NO is used. \n",
                    FALSE
                 );
  cb = addBool(
                    "WARNINGS",
                    "The WARNINGS tag can be used to turn on/off the warning messages that are \n"
                    "generated by doxygen. Possible values are YES and NO. If left blank \n"
                    "NO is used. \n",
                    TRUE
                 );
  cb = addBool(
                    "WARN_IF_UNDOCUMENTED",
                    "If WARN_IF_UNDOCUMENTED is set to YES, then doxygen will generate warnings \n"
                    "for undocumented members. If EXTRACT_ALL is set to YES then this flag will \n"
                    "automatically be disabled. \n",
                    TRUE
                 );
  cb = addBool(
                    "WARN_IF_DOC_ERROR",
                    "If WARN_IF_DOC_ERROR is set to YES, doxygen will generate warnings for \n"
                    "potential errors in the documentation, such as not documenting some \n"
		    "parameters in a documented function, or documenting parameters that \n"
		    "don't exist or using markup commands wrongly. \n",
                    TRUE
                 );
  cs = addString(
                    "WARN_FORMAT",
                    "The WARN_FORMAT tag determines the format of the warning messages that \n"
                    "doxygen can produce. The string should contain the $file, $line, and $text \n"
                    "tags, which will be replaced by the file and line number from which the \n"
                    "warning originated and the warning text. \n"
                   ); 
  cs->setDefaultValue("$file:$line: $text");
  cs = addString(
                    "WARN_LOGFILE",
                    "The WARN_LOGFILE tag can be used to specify a file to which warning \n"
                    "and error messages should be written. If left blank the output is written \n"
                    "to stderr. \n"
                   );
  //-----------------------------------------------------------------------------------------------
  addInfo(  "Input","configuration options related to the input files");
  //-----------------------------------------------------------------------------------------------
  cl = addList(
                    "INPUT",
                    "The INPUT tag can be used to specify the files and/or directories that contain \n"
                    "documented source files. You may enter file names like \"myfile.cpp\" or \n"
                    "directories like \"/usr/src/myproject\". Separate the files or directories \n"
                    "with spaces. \n"
                 );
  cl->setWidgetType(ConfigList::FileAndDir);
  cl = addList(
                    "FILE_PATTERNS",
                    "If the value of the INPUT tag contains directories, you can use the \n"
                    "FILE_PATTERNS tag to specify one or more wildcard pattern (like *.cpp \n"
                    "and *.h) to filter out the source-files in the directories. If left \n"
                    "blank the following patterns are tested: \n"
		    "*.c *.cc *.cxx *.cpp *.c++ *.java *.ii *.ixx *.ipp *.i++ *.inl *.h *.hh *.hxx *.hpp \n"
		    "*.h++ *.idl *.odl\n"
                 );
  cb = addBool(
                    "RECURSIVE",
                    "The RECURSIVE tag can be used to turn specify whether or not subdirectories \n"
                    "should be searched for input files as well. Possible values are YES and NO. \n"
                    "If left blank NO is used. \n",
                    FALSE
                 );
  cl = addList(
                    "EXCLUDE",
                    "The EXCLUDE tag can be used to specify files and/or directories that should \n"
                    "excluded from the INPUT source files. This way you can easily exclude a \n"
                    "subdirectory from a directory tree whose root is specified with the INPUT tag. \n"
                 );
  cb = addBool(
                    "EXCLUDE_SYMLINKS",
                    "The EXCLUDE_SYMLINKS tag can be used select whether or not files or directories \n"
                    "that are symbolic links (a Unix filesystem feature) are excluded from the input. \n",
                    FALSE
                 );
  cl->setWidgetType(ConfigList::FileAndDir);
  cl = addList(
                    "EXCLUDE_PATTERNS",
                    "If the value of the INPUT tag contains directories, you can use the \n"
                    "EXCLUDE_PATTERNS tag to specify one or more wildcard patterns to exclude \n"
                    "certain files from those directories. \n"
                 );
  cl = addList(
                    "EXAMPLE_PATH",
                    "The EXAMPLE_PATH tag can be used to specify one or more files or \n"
                    "directories that contain example code fragments that are included (see \n"
                    "the \\include command). \n"
                 );
  cl->setWidgetType(ConfigList::Dir);
  cl = addList(
                    "EXAMPLE_PATTERNS",
                    "If the value of the EXAMPLE_PATH tag contains directories, you can use the \n"
                    "EXAMPLE_PATTERNS tag to specify one or more wildcard pattern (like *.cpp \n"
                    "and *.h) to filter out the source-files in the directories. If left \n"
                    "blank all files are included. \n"
                 );
  cb = addBool(
                    "EXAMPLE_RECURSIVE",
                    "If the EXAMPLE_RECURSIVE tag is set to YES then subdirectories will be \n"
                    "searched for input files to be used with the \\include or \\dontinclude \n"
                    "commands irrespective of the value of the RECURSIVE tag. \n"
                    "Possible values are YES and NO. If left blank NO is used. \n",
                    FALSE
                 );
  cl = addList(
                    "IMAGE_PATH",
                    "The IMAGE_PATH tag can be used to specify one or more files or \n"
                    "directories that contain image that are included in the documentation (see \n"
                    "the \\image command). \n"
                 );
  cl->setWidgetType(ConfigList::Dir);
  cs = addString(
                    "INPUT_FILTER",
                    "The INPUT_FILTER tag can be used to specify a program that doxygen should \n"
                    "invoke to filter for each input file. Doxygen will invoke the filter program \n"
                    "by executing (via popen()) the command <filter> <input-file>, where <filter> \n"
                    "is the value of the INPUT_FILTER tag, and <input-file> is the name of an \n"
                    "input file. Doxygen will then use the output that the filter program writes \n"
                    "to standard output. \n"
                   );
  cl->setWidgetType(ConfigList::File);
  cb = addBool(
                    "FILTER_SOURCE_FILES",
                    "If the FILTER_SOURCE_FILES tag is set to YES, the input filter (if set using \n"
                    "INPUT_FILTER) will be used to filter the input files when producing source \n"
                    "files to browse (i.e. when SOURCE_BROWSER is set to YES). \n",
                    FALSE
                );    
  //-----------------------------------------------------------------------------------------------
  addInfo(  "Source Browser","configuration options related to source browsing");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "SOURCE_BROWSER",
                    "If the SOURCE_BROWSER tag is set to YES then a list of source files will \n"
                    "be generated. Documented entities will be cross-referenced with these sources. \n",
                    FALSE
                 );
  cb = addBool(
                    "INLINE_SOURCES",
                    "Setting the INLINE_SOURCES tag to YES will include the body \n"
                    "of functions and classes directly in the documentation. \n",
                    FALSE
                 );
  cb = addBool(
                    "STRIP_CODE_COMMENTS",
                    "Setting the STRIP_CODE_COMMENTS tag to YES (the default) will instruct \n"
                    "doxygen to hide any special comment blocks from generated source code \n"
                    "fragments. Normal C and C++ comments will always remain visible. \n",
                    TRUE
                 );
  cb = addBool(     "REFERENCED_BY_RELATION",
                    "If the REFERENCED_BY_RELATION tag is set to YES (the default) \n"
		    "then for each documented function all documented \n"
		    "functions referencing it will be listed. \n",
                    TRUE
              );
  cb->addDependency("SOURCE_BROWSER");
  cb = addBool(     "REFERENCES_RELATION",
                    "If the REFERENCES_RELATION tag is set to YES (the default) \n"
		    "then for each documented function all documented entities \n"
		    "called/used by that function will be listed. \n",
                    TRUE
              );
  cb->addDependency("SOURCE_BROWSER");
  //-----------------------------------------------------------------------------------------------
  addInfo(  "Index","configuration options related to the alphabetical class index");
  //-----------------------------------------------------------------------------------------------
  
  cb = addBool(
                    "ALPHABETICAL_INDEX",
                    "If the ALPHABETICAL_INDEX tag is set to YES, an alphabetical index \n"
                    "of all compounds will be generated. Enable this if the project \n"
                    "contains a lot of classes, structs, unions or interfaces. \n",
                    FALSE
                 );
  ci = addInt(
                    "COLS_IN_ALPHA_INDEX",
                    "If the alphabetical index is enabled (see ALPHABETICAL_INDEX) then \n"
                    "the COLS_IN_ALPHA_INDEX tag can be used to specify the number of columns \n"
                    "in which this list will be split (can be a number in the range [1..20]) \n",
                    1,20,5
                );
  cl = addList(
                    "IGNORE_PREFIX",
                    "In case all classes in a project start with a common prefix, all \n"
                    "classes will be put under the same header in the alphabetical index. \n"
                    "The IGNORE_PREFIX tag can be used to specify one or more prefixes that \n"
                    "should be ignored while generating the index headers. \n"
                 );
  //-----------------------------------------------------------------------------------------------
  addInfo(  "HTML","configuration options related to the HTML output");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "GENERATE_HTML",
                    "If the GENERATE_HTML tag is set to YES (the default) Doxygen will \n"
                    "generate HTML output. \n",
                    TRUE
                 );
  cs = addString(
                    "HTML_OUTPUT",
                    "The HTML_OUTPUT tag is used to specify where the HTML docs will be put. \n"
                    "If a relative path is entered the value of OUTPUT_DIRECTORY will be \n"
                    "put in front of it. If left blank `html' will be used as the default path. \n"
                   );
  cs->setDefaultValue("html");
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("GENERATE_HTML");
  cs = addString(
                    "HTML_FILE_EXTENSION",
                    "The HTML_FILE_EXTENSION tag can be used to specify the file extension for \n"
                    "each generated HTML page (for example: .htm,.php,.asp). If it is left blank \n"
                    "doxygen will generate files with .html extension.\n"
                   );
  cs->setDefaultValue(".html");
  cs->addDependency("GENERATE_HTML");
  cs = addString(
                    "HTML_HEADER",
                    "The HTML_HEADER tag can be used to specify a personal HTML header for \n"
                    "each generated HTML page. If it is left blank doxygen will generate a \n"
                    "standard header.\n"
                   );
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_HTML");
  cs = addString(
                    "HTML_FOOTER",
                    "The HTML_FOOTER tag can be used to specify a personal HTML footer for \n"
                    "each generated HTML page. If it is left blank doxygen will generate a \n"
                    "standard footer.\n"
                   );
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_HTML");
  cs = addString(
                    "HTML_STYLESHEET",
                    "The HTML_STYLESHEET tag can be used to specify a user defined cascading \n"
                    "style sheet that is used by each HTML page. It can be used to \n"
                    "fine-tune the look of the HTML output. If the tag is left blank doxygen \n"
                    "will generate a default style sheet \n"
                   );
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_HTML");
  cb = addBool( 
                    "HTML_ALIGN_MEMBERS",
                    "If the HTML_ALIGN_MEMBERS tag is set to YES, the members of classes, \n"
                    "files or namespaces will be aligned in HTML using tables. If set to \n"
                    "NO a bullet list will be used. \n",
                    TRUE
                 );
  cb->addDependency("GENERATE_HTML");
  cb = addBool(
                    "GENERATE_HTMLHELP",
                    "If the GENERATE_HTMLHELP tag is set to YES, additional index files \n"
                    "will be generated that can be used as input for tools like the \n"
                    "Microsoft HTML help workshop to generate a compressed HTML help file (.chm) \n"
                    "of the generated HTML documentation. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_HTML");
  cs = addString(
                    "CHM_FILE",
                    "If the GENERATE_HTMLHELP tag is set to YES, the CHM_FILE tag can \n"
                    "be used to specify the file name of the resulting .chm file. You \n"
		    "can add a path in front of the file if the result should not be \n"
		    "written to the html output dir. \n"
                   );
  cs->addDependency("GENERATE_HTML");
  cs = addString(
                    "HHC_LOCATION",
                    "If the GENERATE_HTMLHELP tag is set to YES, the HHC_LOCATION tag can \n"
                    "be used to specify the location (absolute path including file name) of \n"
		    "the HTML help compiler (hhc.exe). If non empty doxygen will try to run \n"
		    "the html help compiler on the generated index.hhp.\n"
                   );
  cs->addDependency("GENERATE_HTML");
  cb = addBool(
                    "GENERATE_CHI",
                    "If the GENERATE_HTMLHELP tag is set to YES, the GENERATE_CHI flag \n"
                    "controls if a separate .chi index file is generated (YES) or that \n"
                    "it should be included in the master .chm file (NO).\n",
                    FALSE
                 );
  cb->addDependency("GENERATE_HTML");
  cb = addBool(
                    "BINARY_TOC",
                    "If the GENERATE_HTMLHELP tag is set to YES, the BINARY_TOC flag \n"
                    "controls whether a binary table of contents is generated (YES) or a \n"
                    "normal table of contents (NO) in the .chm file.\n",
                    FALSE
                 );
  cb->addDependency("GENERATE_HTML");
  cb = addBool(
                    "TOC_EXPAND",
                    "The TOC_EXPAND flag can be set to YES to add extra items for group members \n"
                    "to the contents of the Html help documentation and to the tree view. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_HTML");
  cb = addBool(
                    "DISABLE_INDEX",
                    "The DISABLE_INDEX tag can be used to turn on/off the condensed index at \n"
                    "top of each HTML page. The value NO (the default) enables the index and \n"
                    "the value YES disables it. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_HTML");
  ci = addInt(
                    "ENUM_VALUES_PER_LINE",
                    "This tag can be used to set the number of enum values (range [1..20]) \n"
                    "that doxygen will group on one line in the generated HTML documentation. \n",
                    1,20,4
                );
  ci->addDependency("GENERATE_HTML");
  cb = addBool(
                    "GENERATE_TREEVIEW",
                    "If the GENERATE_TREEVIEW tag is set to YES, a side panel will be\n"
                    "generated containing a tree-like index structure (just like the one that \n"
                    "is generated for HTML Help). For this to work a browser that supports \n"
                    "JavaScript, DHTML, CSS and frames is required (for instance Mozilla 1.0+, \n"
		    "Netscape 6.0+, Internet explorer 5.0+, or Konqueror). Windows users are \n"
		    "probably better off using the HTML help feature. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_HTML");
  ci = addInt(
                    "TREEVIEW_WIDTH",
                    "If the treeview is enabled (see GENERATE_TREEVIEW) then this tag can be \n"
                    "used to set the initial width (in pixels) of the frame in which the tree \n"
                    "is shown. \n",
                    0,1500,250
                 );
  ci->addDependency("GENERATE_HTML");

  //-----------------------------------------------------------------------------------------------
  addInfo(  "LaTeX","configuration options related to the LaTeX output");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "GENERATE_LATEX",
                    "If the GENERATE_LATEX tag is set to YES (the default) Doxygen will \n"
                    "generate Latex output. \n",
                    TRUE
                 );
  cs = addString(
                    "LATEX_OUTPUT",
                    "The LATEX_OUTPUT tag is used to specify where the LaTeX docs will be put. \n"
                    "If a relative path is entered the value of OUTPUT_DIRECTORY will be \n"
                    "put in front of it. If left blank `latex' will be used as the default path. \n"
                   );
  cs->setDefaultValue("latex");
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("GENERATE_LATEX");
  cs = addString(
                    "LATEX_CMD_NAME",
                    "The LATEX_CMD_NAME tag can be used to specify the LaTeX command name to be \n"
                    "invoked. If left blank `latex' will be used as the default command name. \n"
                   );
  cs->setDefaultValue("latex");
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_LATEX");
  cs = addString(
                    "MAKEINDEX_CMD_NAME",
                    "The MAKEINDEX_CMD_NAME tag can be used to specify the command name to \n"
		    "generate index for LaTeX. If left blank `makeindex' will be used as the \n"
		    "default command name. \n"
                   );
  cs->setDefaultValue("makeindex");
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_LATEX");
  cb = addBool(
                    "COMPACT_LATEX",
                    "If the COMPACT_LATEX tag is set to YES Doxygen generates more compact \n"
                    "LaTeX documents. This may be useful for small projects and may help to \n"
                    "save some trees in general. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_LATEX");
  ce = addEnum(
                    "PAPER_TYPE",
                    "The PAPER_TYPE tag can be used to set the paper type that is used \n"
                    "by the printer. Possible values are: a4, a4wide, letter, legal and \n"
                    "executive. If left blank a4wide will be used. \n",
                    "a4wide"
                   );
  ce->addValue("a4");
  ce->addValue("a4wide");
  ce->addValue("letter");
  ce->addValue("legal");
  ce->addValue("executive");
  ce->addDependency("GENERATE_LATEX");
  cl = addList(
                    "EXTRA_PACKAGES",
                    "The EXTRA_PACKAGES tag can be to specify one or more names of LaTeX \n"
                    "packages that should be included in the LaTeX output. \n"
                 );
  cl->addDependency("GENERATE_LATEX");
  cs = addString(
                    "LATEX_HEADER",
                    "The LATEX_HEADER tag can be used to specify a personal LaTeX header for \n"
                    "the generated latex document. The header should contain everything until \n"
                    "the first chapter. If it is left blank doxygen will generate a \n"
                    "standard header. Notice: only use this tag if you know what you are doing! \n"
                 );
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_LATEX");
  cb = addBool(
                    "PDF_HYPERLINKS",
                    "If the PDF_HYPERLINKS tag is set to YES, the LaTeX that is generated \n"
                    "is prepared for conversion to pdf (using ps2pdf). The pdf file will \n"
                    "contain links (just like the HTML output) instead of page references \n"
                    "This makes the output suitable for online browsing using a pdf viewer. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_LATEX");
  cb = addBool(
                    "USE_PDFLATEX",
                    "If the USE_PDFLATEX tag is set to YES, pdflatex will be used instead of \n"
                    "plain latex in the generated Makefile. Set this option to YES to get a \n"
                    "higher quality PDF documentation. \n",
                    FALSE
                 );   
  cb->addDependency("GENERATE_LATEX");
  cb = addBool(
                    "LATEX_BATCHMODE",
                    "If the LATEX_BATCHMODE tag is set to YES, doxygen will add the \\\\batchmode. \n"
                    "command to the generated LaTeX files. This will instruct LaTeX to keep \n"
                    "running if errors occur, instead of asking the user for help. \n"
                    "This option is also used when generating formulas in HTML. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_LATEX");
  cb = addBool(
                    "LATEX_HIDE_INDICES",
                    "If LATEX_HIDE_INDICES is set to YES then doxygen will not \n"
                    "include the index chapters (such as File Index, Compound Index, etc.) \n"
                    "in the output. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_LATEX");
  //-----------------------------------------------------------------------------------------------
  addInfo(  "RTF","configuration options related to the RTF output");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "GENERATE_RTF",
                    "If the GENERATE_RTF tag is set to YES Doxygen will generate RTF output \n"
                    "The RTF output is optimised for Word 97 and may not look very pretty with \n"
                    "other RTF readers or editors.\n",
                    FALSE
                 );
  cs = addString(
                    "RTF_OUTPUT",
                    "The RTF_OUTPUT tag is used to specify where the RTF docs will be put. \n"
                    "If a relative path is entered the value of OUTPUT_DIRECTORY will be \n"
                    "put in front of it. If left blank `rtf' will be used as the default path. \n"
                 );
  cs->setDefaultValue("rtf");
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("GENERATE_RTF");
  cb = addBool(
                    "COMPACT_RTF",
                    "If the COMPACT_RTF tag is set to YES Doxygen generates more compact \n"
                    "RTF documents. This may be useful for small projects and may help to \n"
                    "save some trees in general. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_RTF");
  cb = addBool(
                    "RTF_HYPERLINKS",
                    "If the RTF_HYPERLINKS tag is set to YES, the RTF that is generated \n"
                    "will contain hyperlink fields. The RTF file will \n"
                    "contain links (just like the HTML output) instead of page references. \n"
                    "This makes the output suitable for online browsing using WORD or other \n"
                    "programs which support those fields. \n"
                    "Note: wordpad (write) and others do not support links. \n",
                    FALSE
                 );
  cb->addDependency("GENERATE_RTF");
  cs = addString(
                    "RTF_STYLESHEET_FILE",
                    "Load stylesheet definitions from file. Syntax is similar to doxygen's \n"
                    "config file, i.e. a series of assigments. You only have to provide \n"
                    "replacements, missing definitions are set to their default value. \n"
                   );
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_RTF");
  cs = addString(
                    "RTF_EXTENSIONS_FILE",
                    "Set optional variables used in the generation of an rtf document. \n"
                    "Syntax is similar to doxygen's config file.\n"
                );                  
  cs->setWidgetType(ConfigString::File);
  cs->addDependency("GENERATE_RTF");

  //-----------------------------------------------------------------------------------------------
  addInfo(  "Man","configuration options related to the man page output");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "GENERATE_MAN",
                    "If the GENERATE_MAN tag is set to YES (the default) Doxygen will \n"
                    "generate man pages \n",
                    FALSE
                   );
  cs = addString(
                    "MAN_OUTPUT",
                    "The MAN_OUTPUT tag is used to specify where the man pages will be put. \n"
                    "If a relative path is entered the value of OUTPUT_DIRECTORY will be \n"
                    "put in front of it. If left blank `man' will be used as the default path. \n"
                   );
  cs->setDefaultValue("man");
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("GENERATE_MAN");
  cs = addString(
                    "MAN_EXTENSION",
                    "The MAN_EXTENSION tag determines the extension that is added to \n"
                    "the generated man pages (default is the subroutine's section .3) \n"
                   );
  cs->setDefaultValue(".3");
  cs->addDependency("GENERATE_MAN");
  cb = addBool(
                    "MAN_LINKS",
                    "If the MAN_LINKS tag is set to YES and Doxygen generates man output, \n"
                    "then it will generate one additional man file for each entity \n"
                    "documented in the real man page(s). These additional files \n"
                    "only source the real man page, but without them the man command \n"
                    "would be unable to find the correct page. The default is NO. \n",
                    FALSE
                   );
  cb->addDependency("GENERATE_MAN");
  //-----------------------------------------------------------------------------------------------
  addInfo(  "XML","configuration options related to the XML output");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(  
                    "GENERATE_XML",
                    "If the GENERATE_XML tag is set to YES Doxygen will \n"
                    "generate an XML file that captures the structure of \n"
                    "the code including all documentation. Note that this \n"
		    "feature is still experimental and incomplete at the \n"
		    "moment. \n",
		    FALSE
                 );
  cs = addString(
                    "XML_SCHEMA",
		    "The XML_SCHEMA tag can be used to specify an XML schema, \n"
		    "which can be used by a validating XML parser to check the \n"
		    "syntax of the XML files. \n"
                 );
  cs->addDependency("GENERATE_XML");
  cs = addString(
                    "XML_DTD",
		    "The XML_DTD tag can be used to specify an XML DTD, \n"
		    "which can be used by a validating XML parser to check the \n"
		    "syntax of the XML files. \n"
                 );
  cs->addDependency("GENERATE_XML");
  //--------------------------------------------------------------------------
  addInfo(  "DEF","configuration options for the AutoGen Definitions output");
  //--------------------------------------------------------------------------
  cb = addBool("GENERATE_AUTOGEN_DEF",

               "If the GENERATE_AUTOGEN_DEF tag is set to YES Doxygen will \n"
               "generate an AutoGen Definitions (see autogen.sf.net) file \n"
	       "that captures the structure of the code including all \n"
	       "documentation. Note that this feature is still experimental \n"
	       "and incomplete at the moment. \n",

               FALSE );
  //-----------------------------------------------------------------------------------------------
  addInfo(  "PerlMod","configuration options related to the Perl module output");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(  
                    "GENERATE_PERLMOD",
                    "If the GENERATE_PERLMOD tag is set to YES Doxygen will \n"
                    "generate a Perl module file that captures the structure of \n"
                    "the code including all documentation. Note that this \n"
		    "feature is still experimental and incomplete at the \n"
		    "moment. \n",
		    FALSE
                 );
  cb = addBool(  
                    "PERLMOD_LATEX",
                    "If the PERLMOD_LATEX tag is set to YES Doxygen will generate \n"
                    "the necessary Makefile rules, Perl scripts and LaTeX code to be able \n"
		    "to generate PDF and DVI output from the Perl module output. \n",
		    FALSE
                 );
  cb->addDependency("GENERATE_PERLMOD");
  cb = addBool(  
                    "PERLMOD_PRETTY",
                    "If the PERLMOD_PRETTY tag is set to YES the Perl module output will be \n"
		    "nicely formatted so it can be parsed by a human reader.  This is useful \n"
		    "if you want to understand what is going on.  On the other hand, if this \n"
		    "tag is set to NO the size of the Perl module output will be much smaller \n"
		    "and Perl will parse it just the same. \n",
		    TRUE
                 );
  cb->addDependency("GENERATE_PERLMOD");
  cs = addString(  
                    "PERLMOD_MAKEVAR_PREFIX",
                    "The names of the make variables in the generated doxyrules.make file \n"
                    "are prefixed with the string contained in PERLMOD_MAKEVAR_PREFIX. \n"
		    "This is useful so different doxyrules.make files included by the same \n"
		    "Makefile don't overwrite each other's variables."
                 );
  cs->addDependency("GENERATE_PERLMOD");

  //-----------------------------------------------------------------------------------------------
  addInfo(  "Preprocessor","Configuration options related to the preprocessor   ");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "ENABLE_PREPROCESSING",
                    "If the ENABLE_PREPROCESSING tag is set to YES (the default) Doxygen will \n"
                    "evaluate all C-preprocessor directives found in the sources and include \n"
                    "files. \n",
                    TRUE
                 );
  cb = addBool(
                    "MACRO_EXPANSION",
                    "If the MACRO_EXPANSION tag is set to YES Doxygen will expand all macro \n"
                    "names in the source code. If set to NO (the default) only conditional \n"
                    "compilation will be performed. Macro expansion can be done in a controlled \n"
                    "way by setting EXPAND_ONLY_PREDEF to YES. \n",
                    FALSE
                 );
  cb->addDependency("ENABLE_PREPROCESSING");
  cb = addBool(
                    "EXPAND_ONLY_PREDEF",
                    "If the EXPAND_ONLY_PREDEF and MACRO_EXPANSION tags are both set to YES \n"
                    "then the macro expansion is limited to the macros specified with the \n"
                    "PREDEFINED and EXPAND_AS_PREDEFINED tags. \n",
                    FALSE
                 );
  cb->addDependency("ENABLE_PREPROCESSING");
  cb = addBool(
                    "SEARCH_INCLUDES",
                    "If the SEARCH_INCLUDES tag is set to YES (the default) the includes files \n"
                    "in the INCLUDE_PATH (see below) will be search if a #include is found. \n",
                    TRUE
                 );
  cb->addDependency("ENABLE_PREPROCESSING");
  cl = addList(
                    "INCLUDE_PATH",
                    "The INCLUDE_PATH tag can be used to specify one or more directories that \n"
                    "contain include files that are not input files but should be processed by \n"
                    "the preprocessor. \n"
                 );
  cl->setWidgetType(ConfigList::Dir);
  cl->addDependency("ENABLE_PREPROCESSING");
  cl = addList(
                    "INCLUDE_FILE_PATTERNS",
                    "You can use the INCLUDE_FILE_PATTERNS tag to specify one or more wildcard \n"
                    "patterns (like *.h and *.hpp) to filter out the header-files in the \n"
                    "directories. If left blank, the patterns specified with FILE_PATTERNS will \n"
                    "be used. \n"
                 );
  cl->addDependency("ENABLE_PREPROCESSING");
  cl = addList(
                    "PREDEFINED",
                    "The PREDEFINED tag can be used to specify one or more macro names that \n"
                    "are defined before the preprocessor is started (similar to the -D option of \n"
                    "gcc). The argument of the tag is a list of macros of the form: name \n"
                    "or name=definition (no spaces). If the definition and the = are \n"
                    "omitted =1 is assumed. \n"
                 );
  cl->addDependency("ENABLE_PREPROCESSING");
  cl = addList(
                    "EXPAND_AS_DEFINED",
                    "If the MACRO_EXPANSION and EXPAND_ONLY_PREDEF tags are set to YES then \n"
                    "this tag can be used to specify a list of macro names that should be expanded. \n"
                    "The macro definition that is found in the sources will be used. \n"
                    "Use the PREDEFINED tag if you want to use a different macro definition. \n" 
                 );
  cl->addDependency("ENABLE_PREPROCESSING");
  cb = addBool(
                    "SKIP_FUNCTION_MACROS",
                    "If the SKIP_FUNCTION_MACROS tag is set to YES (the default) then \n"
		    "doxygen's preprocessor will remove all function-like macros that are alone \n"
		    "on a line, have an all uppercase name, and do not end with a semicolon. Such \n"
		    "function macros are typically used for boiler-plate code, and will confuse the \n"
		    "parser if not removed. \n",
                    TRUE
                 );
  cb->addDependency("ENABLE_PREPROCESSING");
  //-----------------------------------------------------------------------------------------------
  addInfo(  "External","Configuration::addtions related to external references   ");
  //-----------------------------------------------------------------------------------------------
  cl = addList(
                    "TAGFILES",
                    "The TAGFILES option can be used to specify one or more tagfiles. \n"
		    "Optionally an initial location of the external documentation \n"
		    "can be added for each tagfile. The format of a tag file without \n"
		    "this location is as follows: \n"
		    "  TAGFILES = file1 file2 ... \n" 
		    "Adding location for the tag files is done as follows: \n"
		    "  TAGFILES = file1=loc1 \"file2 = loc2\" ... \n"
	            "where \"loc1\" and \"loc2\" can be relative or absolute paths or \n"
		    "URLs. If a location is present for each tag, the installdox tool \n"
		    "does not have to be run to correct the links.\n"
		    "Note that each tag file must have a unique name\n"
		    "(where the name does NOT include the path)\n"
	            "If a tag file is not located in the directory in which doxygen \n"
	            "is run, you must also specify the path to the tagfile here. \n"
               );
  cl->setWidgetType(ConfigList::File);
  cs = addString(
                    "GENERATE_TAGFILE",
                    "When a file name is specified after GENERATE_TAGFILE, doxygen will create \n"
                    "a tag file that is based on the input files it reads. \n"
                   );
  cs->setWidgetType(ConfigString::File);
  cb = addBool(
                    "ALLEXTERNALS",
                    "If the ALLEXTERNALS tag is set to YES all external classes will be listed \n"
                    "in the class index. If set to NO only the inherited external classes \n"
                    "will be listed. \n",
                    FALSE
                 );
  cb = addBool(
                    "EXTERNAL_GROUPS",
                    "If the EXTERNAL_GROUPS tag is set to YES all external groups will be listed \n"
                    "in the modules index. If set to NO, only the current project's groups will \n"
                    "be listed. \n",
                    TRUE
                 );
  cs = addString(
                    "PERL_PATH",
                    "The PERL_PATH should be the absolute path and name of the perl script \n"
                    "interpreter (i.e. the result of `which perl'). \n"
                   );
  cs->setDefaultValue("/usr/bin/perl");
  cs->setWidgetType(ConfigString::Dir);

  //-----------------------------------------------------------------------------------------------
  addInfo(  "Dot","Configuration options related to the dot tool   ");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "CLASS_DIAGRAMS",
                    "If the CLASS_DIAGRAMS tag is set to YES (the default) Doxygen will \n"
                    "generate a inheritance diagram (in Html, RTF and LaTeX) for classes with base or \n"
                    "super classes. Setting the tag to NO turns the diagrams off. Note that this \n"
		    "option is superceded by the HAVE_DOT option below. This is only a fallback. It is \n"
		    "recommended to install and use dot, since it yield more powerful graphs. \n",
                    TRUE
                 );
  cb = addBool(
                    "HIDE_UNDOC_RELATIONS",
		    "If set to YES, the inheritance and collaboration graphs will hide \n"
		    "inheritance and usage relations if the target is undocumented \n"
		    "or is not a class. \n",
                    TRUE
                 );
  cb = addBool(
                    "HAVE_DOT",
                    "If you set the HAVE_DOT tag to YES then doxygen will assume the dot tool is \n"
                    "available from the path. This tool is part of Graphviz, a graph visualization \n"
                    "toolkit from AT&T and Lucent Bell Labs. The other options in this section \n"
                    "have no effect if this option is set to NO (the default) \n",
                    FALSE
                 );
  cb = addBool(
                    "CLASS_GRAPH",
                    "If the CLASS_GRAPH and HAVE_DOT tags are set to YES then doxygen \n"
                    "will generate a graph for each documented class showing the direct and \n"
                    "indirect inheritance relations. Setting this tag to YES will force the \n"
                    "the CLASS_DIAGRAMS tag to NO.\n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  cb = addBool(
                    "COLLABORATION_GRAPH",
                    "If the COLLABORATION_GRAPH and HAVE_DOT tags are set to YES then doxygen \n"
                    "will generate a graph for each documented class showing the direct and \n"
                    "indirect implementation dependencies (inheritance, containment, and \n"
                    "class references variables) of the class with other documented classes. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  cb = addBool(
                    "TEMPLATE_RELATIONS",
		    "If set to YES, the inheritance and collaboration graphs will show the \n"
		    "relations between templates and their instances. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  cb = addBool(
                    "INCLUDE_GRAPH",
                    "If the ENABLE_PREPROCESSING, SEARCH_INCLUDES, INCLUDE_GRAPH, and HAVE_DOT \n"
                    "tags are set to YES then doxygen will generate a graph for each documented \n"
                    "file showing the direct and indirect include dependencies of the file with \n"
                    "other documented files. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  cb = addBool(
                    "INCLUDED_BY_GRAPH",
                    "If the ENABLE_PREPROCESSING, SEARCH_INCLUDES, INCLUDED_BY_GRAPH, and \n"
                    "HAVE_DOT tags are set to YES then doxygen will generate a graph for each \n"
                    "documented header file showing the documented files that directly or \n"
                    "indirectly include this file. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  cb = addBool(
                    "GRAPHICAL_HIERARCHY",
                    "If the GRAPHICAL_HIERARCHY and HAVE_DOT tags are set to YES then doxygen \n"
                    "will graphical hierarchy of all classes instead of a textual one. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  ce = addEnum(
                    "DOT_IMAGE_FORMAT",
                    "The DOT_IMAGE_FORMAT tag can be used to set the image format of the images \n"
		    "generated by dot. Possible values are png, jpg, or gif\n"
                    "If left blank png will be used. \n",
                    "png"
                   );
  ce->addValue("png");
  ce->addValue("jpg");
  ce->addValue("gif");
  ce->addDependency("HAVE_DOT");
  cs = addString(
                    "DOT_PATH",
                    "The tag DOT_PATH can be used to specify the path where the dot tool can be \n"
                    "found. If left blank, it is assumed the dot tool can be found on the path. \n"
                   );
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("HAVE_DOT");
  cl = addList(
                    "DOTFILE_DIRS",
                    "The DOTFILE_DIRS tag can be used to specify one or more directories that \n"
		    "contain dot files that are included in the documentation (see the \n"
                    "\\dotfile command). \n"
                 );
  cl->setWidgetType(ConfigList::Dir);
  cl->addDependency("HAVE_DOT");
  ci = addInt(
                    "MAX_DOT_GRAPH_WIDTH",
                    "The MAX_DOT_GRAPH_WIDTH tag can be used to set the maximum allowed width \n"
                    "(in pixels) of the graphs generated by dot. If a graph becomes larger than \n"
                    "this value, doxygen will try to truncate the graph, so that it fits within \n"
                    "the specified constraint. Beware that most browsers cannot cope with very \n"
                    "large images. \n",
                    100,30000, 1024
                );
  ci->addDependency("HAVE_DOT");
  ci = addInt(
                    "MAX_DOT_GRAPH_HEIGHT",
                    "The MAX_DOT_GRAPH_HEIGHT tag can be used to set the maximum allows height \n"
                    "(in pixels) of the graphs generated by dot. If a graph becomes larger than \n"
                    "this value, doxygen will try to truncate the graph, so that it fits within \n"
                    "the specified constraint. Beware that most browsers cannot cope with very \n"
                    "large images. \n",
                    100,30000,1024
                );
  ci = addInt(
                    "MAX_DOT_GRAPH_DEPTH",
                    "The MAX_DOT_GRAPH_DEPTH tag can be used to set the maximum depth of the \n"
                    "graphs generated by dot. A depth value of 3 means that only nodes reachable \n"
                    "from the root by following a path via at most 3 edges will be shown. Nodes that \n"
		    "lay further from the root node will be omitted. Note that setting this option to \n"
		    "1 or 2 may greatly reduce the computation time needed for large code bases. Also \n"
		    "note that a graph may be further truncated if the graph's image dimensions are \n"
		    "not sufficient to fit the graph (see MAX_DOT_GRAPH_WIDTH and MAX_DOT_GRAPH_HEIGHT). \n"
		    "If 0 is used fot the depth value (the default), the graph is not depth constraint. \n",
                    0,1000,0
                );
  ci->addDependency("HAVE_DOT");
  cb = addBool(
                    "GENERATE_LEGEND",
                    "If the GENERATE_LEGEND tag is set to YES (the default) Doxygen will \n"
                    "generate a legend page explaining the meaning of the various boxes and \n"
                    "arrows in the dot generated graphs. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");
  cb = addBool(
                    "DOT_CLEANUP",
                    "If the DOT_CLEANUP tag is set to YES (the default) Doxygen will \n"
                    "remove the intermedate dot files that are used to generate \n"
                    "the various graphs. \n",
                    TRUE
                 );
  cb->addDependency("HAVE_DOT");

  //-----------------------------------------------------------------------------------------------
  addInfo(  "Search","Configuration::addtions related to the search engine   ");
  //-----------------------------------------------------------------------------------------------
  cb = addBool(
                    "SEARCHENGINE",
                    "The SEARCHENGINE tag specifies whether or not a search engine should be \n"
                    "used. If set to NO the values of all tags below this one will be ignored. \n",
                    FALSE
                 );
  cs = addString(
                    "CGI_NAME",
                    "The CGI_NAME tag should be the name of the CGI script that \n"
                    "starts the search engine (doxysearch) with the correct parameters. \n"
                    "A script with this name will be generated by doxygen. \n"
                   );
  cs->setDefaultValue("search.cgi");
  cs->addDependency("SEARCHENGINE");
  cs = addString(
                    "CGI_URL",
                    "The CGI_URL tag should be the absolute URL to the directory where the \n"
                    "cgi binaries are located. See the documentation of your http daemon for \n"
                    "details. \n"
                   );
  cs->addDependency("SEARCHENGINE");
  cs = addString(
                    "DOC_URL",
                    "The DOC_URL tag should be the absolute URL to the directory where the \n"
                    "documentation is located. If left blank the absolute path to the \n"
                    "documentation, with file:// prepended to it, will be used. \n"
                   );
  cs->addDependency("SEARCHENGINE");
  cs = addString(
                    "DOC_ABSPATH",
                    "The DOC_ABSPATH tag should be the absolute path to the directory where the \n"
                    "documentation is located. If left blank the directory on the local machine \n"
                    "will be used. \n"
                   );
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("SEARCHENGINE");
  cs = addString(
                    "BIN_ABSPATH",
                    "The BIN_ABSPATH tag must point to the directory where the doxysearch binary \n"
                    "is installed. \n"
                   );
  cs->setDefaultValue("/usr/local/bin/");
  cs->setWidgetType(ConfigString::Dir);
  cs->addDependency("SEARCHENGINE");
  cl = addList(
                    "EXT_DOC_PATHS",
                    "The EXT_DOC_PATHS tag can be used to specify one or more paths to \n"
                    "documentation generated for other projects. This allows doxysearch to search \n"
                    "the documentation for these projects as well. \n"
                 );
  cl->setWidgetType(ConfigList::Dir);
  cl->addDependency("SEARCHENGINE");

  // The IMAGE_PATTERNS tag is now officially obsolete.
}

static QCString configFileToString(const char *name)
{
  if (name==0 || name[0]==0) return 0;
  QFile f;

  bool fileOpened=FALSE;
  if (name[0]=='-' && name[1]==0) // read from stdin
  {
    fileOpened=f.open(IO_ReadOnly,stdin);
    if (fileOpened)
    {
      const int bSize=4096;
      QCString contents(bSize);
      int totalSize=0;
      int size;
      while ((size=f.readBlock(contents.data()+totalSize,bSize))==bSize)
      {
        totalSize+=bSize;
        contents.resize(totalSize+bSize); 
      }
      totalSize+=size+2;
      contents.resize(totalSize);
      contents.at(totalSize-2)='\n'; // to help the scanner
      contents.at(totalSize-1)='\0';
      return contents;
    }
  }
  else // read from file
  {
    QFileInfo fi(name);
    if (!fi.exists() || !fi.isFile())
    {
      config_err("Error: file `%s' not found\n",name);
      return "";
    }
      f.setName(name);
      fileOpened=f.open(IO_ReadOnly);
      if (fileOpened)
      {
        int fsize=f.size();
        QCString contents(fsize+2);
        f.readBlock(contents.data(),fsize);
        if (fsize==0 || contents[fsize-1]=='\n') 
          contents[fsize]='\0';
        else
          contents[fsize]='\n'; // to help the scanner
        contents[fsize+1]='\0';
        f.close();
        return contents;
      }
  }
  if (!fileOpened)  
  {
    config_err("Error: cannot open file `%s' for reading\n",name);
  }
  return "";
}


bool Config::parse(const char *fn)
{
  QCString contents = configFileToString(fn); 
  config = Config::instance();
  inputString   = contents.data();
  inputPosition = 0;
  yyLineNr      = 1;
  yyFileName=fn;
  includeStack.setAutoDelete(TRUE);
  includeStack.clear();
  includeDepth  = 0;
  configYYrestart( configYYin );
  BEGIN( Start );
  configYYlex();
  inputString = 0;
  return TRUE;
}

//extern "C" { // some bogus code to keep the compiler happy
//  int  configYYwrap() { return 1 ; }
//}
