/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.1, (the "License"); you may not use this file except in
 * compliance with the License. You should have received a copy of the
 * Erlang Public License along with this software. If not, it can be
 * retrieved via the world wide web at http://www.erlang.org/.
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Original Code is Ericsson Utvecklings AB.
 * Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
 * AB. All Rights Reserved.''
 * 
 *     $Id$
 */
/* This File contains functions which are called if a user hits ^C */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "sys.h"
#include "erl_vm.h"
#include "global.h"
#include "erl_process.h"
#include "version.h"
#include "error.h"
#include "version.h"
#include "erl_db.h"
#include "bif.h"
#include "erl_version.h"

/* Forward declarations */
static FUNCTION(void, process_killer, (_VOID_));
FUNCTION(void, do_break, (_VOID_));
FUNCTION(void, erl_crash_dump, (char *, int, char *, va_list));
#ifdef DEBUG
static FUNCTION(void, bin_check, (_VOID_));
#endif

static FUNCTION(void, print_garb_info, (Process* p, CIO to));
#ifdef OPPROF
static void dump_frequencies(void);
#endif

static void message_info(to)
CIO to;
{
}

static void port_info(to)
CIO to;
{
    int i;
    erl_printf(to,"\nPort Information\n");
    erl_printf(to,"--------------------------------------------------\n");
    for (i = 0; i < erl_max_ports; i++)
	print_port_info(i,to);
}

/* display information for all processes */

void process_info(to)
CIO to;
{
    int i;
    erl_printf(to,"\nProcess Information\n");
    erl_printf(to,"--------------------------------------------------\n");
    for (i = 0; i < max_process; i++) {
	if ((process_tab[i] != NULL) && (process_tab[i]->i != ENULL)) {
	   if (process_tab[i]->status != P_EXITING)
	      print_process_info(process_tab[i],to);
	}
    }

    process_info_zombies(to);

    port_info(to);
}

static void process_killer()
{
    int i,j;
    Process* rp;

    erl_printf(COUT,"\n\nProcess Information\n\n");
    erl_printf(COUT,"--------------------------------------------------\n");
    for (i = max_process-1; i >= 0; i--) {
	if (((rp = process_tab[i]) != NULL) && rp->i != ENULL) {
	    int br;
	    print_process_info(rp,COUT);
	    erl_printf(COUT,"(k)ill (n)ext (r)eturn:\n");
	    while(1) {
		if ((j = sys_get_key(0)) <= 0)
		    halt_0(0);
		switch(j) {
		case 'k':
		    if (rp->status == P_WAITING) 
			schedule_exit(rp, am_killed);
		    else
			erl_printf(COUT,"Can only kill WAITING processes this way\n");

		case 'n': br = 1; break;
		case 'r': return;
		default: return;
		}
		if (br == 1) break;
	    }
	}
    }
}
			       

static void display_ref(Ref *ref, CIO to)
{
    /* Kludge */
    display(make_ref(ref), to);
}

/* Display info about an individual Erlang process */
void print_process_info(p,to)
Process *p; CIO to;
{
    int garbing = 0;

    /* display the PID */
    display(p->id,to);
    /* Display the status */
    switch (p->status) {
    case P_FREE:
	erl_printf(to," Non Existing."); /* Should never happen */
	break;
    case P_RUNABLE:
	erl_printf(to," Scheduled.");
	break;
    case P_WAITING:
	erl_printf(to," Waiting.");
	break;
    case P_SUSPENDED:
	erl_printf(to," Suspended.");
	break;
    case P_RUNNING:
	erl_printf(to," Running.");
	break;
    case P_EXITING:
	erl_printf(to," Exiting.");
	break;
    case P_GARBING:
	erl_printf(to," Process is garbing, limited information.");
	garbing = 1;
	break;
    }

    /*
     * If the process is registered as a global process, display the
     * registered name
     */
    if (p->reg != NULL) {
	erl_printf(to," Registered as: ");
	print_atom(p->reg->name, to);
    }

    /*
     * Display the initial function name
     */
    erl_printf(to,"\nSpawned as: ");
    display(p->initial[INITIAL_MOD], to);
    erl_printf(to,":");
    display(p->initial[INITIAL_FUN], to);
    erl_printf(to,"/");
    erl_printf(to,"%d\n", (int)p->initial[INITIAL_ARI]);

    if (p->current != NULL) {
	erl_printf(to, "Current call: ");
	display(p->current[0], to);
	erl_printf(to, ":");
	display(p->current[1], to);
	erl_printf(to,"/%d\n", p->current[2]);
    }

    /* display the message queue only if there is anything in it */
    if (p->msg.first != NULL && !garbing) {
	ErlMessage* mp = p->msg.first;
	int n = 0;

	while (mp != NULL) {
	   n++;
	   mp = mp->next;
	}

	erl_printf(to,"Message queue (%d message%s): [",
		   n, n == 1 ? "" : "s");

	mp = p->msg.first;
	while (mp != NULL) {
	    display(mp->mesg, to);
	    if ((mp = mp->next) != NULL)
		erl_printf(to, ",");
	}
	erl_printf(to, "]\n");
    }

    {
       long s = 0;
       ErlHeapFragment *m = p->mbuf;
       while (m != NULL) {
	  s += m->size;
	  m = m->next;
       }
       erl_printf(to, "Message buffer data: %d words\n", s);
    }

    if (p->ct != NULL) {
       int i, j;

       erl_printf(to, "Last calls:\n");
	  for (i = 0; i < p->ct->n; i++)
	  {
	     erl_printf(to, "  ");
	     j = p->ct->cur - i - 1;
	     if (j < 0)
		j += p->ct->len;
	     if (p->ct->ct[j] == &exp_send)
		erl_printf(to, "send");
	     else if (p->ct->ct[j] == &exp_receive)
		erl_printf(to, "'receive'");
	     else if (p->ct->ct[j] == &exp_timeout)
		   erl_printf(to, "timeout");
	     else {
		display(p->ct->ct[j]->code[0], to);
		erl_printf(to, ":");
		display(p->ct->ct[j]->code[1], to);
		erl_printf(to, "/%d", p->ct->ct[j]->code[2]);
	     }
	     erl_printf(to, "\n");
	  }
    }

    /* display the links only if there are any*/
    if (p->links != NULL) {
	ErlLink* lnk = p->links;
	erl_printf(to,"Link list: [");
	while(lnk != NULL) {
	    if (lnk->type == LNK_LINK1) {
	       erl_printf(to,"{");
	       if (lnk->item == p->id) {
		  erl_printf(to,"to,");
		  display(lnk->data, to);
	       } else {
		  erl_printf(to,"from,");
		  display(lnk->item, to);
	       }
	       erl_printf(to,",");
	       display_ref(&lnk->ref, to);
	       erl_printf(to,"}");
	    } else {
	       display(lnk->item, to);
	    }
	    if ((lnk = lnk->next) != NULL)
	       erl_printf(to, ",");
	}
	erl_printf(to,"]\n");
    }

    /* and the dictionary */
    if (p->dictionary != NULL && !garbing) {
	erl_printf(to, "Dictionary: ");
	dictionary_dump(p->dictionary, to);
	erl_printf(to, "\n");
    }
    
    /* as well as the debug dictionary */
    if (p->debug_dictionary != NULL && !garbing) {
	erl_printf(to, "$Dictionary: ");
	dictionary_dump(p->debug_dictionary, to);
	erl_printf(to, "\n");
    }
    
    /* print the number of reductions etc */
    erl_printf(to,"Reductions %d stack+heap %d old_heap_sz=%d \n",
	       p->reds, p->heap_sz,
	       (p->old_heap == NULL) ? 0 : 
	       p->old_hend - p->old_heap );
    erl_printf(to,"Heap unused=%d OldHeap unused=%d\n",
	       p->stop - p->htop, 
	       (p->old_heap == NULL) ? 0 : 
	       p->old_hend - p->old_heap);

    if (garbing) {
	print_garb_info(p, to);
    }
    
    erl_printf(to, "Stack dump:\n");
    stack_dump2(p, to);

    erl_printf(to,"--------------------------------------------------\n");
}

static void
print_garb_info(p, to)
Process* p;
CIO to;
{
    erl_printf(to, "new heap: %-8s %-8s %-8s %-8s\n",
	       "start", "top", "sp", "end");
    erl_printf(to, "          %08X %08X %08X %08X\n",
	       p->heap, p->htop, p->stop, p->hend);
    erl_printf(to, "old heap: %-8s %-8s %-8s\n",
	       "start", "top", "end");
    erl_printf(to, "          %08X %08X %08X\n",
	       p->old_heap, p->old_htop, p->old_hend);
}

void info(to)
CIO to;
{
    erl_printf(to,"--------------------------------------------------\n");
    atom_info(to);
    module_info(to);
    export_info(to);
    register_info(to);
    erl_printf(to,"Allocated binary %d\n",tot_bin_allocated);
    erl_printf(to,"Allocated by process_desc %d\n", fix_info(process_desc));
    erl_printf(to,"Allocated by table_desc %d\n",fix_info(table_desc));
    erl_printf(to,"Allocated by link_desc %d\n",fix_info(link_desc));
    erl_printf(to,"Allocated by atom_desc %d\n",fix_info(atom_desc));
    erl_printf(to,"Allocated by export_desc %d\n",fix_info(export_desc));
    erl_printf(to,"Allocated by module_desc %d\n",fix_info(module_desc));
    erl_printf(to,"Allocated by preg_desc %d\n",fix_info(preg_desc));

#ifdef DEBUG
    erl_printf(to,"Allocated by SYSTEM %d\n",tot_allocated);
#endif
    erl_printf(to,"--------------------------------------------------\n");
}

void loaded(to)
CIO to;
{
    int i, old = 0, cur = 0;
    erl_printf(to,"--------------------------------------------------\n");
    for (i = 0; i < module_code_size; i++) {
	if (module_code(i) != NULL &&
	    ((module_code(i)->code_length != 0) ||
	     (module_code(i)->old_code_length != 0))) {
	    print_atom(module_code(i)->module, to);
	    cur += module_code(i)->code_length;
	    erl_printf(to," %d", module_code(i)->code_length );
	    if (module_code(i)->old_code_length != 0) {
		erl_printf(to," (%d old)", module_code(i)->old_code_length );
		old += module_code(i)->old_code_length;
	    }
	    erl_printf(to,"\n");
	}
    }
    erl_printf(to,"\nTotals. Current code = %d Old code = %d\n", cur, old);
    erl_printf(to,"--------------------------------------------------\n");
}


/* break handler */
/* Used in sys.c */
void do_break()
{
    int i;
    erl_printf(COUT, "\nBREAK: (a)bort (c)ontinue (p)roc info (i)nfo (l)oaded\n");
    erl_printf(COUT, "       (v)ersion (k)ill (D)b-tables (d)istribution\n");
    while (1) {
	if ((i = sys_get_key(0)) <= 0)
	    halt_0(0);
	switch (i) {
	case 'q':
	case 'a': 
	case '*': /* 
		   * The asterisk is an read error on windows, 
		   * where sys_get_key isn't that great in console mode.
		   * The usual reason for a read error is Ctrl-C. Treat this as
		   * 'a' to avoid infinite loop.
		   */
	    halt_0(0);
	case 'c':
	    return;
	case 'p':
	    process_info(COUT);
	    return;
	case 'm':
	    message_info(COUT);
	    return;
	case 'o':
	    port_info(COUT);
	    return;
	case 'i':
	    info(COUT);
	    return;
	case 'l':
	    loaded(COUT);
	    return;
	case 'v':
	    erl_printf(COUT, "Erlang (%s) emulator version "
		       ERLANG_VERSION "\n",
		       EMULATOR);
	    erl_printf(COUT, "Compiled on " ERLANG_COMPILE_DATE "\n");
	    return;
	case 'd':
	    distribution_info(COUT);
	    return;
	case 'D':
	    db_info(CERR, 1);
	    return; 
	case 'k':
	    process_killer();
	    return;
#ifdef OPPROF
	case 'X':
	    dump_frequencies();
	    return;
	case 'x':
	    {
		int i;
		for (i = 0; i <= HIGHEST_OP; i++) {
		    if (opc[i].name != NULL) {
			erl_printf(COUT, "%-16s %8d\n", opc[i].name, opc[i].count);
		    }
		}
	    }
	    return;
	case 'z':
	    {
		int i;
		for (i = 0; i <= HIGHEST_OP; i++)
		    opc[i].count = 0;
	    }
	    return;
#endif
#ifdef DEBUG
	case 't':
	    p_slpq();
	    return;
	case 'b':
	    bin_check();
	    return;
	case 'C':
	    abort();
#endif
	case '\n':
	    continue;
	default: 
	    erl_printf(COUT, "Eh?\n\n");
	}
    }
}


#ifdef OPPROF
static void
dump_frequencies(void)
{
    int i;
    FILE* fp;
    time_t now;
    static char name[] = "op_freq.dump";

    fp = fopen(name, "w");
    if (fp == NULL) {
	fprintf(stderr, "Failed to open %s for writing\n", name);
	return;
    }

    time(&now);
    fprintf(fp, "# Generated %s\n", ctime(&now));

    for (i = 0; i <= HIGHEST_OP; i++) {
	if (opc[i].name != NULL) {
	    fprintf(fp, "%s %d\n", opc[i].name, opc[i].count);
	}
    }
    fclose(fp);
    erl_printf(COUT, "Frequencies dumped to %s\n", name);
}
#endif


#ifdef DEBUG

static void 
bin_check(void)
{
    Process  *rp;
    ProcBin *bp;
    int i, printed;

    for (i=0; i < max_process; i++) {
	if ((rp = process_tab[i]) == NULL)
	    continue;
	if (!(bp = rp->off_heap.mso))
	    continue;
	printed = 0;
	while (bp) {
	    if (printed == 0) {
		erl_printf(COUT,"Process "); 
		display(rp->id, COUT);
		erl_printf(COUT," holding binary data \n");
		printed = 1;
	    }
	    erl_printf(COUT,"0x%08x orig_size: %d, norefs = %d\n",
		       (int)bp->val, bp->val->orig_size, bp->val->refc);

	    bp = bp->next;
	}
	if (printed == 1)
	    erl_printf(COUT,"--------------------------------------\n");
    }
    /* db_bin_check() has to be rewritten for the AVL trees... */
    /*db_bin_check();*/ 
}

#endif

/* XXX THIS SHOULD SHOULD BE IN SYSTEM !!!! */
void erl_crash_dump(char *file, int line, char* fmt, va_list args)
{
    int fd;
    time_t now;
    char* dumpname;
    char buf[512];

    dumpname = getenv("ERL_CRASH_DUMP");
    if (!dumpname)
	dumpname = "erl_crash.dump";
#ifndef VXWORKS
    close(3);			/* Make sure we have a free descriptor */
#endif
    fd = open(dumpname,O_WRONLY | O_CREAT | O_TRUNC,0640);
    if(fd < 0) 
	return; /* Can't create the crash dump, skip it */

    time(&now);
    erl_printf(fd,"<Erlang crash dump>\n%s\n",ctime(&now));

    if (file != NULL)
       erl_printf(fd,"The error occurred in file %s, line %d\n", file, line);

    if (fmt != NULL && *fmt != '\0') {
	vsprintf(buf, fmt, args);
	erl_printf(fd,"Slogan: %s\n\n",buf);
    } else {
	erl_printf(fd,"No slogan.\n\n");
    }
    erl_printf(fd,"Erlang (%s) emulator version " ERLANG_VERSION "\n",EMULATOR);
    erl_printf(fd,"Compiled on " ERLANG_COMPILE_DATE "\n");

    if (process_tab != NULL)  /* XXX true at init */
	process_info(fd); /* Info about each process and port */
    erl_printf(fd,"\nInternal Table Information\n");
    info(fd); /* General system info */
    erl_printf(fd,"\nETS tables\n");
    erl_printf(fd,"--------------------------------------------------\n");
    db_info(fd, 0);
    erl_printf(fd,"\nTimers\n");
    erl_printf(fd,"--------------------------------------------------\n");
    print_timer_info(fd);
    erl_printf(fd,"--------------------------------------------------\n");
    erl_printf(fd,"\nDistribution Information\n");
    distribution_info(fd);
    erl_printf(fd,"\nLoaded Modules Information\n");
    loaded(fd);
    erl_printf(fd,"\nAtoms\n");
    erl_printf(fd,"--------------------------------------------------\n");
    dump_atoms(fd);


#ifdef INSTRUMENT
    erl_printf(fd,"\nMemory allocation information\n");
    erl_printf(fd,"--------------------------------------------------\n");
    dump_memory_to_fd(fd);
#endif

    erl_printf(fd,"\n<End of Erlang crash dump>\n");
    close(fd);
}
