%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id$
%%
-module(asn1ct_gen).

-include("asn1_records.hrl").
%%-compile(export_all).
-export([pgen_exports/3,
	 pgen_hrl/4,
	 gen_head/3,
	 demit/1,
	 emit/1,
	 fopen/2,
	 get_inner/1,type/1,def_to_tag/1,prim_bif/1,
	 type_from_object/1,
	 get_typefromobject/1,get_fieldcategory/2,
	 get_classfieldcategory/2,
	 list2name/1,
	 list2rname/1,
	 constructed_suffix/2,
	 unify_if_string/1,
	 gen_check_call/7,
	 get_constraint/2,
	 insert_once/2,
	 rt2ct_suffix/1,rt2ct_suffix/0]).
-export([pgen/4,pgen_module/5,mk_var/1]).
-export([gen_encode_constructed/4,gen_decode_constructed/4]).

%% pgen(Erules, Module, TypeOrVal)
%% Generate Erlang module (.erl) and (.hrl) file corresponding to an ASN.1 module
%% .hrl file is only generated if necessary
%% Erules = per | ber | ber_bin | per_bin
%% Module = atom()
%% TypeOrVal = {TypeList,ValueList}
%% TypeList = ValueList = [atom()]

pgen(OutFile,Erules,Module,TypeOrVal) ->
    pgen_module(OutFile,Erules,Module,TypeOrVal,true).


pgen_module(OutFile,Erules,Module,TypeOrVal,Indent) ->
    put(outfile,OutFile),
    HrlGenerated = asn1ct_gen:pgen_hrl(Erules,Module,TypeOrVal,Indent),
    asn1ct_name:start(),
    ErlFile = lists:concat([OutFile,".erl"]),
    Fid = asn1ct_gen:fopen(ErlFile,write),
    put(gen_file_out,Fid),
    asn1ct_gen:gen_head(Erules,Module,HrlGenerated),
    pgen_exports(Erules,Module,TypeOrVal),
    pgen_dispatcher(Erules,Module,TypeOrVal),
    pgen_typeorval(Erules,Module,TypeOrVal),
% gen_vars(asn1_db:mod_to_vars(Module)),
% gen_tag_table(AllTypes),
    file:close(Fid),
    io:format("--~p--~n",[{generated,ErlFile}]).


pgen_typeorval(Erules,Module,{Types,Values,Ptypes,Classes,Objects,ObjectSets}) ->
    pgen_types(Erules,Module,Types),
    pgen_values(Erules,Module,Values),
    pgen_objects(Erules,Module,Objects),
    pgen_objectsets(Erules,Module,ObjectSets),
    case catch lists:member(der,get(encoding_options)) of
	true ->
	    pgen_check_defaultval(Erules,Module);
	_ -> ok
    end.

pgen_values(_,_,[]) ->
    true;
pgen_values(Erules,Module,[H|T]) ->
    Valuedef = asn1_db:dbget(Module,H),
    gen_value(Valuedef),
    pgen_values(Erules,Module,T).

pgen_types(_,_,[]) ->
    true;
pgen_types(Erules,Module,[H|T]) ->
    Rtmod = list_to_atom(lists:concat(["asn1ct_gen_",erule(Erules),
				       rt2ct_suffix(Erules)])),
    asn1ct_name:clear(),
    Typedef = asn1_db:dbget(Module,H),
    Rtmod:gen_encode(Erules,Typedef),
    asn1ct_name:clear(),
    Rtmod:gen_decode(Erules,Typedef),
    pgen_types(Erules,Module,T).

pgen_objects(_,_,[]) ->
    true;
pgen_objects(Erules,Module,[H|T]) ->
    Rtmod = list_to_atom(lists:concat(["asn1ct_gen_",erule(Erules),
				       rt2ct_suffix(Erules)])),
    asn1ct_name:clear(),
    Typedef = asn1_db:dbget(Module,H),
    Rtmod:gen_obj_code(Erules,Module,Typedef),
    pgen_objects(Erules,Module,T).

pgen_objectsets(_,_,[]) ->
    true;
pgen_objectsets(Erules,Module,[H|T]) ->
    Rtmod = list_to_atom(lists:concat(["asn1ct_gen_",erule(Erules),
				       rt2ct_suffix(Erules)])),
    asn1ct_name:clear(),
    TypeDef = asn1_db:dbget(Module,H),
    Rtmod:gen_objectset_code(Erules,TypeDef),
    pgen_objectsets(Erules,Module,T).

pgen_check_defaultval(Erules,Module) ->
    CheckObjects = ets:tab2list(check_functions),
    FileName = lists:concat([Module,'.table']),
    {ok,IoDevice} = file:open(FileName,[write]),
    Fun =
	fun(X)->
		io:format(IoDevice,"~n~n************~n~n~p~n~n*****"
			  "********~n~n",[X]) 
	end,
    lists:foreach(Fun,CheckObjects),
    file:close(IoDevice),
    gen_check_defaultval(Erules,Module,CheckObjects).
				    
    
gen_types(Erules,Tname,{RootList,ExtList}) when list(RootList) ->
    gen_types(Erules,Tname,RootList),
    gen_types(Erules,Tname,ExtList);
gen_types(Erules,Tname,[{'EXTENSIONMARK',_,_}|Rest]) ->
    gen_types(Erules,Tname,Rest);
gen_types(Erules,Tname,[ComponentType|Rest]) ->
    Rtmod = list_to_atom(lists:concat(["asn1ct_gen_",erule(Erules),
				       rt2ct_suffix(Erules)])),
    asn1ct_name:clear(),
    Rtmod:gen_encode(Erules,Tname,ComponentType),
    asn1ct_name:clear(),
    Rtmod:gen_decode(Erules,Tname,ComponentType),
    gen_types(Erules,Tname,Rest);
gen_types(_,Tname,[]) ->
    true;
gen_types(Erules,Tname,Type) when record(Type,type) ->
    Rtmod = list_to_atom(lists:concat(["asn1ct_gen_",erule(Erules),
				       rt2ct_suffix(Erules)])),
    asn1ct_name:clear(),
    Rtmod:gen_encode(Erules,Tname,Type),
    asn1ct_name:clear(),
    Rtmod:gen_decode(Erules,Tname,Type).


gen_check_defaultval(Erules,Module,[{Name,Type}|Rest]) ->
    gen_check_func(Name,Type),
    gen_check_defaultval(Erules,Module,Rest);
gen_check_defaultval(_,_,[]) ->
    ok.

gen_check_func(Name,FType = #type{def=Def}) ->
    emit({Name,"(V,asn1_DEFAULT) ->",nl,"   true;",nl}),
    emit({Name,"(V,V) ->",nl,"   true;",nl}),
    emit({Name,"(V,{_,V}) ->",nl,"   true;",nl}),
    case Def of
	{'SEQUENCE OF',Type} ->
	    gen_check_sof(Name,'SEQOF',Type);
	{'SET OF',Type} ->
	    gen_check_sof(Name,'SETOF',Type);
	#'SEQUENCE'{components=Components} ->
	    gen_check_sequence(Name,Components);
	#'SET'{components=Components} ->
	    gen_check_sequence(Name,Components);
	{'CHOICE',Components} ->
	    gen_check_choice(Name,Components);
	#'Externaltypereference'{type=T} ->
	    emit({Name,"(DefaultValue,Value) ->",nl}),
	    emit({"   ",list2name([T,check]),"(DefaultValue,Value).",nl});
	MaybePrim ->
	    InnerType = get_inner(MaybePrim),
	    case type(InnerType) of
		{primitive,bif} ->
		    emit({Name,"(DefaultValue,Value) ->",nl,"   "}),
		    gen_prim_check_call(InnerType,"DefaultValue","Value",
					FType),
		    emit({".",nl,nl});
		_ ->
		    throw({asn1_error,{unknown,type,MaybePrim}})
	    end
    end.

gen_check_sof(Name,SOF,Type) ->
    NewName = list2name([sorted,Name]),
    emit({Name,"(V1,V2) ->",nl}),
    emit({"   ",NewName,"(lists:sort(V1),lists:sort(V2)).",nl,nl}),
    emit({NewName,"([],[]) ->",nl,"   true;",nl}),
    emit({NewName,"([DV|DVs],[V|Vs]) ->",nl,"   "}),
    InnerType = get_inner(Type#type.def),
    case type(InnerType) of
	{primitive,bif} ->
	    gen_prim_check_call(InnerType,"DV","V",Type),
	    emit({",",nl});
	{constructed,bif} ->
	    emit({list2name([SOF,Name]),"(DV, V),",nl});
	#'Externaltypereference'{type=T} ->
	    emit({list2name([T,check]),"(DV,V),",nl})
    end,
    emit({"   ",NewName,"(DVs,Vs).",nl,nl}).

gen_check_sequence(Name,Components) ->
    emit({Name,"(DefaultValue,Value) ->",nl}),
    gen_check_sequence(Name,Components,1).
gen_check_sequence(Name,[#'ComponentType'{name=N,typespec=Type}|Cs],Num) ->
    InnerType = get_inner(Type#type.def),
%    NthDefV = lists:concat(["lists:nth(",Num,",DefaultValue)"]),
    NthDefV = ["element(",Num+1,",DefaultValue)"],
%    NthV = lists:concat(["lists:nth(",Num,",Value)"]),
    NthV = ["element(",Num+1,",Value)"],
    gen_check_func_call(Name,Type,InnerType,NthDefV,NthV,N),
    case Cs of
	[] ->
	    emit({".",nl,nl});
	_ ->
	    emit({",",nl}),
	    gen_check_sequence(Name,Cs,Num+1)
    end;
gen_check_sequence(_,[],_) ->
    ok.

gen_check_choice(Name,CList=[#'ComponentType'{name=N,typespec=Type}|Cs]) ->
    emit({Name,"({Id,DefaultValue},{Id,Value}) ->",nl}),
    emit({"   case Id of",nl}),
    gen_check_choice_components(Name,CList,1).

gen_check_choice_components(_,[],_)->
    ok;
gen_check_choice_components(Name,[#'ComponentType'{name=N,typespec=Type}|
				  Cs],Num) ->
    Ind6 = "      ",
    InnerType = get_inner(Type#type.def),
%    DefVal = ["element(2,lists:nth(",Num,",DefaultValue))"],
    emit({Ind6,N," ->",nl,Ind6}),
    gen_check_func_call(Name,Type,InnerType,{var,"defaultValue"},
			{var,"value"},N),
    case Cs of
	[] ->
	    emit({nl,"   end.",nl,nl});
	_ ->
	    emit({";",nl}),
	    gen_check_choice_components(Name,Cs,Num+1)
    end.

gen_check_func_call(Name,Type,InnerType,DefVal,Val,N) ->
    case type(InnerType) of
	{primitive,bif} ->
	    emit("   "),
	    gen_prim_check_call(InnerType,DefVal,Val,Type);
	#'Externaltypereference'{module=M,type=T} ->
	    emit({"   ",list2name([T,check]),"(",DefVal,",",Val,")"});
	_ ->
	    emit({"   ",list2name([N,Name]),"(",DefVal,",",Val,")"})
    end.
		  
    
%% VARIOUS GENERATOR STUFF 
%% *************************************************
%%**************************************************

mk_var(X) when atom(X) ->
    list_to_atom(mk_var(atom_to_list(X)));

mk_var([H|T]) ->
    [H-32|T].


%% Generate value functions ***************
%% ****************************************
%% Generates a function 'V'/0 for each Value V defined in the ASN.1 module
%% the function returns the value in an Erlang representation which can be
%% used as  input to the runtime encode functions

gen_value(Value) when record(Value,valuedef) ->
%%    io:format(" ~w ",[Value#valuedef.name]),
    emit({"'",Value#valuedef.name,"'() ->",nl}),
    V = Value#valuedef.value,
    emit([{asis,V},".",nl,nl]).

gen_encode_constructed(Erules,Typename,InnerType,D) when record(D,type) ->

    Rtmod = list_to_atom(lists:concat(["asn1ct_constructed_",erule(Erules)])),
    case InnerType of
	'SET' ->
	    Rtmod:gen_encode_set(Erules,Typename,D),
	    #'SET'{components=Components} = D#type.def,
	    gen_types(Erules,Typename,Components);
	'SEQUENCE' ->
	    Rtmod:gen_encode_sequence(Erules,Typename,D),
	    #'SEQUENCE'{components=Components} = D#type.def,
	    gen_types(Erules,Typename,Components);
	'CHOICE' ->
	    Rtmod:gen_encode_choice(Erules,Typename,D),
	    {_,Components} = D#type.def,
	    gen_types(Erules,Typename,Components);
	'SEQUENCE OF' ->
	    Rtmod:gen_encode_sof(Erules,Typename,InnerType,D),
	    {_,Type} = D#type.def,
	    NameSuffix = asn1ct_gen:constructed_suffix(InnerType,Type#type.def),
	    gen_types(Erules,[NameSuffix|Typename],Type);
	'SET OF' ->
	    Rtmod:gen_encode_sof(Erules,Typename,InnerType,D),
	    {_,Type} = D#type.def,
	    NameSuffix = asn1ct_gen:constructed_suffix(InnerType,Type#type.def),
	    gen_types(Erules,[NameSuffix|Typename],Type);
	Other ->
	    exit({nyi,InnerType})
    end;
gen_encode_constructed(Erules,Typename,InnerType,D) 
  when record(D,typedef) ->
    gen_encode_constructed(Erules,Typename,InnerType,D#typedef.typespec).

gen_decode_constructed(Erules,Typename,InnerType,D) when record(D,type) ->
    Rtmod = list_to_atom(lists:concat(["asn1ct_constructed_",erule(Erules)])),
    case InnerType of
	'SET' ->
	    Rtmod:gen_decode_set(Erules,Typename,D);
	'SEQUENCE' ->
	    Rtmod:gen_decode_sequence(Erules,Typename,D);
	'CHOICE' ->
	    Rtmod:gen_decode_choice(Erules,Typename,D);
	'SEQUENCE OF' ->
	    Rtmod:gen_decode_sof(Erules,Typename,InnerType,D);
	'SET OF' ->
	    Rtmod:gen_decode_sof(Erules,Typename,InnerType,D);
	Other ->
	    exit({nyi,InnerType})
    end;


gen_decode_constructed(Erules,Typename,InnerType,D) when record(D,typedef) ->
    gen_decode_constructed(Erules,Typename,InnerType,D#typedef.typespec).


pgen_exports(Erules,Module,{Types,Values,_,_,Objects,ObjectSets}) ->
    emit({"-export([encoding_rule/0]).",nl}),
    case Types of
	[] -> ok;
	_ ->
	    emit({"-export([",nl}),
	    case Erules of
		ber ->
		    gen_exports1(Types,"enc_",2);
		ber_bin ->
		    gen_exports1(Types,"enc_",2);
		_ ->
		    gen_exports1(Types,"enc_",1)
	    end,
	    emit({"-export([",nl}),
	    gen_exports1(Types,"dec_",2),
	    case Erules of
		ber ->
		    emit({"-export([",nl}),
		    gen_exports1(Types,"dec_",3);
		ber_bin ->
		    emit({"-export([",nl}),
		    gen_exports1(Types,"dec_",3);
		_ -> ok
	    end
    end,
    case Values of
	[] -> ok;
	_ ->
	    emit({"-export([",nl}),
	    gen_exports1(Values,"",0)
    end,
    case Objects of
	[] -> ok;
	_ ->
	    case erule(Erules) of
		per ->
		    emit({"-export([",nl}),
		    gen_exports1(Objects,"enc_",3),
		    emit({"-export([",nl}),
		    gen_exports1(Objects,"dec_",4);
		_ -> 
		    emit({"-export([",nl}),
		    gen_exports1(Objects,"enc_",4),
		    emit({"-export([",nl}),
		    gen_exports1(Objects,"dec_",4)
	    end
    end,
    case ObjectSets of
	[] -> ok;
	_ ->
	    emit({"-export([",nl}),
	    gen_exports1(ObjectSets,"getenc_",2),
	    emit({"-export([",nl}),
	    gen_exports1(ObjectSets,"getdec_",2)
    end,
    emit({nl,nl}).

gen_exports1([F1,F2|T],Prefix,Arity) ->
	emit({"'",Prefix,F1,"'/",Arity,com,nl}),
	gen_exports1([F2|T],Prefix,Arity);
gen_exports1([Flast|T],Prefix,Arity) ->
	emit({"'",Prefix,Flast,"'/",Arity,nl,"]).",nl,nl}).


pgen_dispatcher(Erules,Module,{[],Values,_,_,Objects,ObjectSets}) ->
    emit(["encoding_rule() ->",nl]),
    emit([{asis,Erules},".",nl,nl]);
pgen_dispatcher(Erules,Module,{Types,Values,_,_,Objects,ObjectSets}) ->
    emit(["-export([encode/2,decode/2,encode_disp/2,decode_disp/2]).",nl,nl]),
    emit(["encoding_rule() ->",nl]),
    emit(["   ",{asis,Erules},".",nl,nl]),
    Call = case Erules of
	       per -> "?RT_PER:complete(encode_disp(Type,Data))";
	       per_bin -> "?RT_PER:complete(encode_disp(Type,Data))";
	       ber -> "encode_disp(Type,Data)";
	       ber_bin -> "encode_disp(Type,Data)"
	   end,
    emit(["encode(Type,Data) ->",nl,
	  "case catch ",Call," of",nl,
	  "  {'EXIT',{error,Reason}} ->",nl,
	  "    {error,Reason};",nl,
	  "  {'EXIT',Reason} ->",nl,
	  "    {error,{asn1,Reason}};",nl,
	  "  {Bytes,Len} ->",nl,
	  "    {ok,Bytes};",nl,
	  "  X ->",nl,
	  "    {ok,X}",nl,
	  "end.",nl,nl]),

    emit(["decode(Type,Data) ->",nl,
	  "case catch decode_disp(Type,Data) of",nl,
	  "  {'EXIT',{error,Reason}} ->",nl,
	  "    {error,Reason};",nl,
	  "  {'EXIT',Reason} ->",nl,
	  "    {error,{asn1,Reason}};",nl,
	  "  {X,_Rest} ->",nl,
	  "    {ok,X};",nl,
	  "  {X,_Rest,_Len} ->",nl,
	  "    {ok,X}",nl,
	  "end.",nl,nl]),

    case Types of
	[] -> ok;
	_ ->
	    case Erules of
		ber ->
		    gen_dispatcher(Types,"encode_disp","enc_",",[]"),
		    gen_dispatcher(Types,"decode_disp","dec_",",mandatory");
		ber_bin ->
		    gen_dispatcher(Types,"encode_disp","enc_",",[]"),
		    gen_dispatcher(Types,"decode_disp","dec_",",mandatory");
		_ ->
		    gen_dispatcher(Types,"encode_disp","enc_",""),
		    gen_dispatcher(Types,"decode_disp","dec_",",mandatory")
	    end,
	    emit([nl])
    end,
    emit({nl,nl}).

gen_dispatcher([F1,F2|T],FuncName,Prefix,ExtraArg) ->
	emit([FuncName,"('",F1,"',Data) -> '",Prefix,F1,"'(Data",ExtraArg,")",";",nl]),
	gen_dispatcher([F2|T],FuncName,Prefix,ExtraArg);
gen_dispatcher([Flast|T],FuncName,Prefix,ExtraArg) ->
	emit([FuncName,"('",Flast,"',Data) -> '",Prefix,Flast,"'(Data",ExtraArg,")",";",nl]),
	emit([FuncName,"(","Type",",Data) -> exit({error,{asn1,{undefined_type,Type}}}).",nl,nl,nl]).

open_hrl(OutFile,Module) ->
    File = lists:concat([OutFile,".hrl"]),
    Fid = fopen(File,write),
    put(gen_file_out,Fid),
    gen_hrlhead(Module).

%% EMIT functions ************************
%% ***************************************

						% debug generation
demit(Term) ->
    case get(asndebug) of
	true -> emit(Term);
	_ ->true
    end.

						% always generation

emit({external,M,T}) ->
    emit(T);

emit({prev,Variable}) when atom(Variable) ->
    emit({var,asn1ct_name:prev(Variable)});

emit({next,Variable}) when atom(Variable) ->
    emit({var,asn1ct_name:next(Variable)});

emit({curr,Variable}) when atom(Variable) ->
    emit({var,asn1ct_name:curr(Variable)});
    
emit({var,Variable}) when atom(Variable) ->
    [Head|V] = atom_to_list(Variable),
    emit([Head-32|V]);

emit({var,Variable}) ->
    [Head|V] = Variable,
    emit([Head-32|V]);

emit({asis,What}) ->
    format(get(gen_file_out),"~w",[What]);

emit(nl) ->
    nl(get(gen_file_out));

emit(com) ->
    emit(",");

emit(tab) ->
    put_chars(get(gen_file_out),"     ");

emit(What) when integer(What) ->
    put_chars(get(gen_file_out),integer_to_list(What));

emit(What) when list(What), integer(hd(What)) ->
    put_chars(get(gen_file_out),What);

emit(What) when atom(What) ->
    put_chars(get(gen_file_out),atom_to_list(What));

emit(What) when tuple(What) ->
    emit_parts(tuple_to_list(What));

emit(What) when list(What) ->
    emit_parts(What);

emit(X) ->
    exit({'cant emit ',X}).

emit_parts([]) -> true;
emit_parts([H|T]) ->
    emit(H),
    emit_parts(T).

format(undefined,X,Y) ->
    io:format(X,Y);
format(X,Y,Z) ->
    io:format(X,Y,Z).

nl(undefined) -> io:nl();
nl(X) -> io:nl(X).

put_chars(undefined,X) ->
    io:put_chars(X);
put_chars(Y,X) ->
    io:put_chars(Y,X).

fopen(F, Mode) ->
    case file:open(F, Mode) of
	{ok, Fd} -> 
	    Fd;
	{error, Reason} ->
	    io:format("** Can't open file ~p ~n", [F]),
	    exit({error,Reason})
    end.

pgen_hrl(Erules,Module,TypeOrVal,_Indent) ->
    put(currmod,Module),
    {Types,Values,Ptypes,_,_,_} = TypeOrVal,
    Ret =
	case pgen_hrltypes(Erules,Module,Ptypes++Types,0) of
	    0 -> 
		case Values of
		    [] ->
			0;
		    _ ->
			open_hrl(get(outfile),get(currmod)),
			pgen_macros(Erules,Module,Values),
			1
		end;
	    X ->
		pgen_macros(Erules,Module,Values),
		X
	end,
    case Ret of
	0 ->
	    0;
	Y ->
	    Fid = get(gen_file_out),
	    file:close(Fid),
	    io:format("--~p--~n",
		      [{generated,lists:concat([get(outfile),".hrl"])}]),
	    Y
    end.

pgen_macros(_,_,[]) ->
    true;
pgen_macros(Erules,Module,[H|T]) ->
    Valuedef = asn1_db:dbget(Module,H),
    gen_macro(Valuedef),
    pgen_macros(Erules,Module,T).

pgen_hrltypes(_,_,[],NumRecords) ->
    NumRecords;
pgen_hrltypes(Erules,Module,[H|T],NumRecords) ->
%    io:format("records = ~p~n",NumRecords),
    Typedef = asn1_db:dbget(Module,H),
    AddNumRecords = gen_record(Typedef,NumRecords),
    pgen_hrltypes(Erules,Module,T,NumRecords+AddNumRecords).


%% Generates a macro for value Value defined in the ASN.1 module
gen_macro(Value) when record(Value,valuedef) ->
    emit({"-define('",Value#valuedef.name,"', ",
	  {asis,Value#valuedef.value},").",nl}).

%% Generate record functions **************
%% Generates an Erlang record for each named and unnamed SEQUENCE and SET in the ASN.1 
%% module. If no SEQUENCE or SET is found there is no .hrl file generated


gen_record(Tdef,NumRecords) when record(Tdef,typedef) ->
    Name = [Tdef#typedef.name],
    Type = Tdef#typedef.typespec,
    gen_record(type,Name,Type,NumRecords);

gen_record(Tdef,NumRecords) when record(Tdef,ptypedef) ->
    Name = [Tdef#ptypedef.name],
    Type = Tdef#ptypedef.typespec,
    gen_record(ptype,Name,Type,NumRecords).
    
gen_record(TorPtype,Name,[#'ComponentType'{name=Cname,typespec=Type}|T],Num) ->
    Num2 = gen_record(TorPtype,[Cname|Name],Type,Num),
    gen_record(TorPtype,Name,T,Num2);
gen_record(TorPtype,Name,[H|T],Num) -> % skip EXTENSIONMARK
    gen_record(TorPtype,Name,T,Num);
gen_record(_TorPtype,_Name,[],Num) ->
    Num;

gen_record(TorPtype,Name,Type,Num) when record(Type,type) ->    
    Def = Type#type.def,
    Rec = case Def of
	      Seq when record(Seq,'SEQUENCE') ->
		  case Seq#'SEQUENCE'.pname of
		      false ->
			  {record,Seq#'SEQUENCE'.components};
		      Pname when TorPtype == type ->
			  false;
		      _ ->
			  {record,Seq#'SEQUENCE'.components}
		  end;
	      Set when record(Set,'SET') ->
		  case Set#'SET'.pname of
		      false ->
			  {record,Set#'SET'.components};
		      Pname when TorPtype == type ->
			  false;
		      _ ->
			  {record,Set#'SET'.components}
		  end;
%	      {'SET',{_,_CompList}} -> 
%		  {record,_CompList}; 
	      {'CHOICE',_CompList} -> {inner,Def};
	      {'SEQUENCE OF',_CompList} -> {['SEQOF'|Name],Def};
	      {'SET OF',_CompList} -> {['SETOF'|Name],Def};
	      _ -> false
    end,
    case Rec of
	false -> Num;
	{record,CompList} ->
	    case Num of
		0 -> open_hrl(get(outfile),get(currmod));
		_ -> true
	    end,
	    emit({"-record('",list2name(Name),"',{",nl}),
	    RootList = case CompList of
			   _ when list(CompList) ->
			       CompList;
			   {_Rl,_} -> _Rl
		       end,
	    gen_record2(Name,'SEQUENCE',RootList),
	    NewCompList = 
		case CompList of
		    {CompList1,[]} ->
			emit({"}). % with extension mark",nl,nl}),
			CompList1;
		    {Tr,ExtensionList2} ->
			case Tr of
			    [] -> true;
			    _ -> emit({",",nl})
			end,
			emit({"%% with extensions",nl}),
			gen_record2(Name, 'SEQUENCE', ExtensionList2,
				    "", ext),
			emit({"}).",nl,nl}),
			Tr ++ ExtensionList2;
		    _ -> 
			emit({"}).",nl,nl}),
			CompList
		end,
	    gen_record(TorPtype,Name,NewCompList,Num+1);
	{inner,{'CHOICE', CompList}} ->
	    gen_record(TorPtype,Name,CompList,Num);
	{NewName,{_, CompList}} ->
	    gen_record(TorPtype,NewName,CompList,Num)
    end;
gen_record(_,_,_,NumRecords) -> % skip CLASS etc for now.
     NumRecords.
		    
gen_head(Erules,Mod,Hrl) ->
    {Rtmac,Rtmod} = case Erules of
			per ->
			    emit({"%% Generated by the Erlang ASN.1 PER-"
				  "compiler version:",asn1ct:vsn(),nl}),
			    {"RT_PER",?RT_PER};
			ber ->
			    emit({"%% Generated by the Erlang ASN.1 BER-"
				  "compiler version:",asn1ct:vsn(),nl}),
			    {"RT_BER",?RT_BER};
			per_bin ->
			    emit({"%% Generated by the Erlang ASN.1 BER-"
				  "compiler version, utilizing bit-syntax:",
				  asn1ct:vsn(),nl}),
			    %% temporary code to enable rt2ct optimization
			    Options = get(encoding_options),
			    case lists:member(optimize,Options) of
				true -> {"RT_PER","asn1rt_per_bin_rt2ct"};
				_ ->
				    {"RT_PER",?RT_PER_BIN}
			    end;
			ber_bin ->
			    emit({"%% Generated by the Erlang ASN.1 BER-"
				  "compiler version, utilizing bit-syntax:",
				  asn1ct:vsn(),nl}),
			    {"RT_BER",?RT_BER_BIN}
    end,
    emit({"%% Purpose: encoder and decoder to the types in mod ",Mod,nl,nl}),
    emit({"-module('",Mod,"').",nl}),
    put(currmod,Mod),
    %emit({"-compile(export_all).",nl}),
    case Hrl of
	0 -> true;
	_ -> 
	    emit({"-include(\"",Mod,".hrl\").",nl})
    end,
    emit(["-define('",Rtmac,"',",Rtmod,").",nl]).

gen_hrlhead(Mod) ->
    emit({"%% Generated by the Erlang ASN.1 compiler version:",asn1ct:vsn(),nl}),
    emit({"%% Purpose: Erlang record definitions for each named and unnamed",nl}),
    emit({"%% SEQUENCE and SET, and macro definitions for each value",nl}),
    emit({"%% definition,in module ",Mod,nl,nl}),
    emit({nl,nl}).

gen_record2(Name,SeqOrSet,Comps) ->
    gen_record2(Name,SeqOrSet,Comps,"",noext).

gen_record2(Name,SeqOrSet,[],Com,Extension) ->
    true;
gen_record2(Name,SeqOrSet,[{'EXTENSIONMARK',_,_}|T],Com,Extension) ->
    gen_record2(Name,SeqOrSet,T,Com,Extension);
gen_record2(Name,SeqOrSet,[H],Com,Extension) ->
    #'ComponentType'{name=Cname} = H,
    emit(Com),
    emit({asis,Cname}),
    gen_record_default(H, Extension);
gen_record2(Name,SeqOrSet,[H|T],Com, Extension) ->
    #'ComponentType'{name=Cname} = H,
    emit(Com),
    emit({asis,Cname}),
    gen_record_default(H, Extension),
%    emit(", "),
    gen_record2(Name,SeqOrSet,T,", ", Extension).

%gen_record_default(C, ext) ->
%    emit(" = asn1_NOEXTVALUE");
gen_record_default(#'ComponentType'{prop='OPTIONAL'}, _)->
    emit(" = asn1_NOVALUE"); 
gen_record_default(#'ComponentType'{prop={'DEFAULT',Dval}}, _)->
    emit(" = asn1_DEFAULT"); 
gen_record_default(_, _) ->
    true.

gen_check_call(TopType,Cname,Type,InnerType,WhatKind,DefaultValue,Element) ->
    case WhatKind of
	{primitive,bif} ->
	    gen_prim_check_call(InnerType,DefaultValue,Element,Type);
	#'Externaltypereference'{module=M,type=T} ->
	    %% generate function call
	    Name = list2name([T,check]),
	    emit({"'",Name,"'(",DefaultValue,", ",Element,")"}),
	    %% insert in ets table and do look ahead check
	    Typedef = asn1_db:dbget(M,T),
	    RefType = Typedef#typedef.typespec,
	    InType = asn1ct_gen:get_inner(RefType#type.def),
	    case insert_once(check_functions,{Name,RefType}) of
		true ->
		    lookahead_innertype([T],InType,RefType);
%		    case asn1ct_gen:type(InType) of
%			{constructed,bif} ->
%			    lookahead_innertype([T],InType,RefType);
%			#'Externaltypereference'{type=TNew} ->
%			    lookahead_innertype([TNew],InType,RefType);
%			_ ->
%			    ok
%		    end;
		_ ->
		    ok
	    end;
	{constructed,bif} ->
	    NameList = [Cname|TopType],
	    Name = list2name(NameList ++ [check]),
	    emit({"'",Name,"'(",DefaultValue,", ",Element,")"}),
	    ets:insert(check_functions,{Name,Type}),
	    %% Must look for check functions in InnerType,
	    %% that may be referenced  or internal defined 
	    %% constructed types not used elsewhere.
	    lookahead_innertype(NameList,InnerType,Type)
    end.

gen_prim_check_call(PrimType,DefaultValue,Element,Type) ->
    case unify_if_string(PrimType) of
	'BOOLEAN' ->
	    emit({"asn1rt_check:check_bool(",DefaultValue,", ",
		  Element,")"});
	'INTEGER' ->
	    NNL =
		case Type#type.def of
		    {_,NamedNumberList} -> NamedNumberList;
		    _ -> []
		end,
	    emit({"asn1rt_check:check_int(",DefaultValue,", ",
		  Element,", ",{asis,NNL},")"});
	'BIT STRING' ->
	    {_,NBL} = Type#type.def,
	    emit({"asn1rt_check:check_bitstring(",DefaultValue,", ",
		  Element,", ",{asis,NBL},")"});
	'OCTET STRING' ->
	    emit({"asn1rt_check:check_octetstring(",DefaultValue,", ",
		  Element,")"});
	'NULL' ->
	    emit({"asn1rt_check:check_null(",DefaultValue,", ",
		  Element,")"});
	'OBJECT IDENTIFIER' ->
	    emit({"asn1rt_check:check_objectidentifier(",DefaultValue,
		  ", ",Element,")"});
	'ObjectDescriptor' ->
	    emit({"asn1rt_check:check_objectdescriptor(",DefaultValue,
		  ", ",Element,")"});
	'REAL' ->
	    emit({"asn1rt_check:check_real(",DefaultValue,
		  ", ",Element,")"});
	'ENUMERATED' ->
	    {_,Enumerations} = Type#type.def,
	    emit({"asn1rt_check:check_enum(",DefaultValue,
		  ", ",Element,", ",{asis,Enumerations},")"});
	restrictedstring ->
	    emit({"asn1rt_check:check_restrictedstring(",DefaultValue,
		  ", ",Element,")"})
    end.

%% lokahead_innertype/3 traverses Type and checks if check functions
%% have to be generated, i.e. for all constructed or referenced types.
lookahead_innertype(Name,'SEQUENCE',Type) ->
    Components = (Type#type.def)#'SEQUENCE'.components,
    lookahead_components(Name,Components);
lookahead_innertype(Name,'SET',Type) ->
    Components = (Type#type.def)#'SET'.components,
    lookahead_components(Name,Components);
lookahead_innertype(Name,'CHOICE',Type) ->
    {_,Components} = Type#type.def,
    lookahead_components(Name,Components);
lookahead_innertype(Name,'SEQUENCE OF',SeqOf) ->
    lookahead_sof(Name,'SEQOF',SeqOf);
lookahead_innertype(Name,'SET OF',SeqOf) ->
    lookahead_sof(Name,'SETOF',SeqOf);
lookahead_innertype(Name,#'Externaltypereference'{module=M,type=T},
		    Type) ->
    Typedef = asn1_db:dbget(M,T),
    RefType = Typedef#typedef.typespec,
    InType = asn1ct_gen:get_inner(RefType#type.def),
    case type(InType) of
	{constructed,bif} ->
	    NewName = list2name([T,check]),
	    case insert_once(check_functions,{NewName,RefType}) of
		true ->
		    lookahead_innertype([T],InType,RefType);
		_ ->
		    ok
	    end;
	#'Externaltypereference'{type=TNew} ->
	    NewName = list2name([T,check]),
	    case insert_once(check_functions,{NewName,RefType}) of
		true ->
		    lookahead_innertype([T],InType,RefType);
		_ ->
		    ok
	    end;
	_ ->
	    ok
    end;
%    case insert_once(check_functions,{list2name(Name++[check]),Type}) of
%	true ->
%	    InnerType = asn1ct_gen:get_inner(Type#type.def),
%	    case asn1ct_gen:type(InnerType) of
%		{constructed,bif} ->
%		    lookahead_innertype([T],InnerType,Type);
%		#'Externaltypereference'{type=TNew} ->
%		    lookahead_innertype([TNew],InnerType,Type);
%		_ ->
%		    ok
%	    end;
%	_ ->
%	    ok
%    end;
lookahead_innertype(_,_,_) ->
    ok.

lookahead_components(_,[]) -> ok;
lookahead_components(Name,[C|Cs]) ->
    #'ComponentType'{name=Cname,typespec=Type} = C,
    InType = asn1ct_gen:get_inner(Type#type.def),
    case asn1ct_gen:type(InType) of
	{constructed,bif} ->
	    case insert_once(check_functions,
			     {list2name([Cname|Name] ++ [check]),Type}) of
		true ->
		    lookahead_innertype([Cname|Name],InType,Type);
		_ ->
		    ok
	    end;
	#'Externaltypereference'{module=RefMod,type=RefName} ->
	    Typedef = asn1_db:dbget(RefMod,RefName),
	    RefType = Typedef#typedef.typespec,
	    case insert_once(check_functions,{list2name([RefName,check]),
					      RefType}) of
		true ->
		    lookahead_innertype([RefName],InType,RefType);
		_ ->
		    ok
	    end;
	_ ->
	    ok
    end,
    lookahead_components(Name,Cs).

lookahead_sof(Name,SOF,SOFType) ->
    Type = case SOFType#type.def of
	       {_,_Type} -> _Type;
	       _Type -> _Type
	   end,
    InnerType = asn1ct_gen:get_inner(Type#type.def),
    case asn1ct_gen:type(InnerType) of
	{constructed,bif} ->
	    %% this is if a constructed type is defined in
	    %% the SEQUENCE OF type
	    NameList = [SOF|Name],
	    insert_once(check_functions,
			{list2name(NameList ++ [check]),Type}),
	    lookahead_innertype(NameList,InnerType,Type);
	#'Externaltypereference'{module=M,type=T} ->
	    Typedef = asn1_db:dbget(M,T),
	    RefType = Typedef#typedef.typespec,
	    InType = get_inner(RefType#type.def),
	    case insert_once(check_functions,
			     {list2name([T,check]),RefType}) of
		true ->
		    lookahead_innertype([T],InType,RefType);
		_ ->
		    ok
	    end;
	_ ->
	    ok
    end.


insert_once(Table,Object) ->
    case ets:lookup(Table,element(1,Object)) of
	[] ->
	    ets:insert(Table,Object); %returns true
	_ -> false
    end.

unify_if_string(PrimType) ->
    case PrimType of
	'NumericString' ->
	    restrictedstring;
	'PrintableString' ->
	    restrictedstring;
	'TeletexString' ->
	    restrictedstring;
	'VideotexString' ->
	    restrictedstring;
	'IA5String' ->
	    restrictedstring;
	'UTCTime' ->
	    restrictedstring;
	'GeneralizedTime' ->
	    restrictedstring;
	'GraphicString' ->
	    restrictedstring;
	'VisibleString' ->
	    restrictedstring;
	'GeneralString' ->
	    restrictedstring;
	'UniversalString' ->
	    restrictedstring;
	'BMPString' ->
	    restrictedstring;
	Other -> Other
    end.


	
	

get_inner(A) when atom(A) -> A;    
get_inner(Ext) when record(Ext,'Externaltypereference') -> Ext;    
get_inner(Tref) when record(Tref,typereference) -> Tref;
get_inner({fixedtypevaluefield,_,Type}) ->
    if 
	record(Type,type) ->
	    get_inner(Type#type.def);
	true ->
	    get_inner(Type)
    end;
get_inner({typefield,TypeName}) ->
    TypeName;
get_inner(T) when tuple(T) -> 
    case element(1,T) of
	Tuple when tuple(Tuple),element(1,Tuple) == objectclass ->
	    case catch(lists:last(element(2,T))) of
		{valuefieldreference,FieldName} ->
		    get_fieldtype(element(2,Tuple),FieldName);
		{typefieldreference,FieldName} ->
		    get_fieldtype(element(2,Tuple),FieldName);
		{'EXIT',Reason} ->
		    throw({asn1,{'internal error in get_inner/1',Reason}})
	    end;
	_ -> element(1,T)
    end.





type(X) when record(X,'Externaltypereference') ->
    X;
type(X) when record(X,typereference) ->
    X;
type('ASN1_OPEN_TYPE') ->
    'ASN1_OPEN_TYPE';
type({fixedtypevaluefield,Name,Type}) when record(Type,type) ->
    type(get_inner(Type#type.def));
type({typefield,_}) ->
    'ASN1_OPEN_TYPE';
type(X) ->
    %%    io:format("asn1_types:type(~p)~n",[X]),
    case catch type2(X) of
	{'EXIT',_} ->
	    {notype,X};
	Normal ->
	    Normal
    end.

type2(X) ->
    case prim_bif(X) of
	true ->
	    {primitive,bif};
	false ->
	    case construct_bif(X) of
		true ->
		    {constructed,bif};
		false ->
		    {undefined,user}
	    end
    end.

prim_bif(X) ->
    lists:member(X,['INTEGER' ,
		    'ENUMERATED',
		    'OBJECT IDENTIFIER',
		    'ANY',
		    'NULL',
		    'BIT STRING' ,
		    'OCTET STRING' ,
		    'ObjectDescriptor',
		    'NumericString',
		    'TeletexString',
		    'VideotexString',
		    'UTCTime',
		    'GeneralizedTime',
		    'GraphicString',
		    'VisibleString',
		    'GeneralString',
		    'PrintableString',
		    'IA5String',
		    'UniversalString',
		    'BMPString',
		    'ENUMERATED',
		    'BOOLEAN']).

construct_bif(T) ->
    lists:member(T,['SEQUENCE' ,
		    'SEQUENCE OF' ,
		    'CHOICE' ,
		    'SET' ,
		    'SET OF']).

def_to_tag(#tag{class=Class,number=Number}) ->
    {Class,Number};
def_to_tag(Def) ->
    {'UNIVERSAL',get_inner(Def)}.
    

%% Information Object Class

type_from_object(X) ->
    case (catch lists:last(element(2,X))) of
	{'EXIT',_} ->
	    {notype,X};
	Normal ->
	    Normal
    end.


get_fieldtype([],FieldName)->
    {no_type,no_name};
get_fieldtype([Field|Rest],FieldName) ->
    case element(2,Field) of
	FieldName ->
	    case element(1,Field) of
		fixedtypevaluefield ->
		    {element(1,Field),FieldName,element(3,Field)};
		_ ->
		    {element(1,Field),FieldName}
	    end;
	_  ->
	    get_fieldtype(Rest,FieldName)
    end.

get_fieldcategory([],FieldName) ->
    no_cat;
get_fieldcategory([Field|Rest],FieldName) ->
    case element(2,Field) of
	FieldName ->
	    element(1,Field);
	_ ->
	    get_fieldcategory(Rest,FieldName)
    end.

get_typefromobject(Type) when record(Type,type) ->
    case Type#type.def of
	{ClassDef={objectclass,_,_},TypeFrObj} when list(TypeFrObj) ->
	    {_,FieldName} = lists:last(TypeFrObj),
	    FieldName;
	_ ->
	    {no_field}
    end.

get_classfieldcategory(Type,FieldName) ->
    case (catch Type#type.def) of
	{{obejctclass,Fields,_},_} ->
	    get_fieldcategory(Fields,FieldName);
	{'EXIT',_} ->
	    no_cat;
	_ ->
	    no_cat
    end.
%% Information Object Class

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Convert a list of name parts to something that can be output by emit
%% 
%% used to output function names in generated code.

list2name(L) ->
    NewL = list2name1(L),
    lists:concat(lists:reverse(NewL)).

list2name1([{ptype,H1},H2|T]) ->
    [H1,"_",list2name([H2|T])];
list2name1([H1,H2|T]) ->
    [H1,"_",list2name([H2|T])];
list2name1([{ptype,H}|T]) ->
    [H];
list2name1([H|T]) ->
    [H];
list2name1([]) ->
    [].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Convert a list of name parts to something that can be output by emit
%% stops at {ptype,Pname} i.e Pname whill be the first part of the name
%% used to output record names in generated code.

list2rname(L) ->
    NewL = list2rname1(L),
    lists:concat(lists:reverse(NewL)).

list2rname1([{ptype,H1},H2|T]) ->
    [H1];
list2rname1([H1,H2|T]) ->
    [H1,"_",list2name([H2|T])];
list2rname1([{ptype,H}|T]) ->
    [H];
list2rname1([H|T]) ->
    [H];
list2rname1([]) ->
    [].



constructed_suffix(_,#'SEQUENCE'{pname=Ptypename}) when Ptypename =/= false ->
    {ptype, Ptypename};
constructed_suffix(_,#'SET'{pname=Ptypename}) when Ptypename =/= false ->
    {ptype,Ptypename};
constructed_suffix('SEQUENCE OF',_) -> 
    'SEQOF';
constructed_suffix('SET OF',_) -> 
    'SETOF'.

erule(ber) ->
    ber;
erule(ber_bin) ->
    ber;
erule(per) ->
    per;
erule(per_bin) ->
    per.


rt2ct_suffix() ->
    Options = get(encoding_options),
    case {lists:member(optimize,Options),lists:member(per_bin,Options)} of
	{true,true} -> "_rt2ct";
	_ -> ""
    end.
rt2ct_suffix(per_bin) ->
    Options = get(encoding_options),
    case lists:member(optimize,Options) of
	true -> "_rt2ct";
	_ -> ""
    end;
rt2ct_suffix(_) -> "".

get_constraint(C,Key) ->
    case lists:keysearch(Key,1,C) of
	false ->
	     no;
	{value,{_,V}} -> 
	    V
    end.
