/*
 *	Euler - a numerical lab
 *
 *	platform : gtk / X11
 *
 *	file : colbut.c -- color button widget
 */

#include "colbut.h"
#include <stdio.h>

#define GTK_COLOR_BOX(obj)			GTK_CHECK_CAST (obj, gtk_color_box_get_type (), GtkColorBox)
#define GTK_COLOR_BOX_CLASS(klass)	GTK_CHECK_CLASS_CAST (klass, gtk_color_box_get_type (), GtkColorBoxClass)
#define GTK_IS_COLOR_BOX(obj)		GTK_CHECK_TYPE (obj, gtk_color_box_get_type ())

typedef struct _GtkColorBox			GtkColorBox;
typedef struct _GtkColorBoxClass	GtkColorBoxClass;

struct _GtkColorBox
{
	GtkWidget	widget;
	
	gint		width, height;
	
	GdkColormap *cmap;
	GdkColor	color;
	GdkGC 		*gc;
};

struct _GtkColorBoxClass
{
	GtkWidgetClass parent_class;
};

guint			gtk_color_box_get_type			(void);

GtkWidget*		gtk_color_box_new				(gint width, gint height, gushort r, gushort g, gushort b);
void			gtk_color_box_set_color			(GtkWidget *widget, gushort r, gushort g, gushort b);

/*
 *	events methods
 */
static void gtk_color_box_realize (GtkWidget *widget);
static void gtk_color_box_unrealize (GtkWidget *widget);
static void gtk_color_box_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void gtk_color_box_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static gint gtk_color_box_expose (GtkWidget *widget, GdkEventExpose *event);

/*****************************************************************************
 *	Initialization / destruction of the widget
 *
 *****************************************************************************/

#define COLOR_BOX_WIDTH_DEFAULT		15
#define COLOR_BOX_HEIGHT_DEFAULT	10

static GtkWidgetClass *par_class = NULL;

static void gtk_color_box_class_init(GtkColorBoxClass *klass);
static void gtk_color_box_init		(GtkColorBox      *term);
static void gtk_color_box_destroy	(GtkObject        *object);

guint gtk_color_box_get_type()
{
	static guint color_box_type = 0;

	if (!color_box_type)
	{
		GtkTypeInfo color_box_info =
		{
			"GtkColorBox",
			sizeof (GtkColorBox),
			sizeof (GtkColorBoxClass),
			(GtkClassInitFunc) gtk_color_box_class_init,
			(GtkObjectInitFunc) gtk_color_box_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL
		};

		color_box_type = gtk_type_unique (gtk_widget_get_type(), &color_box_info);
	}

	return color_box_type;
}

/*
 *	init of new signals and virtual methods
 */
static void gtk_color_box_class_init(GtkColorBoxClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*)klass;
	GtkWidgetClass *widget_class = (GtkWidgetClass*)klass;
	
	/*
	 *	save a copy of parent class structure to keep access to
	 *	its methods
	 */
	par_class = gtk_type_class (gtk_widget_get_type ());

	/*
	 *	defining virtual methods
	 */
	object_class->destroy = gtk_color_box_destroy;

	widget_class->realize = gtk_color_box_realize;
	widget_class->unrealize = gtk_color_box_unrealize;
	widget_class->size_request = gtk_color_box_size_request;
	widget_class->size_allocate = gtk_color_box_size_allocate;
	widget_class->expose_event = gtk_color_box_expose;
}

/*
 *	init of object parameters
 */
static void gtk_color_box_init(GtkColorBox *cb)
{
	cb->width = cb->height = 0;
	
	cb->cmap = NULL;
	cb->gc = NULL;
}

/*
 *	GtkObject destroy method
 */
static void gtk_color_box_destroy (GtkObject *object)
{
	GtkColorBox *cb;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GTK_IS_COLOR_BOX(object));

	cb = GTK_COLOR_BOX (object);
	
	/*
	 *	call widget class destroy method
	 */
	if (GTK_OBJECT_CLASS (par_class)->destroy)
		(* GTK_OBJECT_CLASS (par_class)->destroy)(object);
}

/****************************************************************************
 *	color box widget
 *
 *	public interface
 *
 ****************************************************************************/

GtkWidget* gtk_color_box_new(gint width, gint height, gushort r, gushort g, gushort b)
{
	GtkColorBox *cb;
	
	g_return_val_if_fail(width>0 && height>0,NULL);

	cb = gtk_type_new (gtk_color_box_get_type ());

	cb->width = width;
	cb->height = height;

	cb->color.red	= r<<8;
	cb->color.green	= g<<8;
	cb->color.blue	= b<<8;

	return GTK_WIDGET (cb);
}

void gtk_color_box_set_color(GtkWidget *widget, gushort r, gushort g, gushort b)
{
	GtkColorBox *cb;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_COLOR_BOX(widget));

	cb = GTK_COLOR_BOX(widget);
	gdk_colormap_free_colors(cb->cmap,&(cb->color),1);
	cb->color.red = r<<8;
	cb->color.green = g<<8;
	cb->color.blue = b<<8;
	if (gdk_color_alloc(cb->cmap,&(cb->color))) {
		gdk_gc_set_foreground(cb->gc,&(cb->color));
		gtk_widget_queue_draw(widget);
	} else
		g_warning("the color could not be allocated\n");
}

/****************************************************************************
 *	meta widget
 *
 *	event interface
 *
 ****************************************************************************/

static void gtk_color_box_realize (GtkWidget *widget)
{
	GtkColorBox *cb;
	GdkWindowAttr attributes;
	gint attributes_mask;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_COLOR_BOX(widget));

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
	cb = GTK_COLOR_BOX(widget);

	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.event_mask = gtk_widget_get_events (widget)
			| GDK_EXPOSURE_MASK;

	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = cb->cmap = gtk_widget_get_colormap (widget);

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
				   &attributes, attributes_mask);
	widget->style = gtk_style_attach (widget->style, widget->window);
	gdk_window_set_user_data (widget->window, widget);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

	gdk_window_set_background (widget->window, &widget->style->white);
	
	cb->gc = gdk_gc_new(widget->window);
	gdk_color_alloc(cb->cmap,&(cb->color));
	
	gdk_gc_set_background(cb->gc,&(widget->style->white));
	gdk_gc_set_foreground(cb->gc,&(cb->color));
}

static void gtk_color_box_unrealize (GtkWidget *widget)
{
	GtkColorBox *cb;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_COLOR_BOX(widget));

	GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED);
	
	cb = GTK_COLOR_BOX(widget);
	
	gdk_gc_destroy(cb->gc);
	gdk_colormap_free_colors(cb->cmap,&(cb->color),1);
	
	if (GTK_WIDGET_CLASS (par_class)->unrealize)
		(*GTK_WIDGET_CLASS (par_class)->unrealize) (widget);
}

/*
 *	size_request : set the minimum widget size
 */
static void gtk_color_box_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	GtkColorBox *cb;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_COLOR_BOX(widget));
	g_return_if_fail (requisition != NULL);

	cb = GTK_COLOR_BOX(widget);
	
	if (cb->width && cb->height) {
		requisition->width = cb->width;
		requisition->height = cb->height;
	} else {
		requisition->width = COLOR_BOX_WIDTH_DEFAULT;
		requisition->height = COLOR_BOX_HEIGHT_DEFAULT;
	}
}

/*
 *	size_allocate : set the actual widget size
 */
static void gtk_color_box_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_COLOR_BOX(widget));
	g_return_if_fail (allocation != NULL);

	widget->allocation = *allocation;

	if (GTK_WIDGET_REALIZED(widget))
	{
		gdk_window_move_resize(widget->window,
							allocation->x,
							allocation->y,
							allocation->width,
							allocation->height);
	}
}

static gint gtk_color_box_expose(GtkWidget *widget, GdkEventExpose *event)
{
//	if (event->count==0) {
		gdk_draw_rectangle(widget->window,GTK_COLOR_BOX(widget)->gc,TRUE,0,0,-1,-1);
//	}
	return FALSE;
}


/***************************************************************************/
static void col_but_destroy(GtkObject *object);
static void col_but_clicked(GtkButton *button);
static void col_but_state_changed(GtkWidget *widget, GtkStateType previous_state);

static void col_but_dialog_ok(GtkWidget *widget, gpointer data);
static void col_but_dialog_cancel(GtkWidget *widget, gpointer data);

enum
{
	COLOR_CHANGED,
	LAST_SIGNAL
};

static guint col_but_signals[LAST_SIGNAL] = { 0 };

static GtkButtonClass *parent_class = NULL;

/*************************************************************************/
static void col_but_destroy(GtkObject *object)
{
	ColBut *cb;
	
	g_return_if_fail(cb = COL_BUT(object));
			
	if (cb->dialog)
		gtk_widget_destroy(cb->dialog);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(* GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}

/*************************************************************************/
static void col_but_class_init(ColButClass *class)
{
	GtkObjectClass *object_class;
	GtkButtonClass *button_class;
	GtkWidgetClass *widget_class;

	object_class = (GtkObjectClass*) class;
	button_class = (GtkButtonClass*) class;
	widget_class = (GtkWidgetClass*) class;

	parent_class = gtk_type_class(gtk_button_get_type());

	col_but_signals[COLOR_CHANGED] = gtk_signal_new("color_changed",
						GTK_RUN_FIRST,
						object_class->type,
						GTK_SIGNAL_OFFSET(ColButClass, color_changed),
						gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
	gtk_object_class_add_signals(object_class, col_but_signals, LAST_SIGNAL);
	class->color_changed = NULL;

	object_class->destroy       = col_but_destroy;
	button_class->clicked       = col_but_clicked;
	widget_class->state_changed = col_but_state_changed;
}

/*************************************************************************/
static void col_but_init(ColBut *cb)
{
	cb->cbox	= NULL;
	cb->dialog	= NULL;
}

/*************************************************************************
*
**************************************************************************/
GtkType col_but_get_type(void)
{
	static guint cb_type = 0;

	if (!cb_type) {
		GtkTypeInfo cb_info = {
			"GtkColorButton",
			sizeof(ColBut),
			sizeof(ColButClass),
			(GtkClassInitFunc)col_but_class_init,
			(GtkObjectInitFunc)col_but_init,
			NULL,
			NULL,
			(GtkClassInitFunc) NULL
		};

		cb_type = gtk_type_unique(gtk_button_get_type(), &cb_info);
	}

	return cb_type;
}

/*************************************************************************/
GtkWidget * col_but_new(gint width, gint height, gushort r, gushort g, gushort b)
{
	ColBut *cb;
  
	g_return_val_if_fail(width > 0 && height > 0, NULL);  

	cb = gtk_type_new(col_but_get_type());

	cb->color[0] = r;
	cb->color[1] = g;
	cb->color[2] = b;
	cb->fcolor[0] = (gdouble)r / 255.0;
	cb->fcolor[1] = (gdouble)g / 255.0;
	cb->fcolor[2] = (gdouble)b / 255.0;

	cb->cbox = gtk_color_box_new(width,height,r,g,b);

	gtk_container_add(GTK_CONTAINER(cb), cb->cbox);
	 
	return (GTK_WIDGET(cb));
}

void col_but_get_color(GtkWidget *widget, gushort color[3])
{
	ColBut *cb;
	int i;

	g_return_if_fail (IS_COL_BUT(widget));
	cb = COL_BUT(widget);

	for (i=0 ; i<3 ; i++)
		color[i]=cb->color[i];
}

void col_but_set_color(GtkWidget *widget, gushort r, gushort g, gushort b)
{
	ColBut *cb;

	g_return_if_fail (IS_COL_BUT(widget));
	cb = COL_BUT(widget);

	cb->color[0] = r;
	cb->color[1] = g;
	cb->color[2] = b;
	cb->fcolor[0] = (gdouble)r / 255.0;
	cb->fcolor[1] = (gdouble)g / 255.0;
	cb->fcolor[2] = (gdouble)b / 255.0;

	gtk_color_box_set_color(cb->cbox,r,g,b);
}

/*************************************************************************/
static void col_but_state_changed(GtkWidget *widget, GtkStateType previous_state)
{  
	g_return_if_fail (IS_COL_BUT(widget));

	if (!GTK_WIDGET_IS_SENSITIVE(widget) && COL_BUT(widget)->dialog)
		gtk_widget_hide(COL_BUT(widget)->dialog);

	if (GTK_WIDGET_CLASS(parent_class)->state_changed)
		(* GTK_WIDGET_CLASS(parent_class)->state_changed)(widget, previous_state);
}

/*************************************************************************/
static void col_but_clicked(GtkButton *button)
{
	ColBut *cb;
	GtkColorSelection *colorsel;

	g_return_if_fail (IS_COL_BUT(button));
	cb = COL_BUT(button);

	if (!cb->dialog) {
		cb->dialog = gtk_color_selection_dialog_new("Choose a color");
		colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(cb->dialog)->colorsel);
		gtk_color_selection_set_color(colorsel, cb->fcolor);
		gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(cb->dialog)->help_button);
		gtk_signal_connect(GTK_OBJECT(cb->dialog),"destroy",(GtkSignalFunc)gtk_widget_destroyed,&cb->dialog);
		gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(cb->dialog)->ok_button),"clicked",(GtkSignalFunc)col_but_dialog_ok, cb);
		gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(cb->dialog)->cancel_button),"clicked",(GtkSignalFunc)col_but_dialog_cancel, cb);
		gtk_window_position(GTK_WINDOW(cb->dialog), GTK_WIN_POS_MOUSE);  
	}
	gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(cb->dialog)->colorsel),cb->fcolor);
	gtk_widget_show(cb->dialog);
}

/*************************************************************************/
static void col_but_dialog_ok(GtkWidget *widget, gpointer data)
{
	ColBut *cb;
	gushort new_color[3];
	gint i;

	g_return_if_fail(IS_COL_BUT(data));
	cb = COL_BUT(data);

	gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(cb->dialog)->colorsel), cb->fcolor);
	gtk_widget_hide(cb->dialog);

	for (i=0 ; i<3 ; i++)
		new_color[i] = (gushort)(cb->fcolor[i] * 255.0);
	
	if (new_color[0]!=cb->color[0] || new_color[1]!=cb->color[1] || new_color[2]!=cb->color[2]) {
		cb->color[0] = new_color[0];
		cb->color[1] = new_color[1];
		cb->color[2] = new_color[2];
		gtk_color_box_set_color(cb->cbox,cb->color[0],cb->color[1],cb->color[2]);
		gtk_signal_emit(GTK_OBJECT(cb),col_but_signals[COLOR_CHANGED]);
	}
}

/*************************************************************************/
static void col_but_dialog_cancel(GtkWidget *widget, gpointer data)
{
	ColBut *cb;
  
	g_return_if_fail(IS_COL_BUT(data));
	cb = COL_BUT(data);

	gtk_widget_hide(cb->dialog);
}

