/*
firetalk.c - FireTalk wrapper definitions
Copyright (C) 2000 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>

#define FIRETALK

#include "firetalk.h"
#include "firetalk-int.h"
#include "toc.h"
#include "irc.h"

typedef void (*ptrtotoc)(void *, ...);

/* Global variables */
int firetalkerror;
static struct s_firetalk_handle *handle_head = NULL;
const char *firetalkerrors[] = {
/*  0 */	"Success",
/*  1 */	"Connection failed",
/*  2 */	"Usernames do not match",
/*  3 */	"Packet transfer error",
/*  4 */	"Invalid username or password",
/*  5 */	"Invalid sequence number from server",
/*  6 */	"Invalid frame type from server",
/*  7 */	"Packet too long",
/*  8 */	"Server problem; try again later",
/*  9 */	"Unknown error",
/* 10 */	"You are blocked",
/* 11 */	"Unknown packet received from server",
/* 12 */	"Invalid callback number",
/* 13 */	"Invalid username",
/* 14 */	"Username not found in list",
/* 15 */	"Server disconnected",
/* 16 */	"Unable to create socket",
/* 17 */	"Unable to resolve hostname",
/* 18 */	"Wrong server version",
/* 19 */	"User is currently unavailable",
/* 20 */	"User information is currently unavilable",
/* 21 */	"You are sending messages too fast; last message was dropped",
/* 22 */	"Chat room is currently unavailable",
/* 23 */	"Incoming message delivery failure",
/* 24 */	"User disconnected",
/* 25 */	"Server response was formatted incorrectly",
/* 26 */  "You have requested idle to be reset too fast",
/* 27 */  "Invalid room name",
/* 28 */  "Invalid message (too long?)",
/* 29 */  "Invalid protocol",
/* 30 */  "Not connected",
/* 31 */  "Invalid connection number",
/* 32 */  "No permission to perform operation"
};

const char *firetalkprotocols[] = {
/*  0 */  "AIM/TOC",
/*  1 */  "IRC"
};

/* Internal function definitions */

/* firetalk_find_by_toc searches the firetalk handle list for the toc handle passed, and returns the firetalk handle */
struct s_firetalk_handle *firetalk_find_handle(void * const c) {
	struct s_firetalk_handle *iter;
	iter = handle_head;
	while (iter) {
		if (iter->handle == c)
			return iter;
		iter = iter->next;
	}
	return NULL;
}

int firetalk_im_internal_add_buddy(void * const c, const char * const nickname) {
	struct s_firetalk_buddy *iter;
	struct s_firetalk_handle *conn;

	conn = c;

	iter = conn->buddy_head;
	while (iter) {
		if (!conn->comparenicks(iter->nickname,nickname)) {
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS; /* not an error, user is in buddy list */
		}
		iter = iter->next;
	}

	iter = conn->buddy_head;
	conn->buddy_head = malloc(sizeof(struct s_firetalk_buddy));
	if (!conn->buddy_head) {
		perror("malloc");
		exit(1);
	}
	conn->buddy_head->next = iter;
	conn->buddy_head->nickname = malloc(strlen(nickname)+1);
	if (!conn->buddy_head->nickname) {
		perror("malloc");
		exit(1);
	}
	strcpy(conn->buddy_head->nickname,nickname);
	conn->buddy_head->online = 0;
  conn->buddy_head->away = 0;
	conn->buddy_head->idletime = 0;
	conn->buddy_head->tempint = 0;
	conn->buddy_head->tempint2 = 0;
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_im_internal_add_deny(void * const c, const char * const nickname) {
	struct s_firetalk_deny *iter;
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;

	conn = c;

	iter = conn->deny_head;
	while (iter) {
		if (!conn->comparenicks(iter->nickname,nickname)) {
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS; /* not an error, user is in buddy list */
		}
		iter = iter->next;
	}

	iter = conn->deny_head;
	conn->deny_head = malloc(sizeof(struct s_firetalk_deny));
	if (!conn->deny_head) {
		perror("malloc");
		exit(1);
	}
	conn->deny_head->next = iter;
	conn->deny_head->nickname = malloc(strlen(nickname)+1);
	if (!conn->deny_head->nickname) {
		perror("malloc");
		exit(1);
	}
	strcpy(conn->deny_head->nickname,nickname);

	buddyiter = conn->buddy_head;
	while (buddyiter) {
		if (!conn->comparenicks(buddyiter->nickname,nickname)) {
			if (buddyiter->online && conn->callbacks[FC_IM_BUDDYOFFLINE])
				conn->callbacks[FC_IM_BUDDYOFFLINE](conn,buddyiter->nickname);
		}
		buddyiter = buddyiter->next;
	}

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_chat_internal_add_room(void * const c, const char * const name) {
	struct s_firetalk_room *iter;
	struct s_firetalk_handle *conn;

	conn = c;

	iter = conn->room_head;
	while (iter) {
		if (!conn->comparenicks(iter->name,name)) {
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS; /* not an error, we're already in room */
		}
		iter = iter->next;
	}

	iter = conn->room_head;
	conn->room_head = malloc(sizeof(struct s_firetalk_room));
	if (!conn->room_head) {
		perror("malloc");
		exit(1);
	}
	conn->room_head->next = iter;
	conn->room_head->name = malloc(strlen(name)+1);
	if (!conn->room_head->name) {
		perror("malloc");
		exit(1);
	}
	strcpy(conn->room_head->name,name);
	conn->room_head->member_head = NULL;

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_chat_internal_add_member(void * const c, const char * const room, const char * const nickname) {
	struct s_firetalk_room *iter;
	struct s_firetalk_room *roomhandle;
	struct s_firetalk_handle *conn;
	struct s_firetalk_member *memberiter;

	conn = c;

	iter = conn->room_head;
	roomhandle = NULL;
	while (iter && !roomhandle) {
		if (!conn->comparenicks(iter->name,room))
			roomhandle = iter;
		iter = iter->next;
	}

	if (!roomhandle) { /* we don't know about that room */
		firetalkerror = FE_NOTFOUND;
		return FE_NOTFOUND;
	}

	memberiter = roomhandle->member_head;
	while (memberiter) {
		if (!conn->comparenicks(memberiter->nickname,nickname)) {
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		memberiter = memberiter->next;
	}

	memberiter = roomhandle->member_head;
	roomhandle->member_head = malloc(sizeof(struct s_firetalk_member));
	if (!roomhandle->member_head) {
		perror("malloc");
		exit(1);
	}
	roomhandle->member_head->next = memberiter;
	roomhandle->member_head->nickname = malloc(strlen(nickname)+1);
	if (!roomhandle->member_head->nickname) {
		perror("malloc");
		exit(1);
	}
	strcpy(roomhandle->member_head->nickname,nickname);

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_im_internal_remove_buddy(void * const c, const char * const nickname) {
	struct s_firetalk_buddy *iter;
	struct s_firetalk_buddy *iter2;
	struct s_firetalk_handle *conn;

	conn = c;

	iter2 = NULL;
	iter = conn->buddy_head;
	while (iter) {
		if (!conn->comparenicks(nickname,iter->nickname)) {
			if (iter2)
				iter2->next = iter->next;
			else
				conn->buddy_head = iter->next;
			free(iter);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		iter2 = iter;
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return FE_NOTFOUND;
}

int firetalk_im_internal_remove_deny(void * const c, const char * const nickname) {
	struct s_firetalk_deny *iter;
	struct s_firetalk_deny *iter2;
	struct s_firetalk_handle *conn;

	conn = c;

	iter2 = NULL;
	iter = conn->deny_head;
	while (iter) {
		if (!conn->comparenicks(nickname,iter->nickname)) {
			if (iter2)
				iter2->next = iter->next;
			else
				conn->deny_head = iter->next;
			free(iter);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		iter2 = iter;
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return FE_NOTFOUND;
}

int firetalk_chat_internal_remove_room(void * const c, const char * const name) {
	struct s_firetalk_room *iter;
	struct s_firetalk_room *iter2;
	struct s_firetalk_handle *conn;
	struct s_firetalk_member *memberiter;
	struct s_firetalk_member *memberiter2;

	conn = c;

	iter2 = NULL;
	iter = conn->room_head;
	while (iter) {
		if (!conn->comparenicks(name,iter->name)) {
			memberiter = iter->member_head;
			while (memberiter) {
				memberiter2 = memberiter->next;
				free(memberiter->nickname);
				free(memberiter);
				memberiter = memberiter2;
			}
			if (iter2)
				iter2->next = iter->next;
			else
				conn->room_head = iter->next;
			free(iter);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		iter2 = iter;
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return FE_NOTFOUND;
}

int firetalk_chat_internal_remove_member(void * const c, const char * const room, const char * const nickname) {
	struct s_firetalk_room *iter;
	struct s_firetalk_room *roomhandle;
	struct s_firetalk_handle *conn;
	struct s_firetalk_member *memberiter;
	struct s_firetalk_member *memberiter2;

	conn = c;

	iter = conn->room_head;
	roomhandle = NULL;
	while (iter && !roomhandle) {
		if (!conn->comparenicks(iter->name,room))
			roomhandle = iter;
		iter = iter->next;
	}

	if (!roomhandle) { /* we don't know about that room */
		firetalkerror = FE_NOTFOUND;
		return FE_NOTFOUND;
	}

	memberiter2 = NULL;
	memberiter = roomhandle->member_head;
	while (memberiter) {
		if (!conn->comparenicks(memberiter->nickname,nickname)) {
			if (memberiter2)
				memberiter2->next = memberiter->next;
			else
				roomhandle->member_head = memberiter->next;
		}
		memberiter2 = memberiter;
		memberiter = memberiter->next;
	}

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

void firetalk_callback_im_getmessage(void * const c, const char * const sender, const int automessage, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_IM_GETMESSAGE])
		conn->callbacks[FC_IM_GETMESSAGE](conn,sender,automessage,message);
}

void firetalk_callback_im_getaction(void * const c, const char * const sender, const int automessage, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_IM_GETACTION])
		conn->callbacks[FC_IM_GETACTION](conn,sender,automessage,message);
}

void firetalk_callback_im_buddyonline(void * const c, const char * const nickname, const int online) {
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;

	conn = firetalk_find_handle(c);

	buddyiter = conn->buddy_head;
	while (buddyiter) {
		if (!conn->comparenicks(buddyiter->nickname,nickname)) {
			/* match */
			if (buddyiter->online != online) {
				buddyiter->online = online;
				if (online && conn->callbacks[FC_IM_BUDDYONLINE])
					conn->callbacks[FC_IM_BUDDYONLINE](conn,buddyiter->nickname);
				else if (!online && conn->callbacks[FC_IM_BUDDYOFFLINE])
					conn->callbacks[FC_IM_BUDDYOFFLINE](conn,buddyiter->nickname);
				return;
			}
		}
		buddyiter = buddyiter->next;
	}
	return;
}

void firetalk_callback_im_buddyaway(void * const c, const char * const nickname, const int away) {
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;

	conn = firetalk_find_handle(c);

	buddyiter = conn->buddy_head;
	while (buddyiter) {
		if (!conn->comparenicks(buddyiter->nickname,nickname)) {
			/* match */
			if (buddyiter->away != away && buddyiter->online) {
				buddyiter->away = away;
				if (away && conn->callbacks[FC_IM_BUDDYAWAY])
					conn->callbacks[FC_IM_BUDDYAWAY](conn,buddyiter->nickname);
				else if (!away && conn->callbacks[FC_IM_BUDDYUNAWAY])
					conn->callbacks[FC_IM_BUDDYUNAWAY](conn,buddyiter->nickname);
				return;
			}
		}
		buddyiter = buddyiter->next;
	}
	return;
}

void firetalk_callback_error(void * const c, const int error, const char * const roomoruser) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_ERROR])
		conn->callbacks[FC_ERROR](conn,error,roomoruser);
}

void firetalk_callback_disconnect(void * const c, const int error) {
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;
	struct s_firetalk_buddy *buddyiter2;
	struct s_firetalk_deny *denyiter;
	struct s_firetalk_deny *denyiter2;

	conn = firetalk_find_handle(c);

	if (!conn)
		return;

	conn->handle = NULL;

	buddyiter = conn->buddy_head;
	while (buddyiter) {
		buddyiter2 = buddyiter;
		buddyiter = buddyiter->next;
		free(buddyiter2);
	}
	conn->buddy_head = NULL;

	denyiter = conn->deny_head;
	while (denyiter) {
		denyiter2 = denyiter;
		denyiter = denyiter->next;
		free(denyiter2);
	}
	conn->deny_head = NULL;

	if (conn->callbacks[FC_DISCONNECT])
		conn->callbacks[FC_DISCONNECT](conn,error);
}

void firetalk_callback_gotinfo(void * const c, const char * const nickname, const char * const info, const int warning, const int idle, const int flags) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_GOTINFO])
		conn->callbacks[FC_GOTINFO](conn,nickname,info,warning,idle,flags);
}

void firetalk_callback_idleinfo(void * const c, char const * const nickname, const long idletime) {
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;

	conn = firetalk_find_handle(c);

	if (!conn->callbacks[FC_IDLEINFO])
		return;

	buddyiter = conn->buddy_head;
	while (buddyiter) {
		if (!conn->comparenicks(buddyiter->nickname,nickname)) {
			/* match */
			if (buddyiter->idletime != idletime && buddyiter->online) {
				buddyiter->idletime = idletime;
				conn->callbacks[FC_IDLEINFO](conn,buddyiter->nickname,idletime);
				return;
			}
		}
		buddyiter = buddyiter->next;
	}
	return;
}

void firetalk_callback_doinit(void * const c) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_DOINIT])
		conn->callbacks[FC_DOINIT](conn);
}

void firetalk_callback_setidle(void * const c, long * const idle) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_SETIDLE])
		conn->callbacks[FC_SETIDLE](conn,idle);
}

void firetalk_callback_eviled(void * const c, const int newevil, const char * const eviler) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_EVILED])
		conn->callbacks[FC_EVILED](conn,newevil,eviler);
}

void firetalk_callback_chat_joined(void * const c, const char * const room) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	firetalk_chat_internal_add_room(conn,room);
	if (conn->callbacks[FC_CHAT_JOINED])
		conn->callbacks[FC_CHAT_JOINED](conn,room);
}

void firetalk_callback_chat_left(void * const c, const char * const room) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	firetalk_chat_internal_remove_room(conn,room);
	if (conn->callbacks[FC_CHAT_LEFT])
		conn->callbacks[FC_CHAT_LEFT](conn,room);
}

void firetalk_callback_chat_getmessage(void * const c, const char * const room, const char * const from, const int automessage, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_CHAT_GETMESSAGE])
		conn->callbacks[FC_CHAT_GETMESSAGE](conn,room,from,automessage,message);
}

void firetalk_callback_chat_getaction(void * const c, const char * const room, const char * const from, const int automessage, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_CHAT_GETACTION])
		conn->callbacks[FC_CHAT_GETACTION](conn,room,from,automessage,message);
}

void firetalk_callback_chat_invited(void * const c, const char * const room, const char * const from, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_CHAT_INVITED])
		conn->callbacks[FC_CHAT_INVITED](conn,room,from,message);
}

void firetalk_callback_chat_user_joined(void * const c, const char * const room, const char * const who) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	firetalk_chat_internal_add_member(conn,room,who);
	if (conn->callbacks[FC_CHAT_USER_JOINED])
		conn->callbacks[FC_CHAT_USER_JOINED](conn,room,who);
}

void firetalk_callback_chat_user_left(void * const c, const char * const room, const char * const who) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	firetalk_chat_internal_remove_member(conn,room,who);
	if (conn->callbacks[FC_CHAT_USER_LEFT])
		conn->callbacks[FC_CHAT_USER_LEFT](conn,room,who);
}

void firetalk_callback_chat_user_quit(void * const c, const char * const who) {
	struct s_firetalk_handle *conn;
	struct s_firetalk_room *roomiter;
	struct s_firetalk_member *memberiter;
	struct s_firetalk_member *memberiter2;
	conn = firetalk_find_handle(c);
	
	roomiter = conn->room_head;
	while (roomiter) {
		memberiter = roomiter->member_head;
		while (memberiter) {
			memberiter2 = memberiter->next;
			if (!conn->comparenicks(memberiter->nickname,who))
				firetalk_callback_chat_user_left(c,roomiter->name,memberiter->nickname);
			memberiter = memberiter2;
		}
		roomiter = roomiter->next;
	}
}

void firetalk_callback_chat_gottopic(void * const c, const char * const room, const char * const topic, const char * const author) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_CHAT_GOTTOPIC])
		conn->callbacks[FC_CHAT_GOTTOPIC](conn,room,topic,author);
}

void firetalk_callback_chat_user_opped(void * const c, const char * const room, const char * const who, const char * const by) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_CHAT_USER_OPPED])
		conn->callbacks[FC_CHAT_USER_OPPED](conn,room,who,by);
}

void firetalk_callback_chat_user_deopped(void * const c, const char * const room, const char * const who, const char * const by) {
	struct s_firetalk_handle *conn;
	conn = firetalk_find_handle(c);
	if (conn->callbacks[FC_CHAT_USER_DEOPPED])
		conn->callbacks[FC_CHAT_USER_DEOPPED](conn,room,who,by);
}

/* External function definitions */
void *firetalk_create_handle(const int protocol) {
	struct s_firetalk_handle *c;
	if (protocol < 0 || protocol >= FP_MAX) {
		firetalkerror = FE_BADPROTO;
		return NULL;
	}
	c = handle_head;
	handle_head = malloc(sizeof(struct s_firetalk_handle));
	handle_head->prev = NULL;
	handle_head->next = c;
	handle_head->handle = NULL;
	handle_head->buddy_head = NULL;
	handle_head->deny_head = NULL;
	handle_head->room_head = NULL;
	handle_head->protocol = protocol;
	switch (protocol) {
		case FP_AIMTOC:
			handle_head->comparenicks = toc_compare_nicks;
			break;
		case FP_IRC:
			handle_head->comparenicks = irc_compare_nicks;
			break;
		default:
			handle_head->comparenicks = NULL;
			break;
	}
	return handle_head;
}

void firetalk_destroy_handle(void * const c) {
	struct s_firetalk_handle *conn;
	conn = c;
	if (conn->prev)
		conn->prev->next = conn->next;
	else
		handle_head = conn->next;
	if (conn->next)
		conn->next->prev = conn->prev;
	if (conn->handle) {
		switch (conn->protocol) {
			case FP_AIMTOC:
				toc_disconnect(conn->handle);
				break;
			case FP_IRC:
				irc_disconnect(conn->handle);
				break;
		}
	}
	free(conn);
}

void firetalk_perror(const char * const string) {
	fprintf(stderr,"%s: %s\n",string,firetalkerrors[firetalkerror]);
}

int firetalk_disconnect(void * const c) {
	struct s_firetalk_handle *conn;
	int i;

	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			i = toc_disconnect(conn->handle);
			break;
		case FP_IRC:
			i = irc_disconnect(conn->handle);
			break;
		default:
			i = FE_SUCCESS;
			break;
	}

	conn->handle = NULL;
	return i;
}

int firetalk_signon(void * const c, const char * const server, const int port, const char * const username, const char * const password) {
	struct s_firetalk_handle *conn;
	int i;

	conn = c;
	switch (conn->protocol) {
		case FP_AIMTOC:
			if (! (conn->handle = toc_connect_server(server,port)))
				return firetalkerror;
			/* register ourselves for the disconnect event so we can clean up */
			if ((i = toc_signon(conn->handle,username,password))) {
				conn->handle = NULL;
				firetalkerror = i;
				return i;
			}
			break;
		case FP_IRC:
			if (! (conn->handle = irc_connect_server(server,port,username,password)))
				return firetalkerror;
			firetalk_callback_doinit(conn->handle);
			break;
	}
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_get_protocol(void * const c) {
	return ((struct s_firetalk_handle *)c)->protocol;
}

int firetalk_register_callback(void * const c, const int type, void (*function)(void *, ...)) {
	struct s_firetalk_handle *conn;
	conn = c;
	/* here is a translation table from FIRETALK to the sub modules */
	if (type < 0 || type > FC_MAX) {
		firetalkerror = FE_CALLBACKNUM;
		return FE_CALLBACKNUM;
	}
	conn->callbacks[type] = function;
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_im_add_buddy(void * const c, const char * const nickname) {
	struct s_firetalk_handle *conn;
	int ret;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			ret = toc_im_add_buddy(conn->handle,nickname);
			break;
		case FP_IRC:
			ret = irc_im_add_buddy(conn->handle,nickname);
			break;
		default:
			ret = FE_SUCCESS;
			break;
	}

	if (ret != FE_SUCCESS) {
		firetalkerror = ret;
		return ret;
	}
	
	return firetalk_im_internal_add_buddy(conn,nickname);
}

int firetalk_im_remove_buddy(void * const c, const char * const nickname) {
	struct s_firetalk_handle *conn;
	int ret;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			ret = toc_im_remove_buddy(conn->handle,nickname);
			break;
		default:
			ret = FE_SUCCESS;
			break;
	}

	if (ret != FE_SUCCESS) {
		firetalkerror = ret;
		return ret;
	}

	return firetalk_im_internal_remove_buddy(conn,nickname);
}

int firetalk_im_add_deny(void * const c, const char * const nickname) {
	struct s_firetalk_handle *conn;
	int ret;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			ret = toc_im_add_deny(conn->handle,nickname);
			break;
		default:
			ret = FE_SUCCESS;
			break;
	}

	if (ret != FE_SUCCESS) {
		firetalkerror = ret;
		return ret;
	}

	return firetalk_im_internal_add_deny(conn,nickname);
}

int firetalk_im_remove_deny(void * const c, const char * const nickname) {
	struct s_firetalk_handle *conn;
	int ret;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			ret = toc_im_remove_deny(conn->handle,nickname);
			break;
		default:
			ret = FE_SUCCESS;
			break;
	}

	if (ret != FE_SUCCESS) {
		firetalkerror = ret;
		return ret;
	}

	return firetalk_im_internal_add_deny(conn,nickname);
}

int firetalk_save_config(void * const c) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_save_config(conn->handle);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_im_upload_buddies(void * const c) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_im_upload_buddies(conn->handle);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_im_upload_denies(void * const c) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_im_upload_denies(conn->handle);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_im_send_message(void * const c, const char * const dest, const char * const message, const int auto_flag) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_im_send_message(conn->handle,dest,message,auto_flag);
			break;
		case FP_IRC:
			return irc_im_send_message(conn->handle,dest,message,auto_flag);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_im_send_action(void * const c, const char * const dest, const char * const message, const int auto_flag) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_im_send_action(conn->handle,dest,message,auto_flag);
			break;
		case FP_IRC:
			return irc_im_send_action(conn->handle,dest,message,auto_flag);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_get_info(void * const c, const char * const nickname) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_get_info(conn->handle,nickname);
			break;
		case FP_IRC:
			return irc_get_info(conn->handle,nickname);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_set_info(void * const c, const char * const info) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_set_info(conn->handle,info);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_im_list_buddies(void * const c) {
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	if (!conn->callbacks[FC_IM_LISTBUDDY]) {
		firetalkerror = FE_SUCCESS;
		return FE_SUCCESS;
	}

	buddyiter = conn->buddy_head;
	while (buddyiter) {
		conn->callbacks[FC_IM_LISTBUDDY](conn,buddyiter->nickname,buddyiter->online,buddyiter->away);
		buddyiter = buddyiter->next;
	}

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int firetalk_set_away(void * const c, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_set_away(conn->handle,message);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_im_evil(void * const c, const char * const who) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_im_evil(conn->handle,who);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_join(void * const c, const char * const room) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_chat_join(conn->handle,room);
			break;
		case FP_IRC:
			return irc_chat_join(conn->handle,room);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_part(void * const c, const char * const room) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_chat_part(conn->handle,room);
			break;
		case FP_IRC:
			return irc_chat_part(conn->handle,room);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_send_message(void * const c, const char * const room, const char * const message, const int auto_flag) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_chat_send_message(conn->handle,room,message);
			break;
		case FP_IRC:
			return irc_chat_send_message(conn->handle,room,message,auto_flag);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_send_action(void * const c, const char * const room, const char * const message, const int auto_flag) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_chat_send_action(conn->handle,room,message);
			break;
		case FP_IRC:
			return irc_chat_send_action(conn->handle,room,message,auto_flag);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_invite(void * const c, const char * const room, const char * const who, const char * const message) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_AIMTOC:
			return toc_chat_invite(conn->handle,room,who,message);
			break;
		case FP_IRC:
			return irc_chat_invite(conn->handle,room,who);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_set_topic(void * const c, const char * const room, const char * const topic) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_IRC:
			return irc_chat_set_topic(conn->handle,room,topic);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_op(void * const c, const char * const room, const char * const who) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_IRC:
			return irc_chat_op(conn->handle,room,who);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_chat_deop(void * const c, const char * const room, const char * const who) {
	struct s_firetalk_handle *conn;
	conn = c;

	if (!conn->handle) {
		firetalkerror = FE_NOTCONNECTED;
		return FE_NOTCONNECTED;
	}

	switch (conn->protocol) {
		case FP_IRC:
			return irc_chat_deop(conn->handle,room,who);
			break;
		default:
			return FE_SUCCESS;
			break;
	}
}

int firetalk_select() {
	return firetalk_select_custom(0,NULL,NULL,NULL,NULL);
}

int firetalk_select_custom(int n, fd_set *fd_read, fd_set *fd_write, fd_set *fd_except, struct timeval *timeout) {
	int ret;
	int tempsock;
	fd_set *my_read;
	fd_set *my_write;
	fd_set *my_except;
	fd_set internal_read;
	fd_set internal_write;
	fd_set internal_except;
	struct timeval internal_timeout;
	struct timeval *my_timeout;
	struct s_firetalk_handle *fchandle;

	my_read = fd_read;
	my_write = fd_write;
	my_except = fd_except;
	my_timeout = timeout;

	if (!my_read) {
		my_read = &internal_read;
		FD_ZERO(my_read);
	}

	if (!my_write) {
		my_write = &internal_write;
		FD_ZERO(my_write);
	}

	if (!my_except) {
		my_except = &internal_except;
		FD_ZERO(my_except);
	}

	if (!my_timeout) {
		my_timeout = &internal_timeout;
		my_timeout->tv_sec = 60;
		my_timeout->tv_usec = 0;
	}

	if (my_timeout->tv_sec > 60)
		my_timeout->tv_sec = 60;

	fchandle = handle_head;
	while (fchandle) {
		if (!fchandle->handle) {
			fchandle = fchandle->next;
			continue;
		}
		switch (fchandle->protocol) {
			case FP_AIMTOC:
				tempsock = toc_get_fd(fchandle->handle);
				toc_set_idle(fchandle->handle);
				break;
			case FP_IRC:
				tempsock = irc_get_fd(fchandle->handle);
				irc_check_buddies(fchandle);
				break;
			default:
				tempsock = -1;
				break;
		}
		if (tempsock >= n)
			n = tempsock + 1;
		FD_SET(tempsock,my_read);
		FD_SET(tempsock,my_except);
		fchandle = fchandle->next;
	}

	ret = select(n,my_read,my_write,my_except,my_timeout);

	fchandle = handle_head;
	while (fchandle) {
		if (!fchandle->handle) {
			fchandle = fchandle->next;
			continue;
		}
		switch (fchandle->protocol) {
			case FP_AIMTOC:
				tempsock = toc_get_fd(fchandle->handle);
				if (FD_ISSET(tempsock,my_read)) {
					ret--;
					toc_got_data(fchandle->handle);
				}
				if (FD_ISSET(tempsock,my_except)) {
					ret--;
					toc_disconnect(fchandle->handle);
				}
				break;
			case FP_IRC:
				tempsock = irc_get_fd(fchandle->handle);
				if (FD_ISSET(tempsock,my_read)) {
					ret--;
					irc_got_data(fchandle->handle);
				}
				if (FD_ISSET(tempsock,my_except)) {
					ret--;
					irc_disconnect(fchandle->handle);
				}
				break;
		}
		fchandle = fchandle->next;
	}
	return ret;
}
