/*
irc.c - FireTalk IRC protocol definitions
Copyright (C) 2000 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#define _GNU_SOURCE
#include <sys/socket.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <strings.h>
#include <sys/time.h>

#include "firetalk.h"
#include "firetalk-int.h"
#include "irc.h"

#define IRC_SERVER "irc.openprojects.net"
#define IRC_PORT 6667

typedef struct s_irc_whois {
	struct s_irc_whois *next;
	char *nickname;
	char *info;
	int flags;
	long idle;
} s_irc_whois;

typedef struct s_irc_connection {
	int s;
	char *nickname;
	char *password;
	char buffer[513];
	time_t lasttime;                                     /* last time we set idles */
	int isons;                                           /* number of ISON's we're waiting on replies to */
	struct s_irc_whois *whois_head;
} s_irc_connection;

static int irc_send_printf(struct s_irc_connection * const c, const char * const format, ...);
static int irc_internal_disconnect(struct s_irc_connection * const c, const int error);
static char **irc_recv_parse(struct s_irc_connection * const c);

/*

from buddy struction:

tempint1 = ison, default 0, used during ISON checks
tempint2 = away, default 0, used during WHOIS checks

*/

static int irc_internal_disconnect(struct s_irc_connection * const c, const int error) {
	shutdown(c->s, SHUT_RDWR);
	if (c->nickname)
		free(c->nickname);
	if (c->password)
		free(c->password);
	free(c);
	firetalk_callback_disconnect(c,error);
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

static int irc_send_printf(struct s_irc_connection * const c, const char * const format, ...) {
	va_list ap;
	int len;
	int i;
	char data[513];
	short datai = 0;
	char *tempchr;

	va_start(ap,format);

	len = strlen(format);
	for (i = 0; i < len; i++) {
		if (format[i] == '%') {
			switch (format[++i]) {
				case 's':
					tempchr = va_arg(ap, char *);
					if (datai + strlen(tempchr) > 509) {
						firetalkerror = FE_PACKETSIZE;
						return FE_PACKETSIZE;
					}
					strcpy(&data[datai],tempchr);
					datai += strlen(tempchr);
					break;
				case '%':
					data[datai++] = '%';
					break;
			}
		} else {
			data[datai++] = format[i];
			if (datai > 509) {
				firetalkerror = FE_PACKETSIZE;
				return FE_PACKETSIZE;
			}
		}
	}
	data[datai] = '\0';

	strcat(data,"\r\n");
	datai = strlen(data);

	if (send(c->s,data,datai,0) != datai) {
		irc_internal_disconnect(c,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

static char **irc_recv_parse(struct s_irc_connection * const c) {
	static char *args[256];
	int curarg;
	char *tempchr;
	char *tempchr2;
	static char data[513];
	char mydata[513];
	short length;

	curarg = strlen(c->buffer);

	args[0] = NULL;

	if (!strstr(c->buffer,"\r\n")) {
		if ((length = recv(c->s,&c->buffer[curarg],512 - curarg,0)) < 1) {
			irc_internal_disconnect(c,FE_DISCONNECT);
			firetalkerror = FE_DISCONNECT;
			return args;
		}
		c->buffer[curarg + length] = '\0';
	}
	if ((tempchr = strstr(c->buffer,"\r\n"))) {
		tempchr[0] = '\0';
		if (c->buffer[0] == ':')
			strcpy(data,&c->buffer[1]);
		else
			strcpy(data,c->buffer);
		strcpy(mydata,&tempchr[2]);
		strcpy(c->buffer,mydata);
	} else {
		if (strlen(c->buffer) >= 512) {
			irc_internal_disconnect(c,FE_PACKETSIZE);
			firetalkerror = FE_PACKETSIZE;
		}
		firetalkerror = FE_SUCCESS;
		return args;
	}

	curarg = 0;
	tempchr = data;

	while (curarg < 256 && (tempchr2 = strchr(tempchr,' '))) {
		args[curarg++] = tempchr;
		tempchr2[0] = '\0';
		tempchr = tempchr2 + 1;
		if (tempchr[0] == ':') {
			tempchr++;
			break;
		}
	}
	args[curarg++] = tempchr;
	args[curarg] = NULL;
	return args;
}

static char *irc_get_nickname(const char * const hostmask) {
	static char data[512];
	char *tempchr;
	strcpy(data,hostmask);
	tempchr = strchr(data,'!');
	if (tempchr)
		tempchr[0] = '\0';
	return data;
}

int irc_disconnect(void * const c) {
	irc_send_printf(c,"QUIT :User disconnected");
	return irc_internal_disconnect(c,FE_USERDISCONNECT);
}

void *irc_connect_server(const char * server, short port, const char * const nickname, const char * const password) {
	struct s_irc_connection *c;
	struct hostent *he;
	struct in_addr addr;
	struct sockaddr_in fulladdr;
	char ourserver[sizeof(IRC_SERVER)] = IRC_SERVER;
	int flag;
	char **args;

	if (!server) {
		server = ourserver;
		port = IRC_PORT;
	}
	
	c = malloc(sizeof(struct s_irc_connection));
	if (!c) {
		perror("malloc");
		exit(1);
	}
	c->nickname = NULL;
	c->lasttime = 0;
	c->isons = 0;
	c->whois_head = NULL;
	c->password = malloc(strlen(password) + 1);
	strcpy(c->password,password);

	if ( (c->s = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
		irc_internal_disconnect(c,FE_SOCKET);
		firetalkerror = FE_SOCKET;
		return NULL;
	}

	if (! inet_aton(server,&addr)) {
		if (! (he = gethostbyname(server))) {
			irc_internal_disconnect(c,FE_RESOLV);
			firetalkerror = FE_RESOLV;
			return NULL;
		}
		memcpy(&addr,he->h_addr_list[0],sizeof(struct in_addr));
	}

	fulladdr.sin_family = AF_INET;
	fulladdr.sin_port = htons(port);
	memcpy(&fulladdr.sin_addr,&addr,sizeof(struct in_addr));

	if (connect(c->s,(const struct sockaddr *)&fulladdr,sizeof(struct sockaddr_in))) {
		irc_internal_disconnect(c,FE_CONNECT);
		firetalkerror = FE_CONNECT;
		return NULL;
	}

	irc_send_printf(c,"USER %s %s %s :%s",nickname,server,server,nickname);
	irc_send_printf(c,"NICK %s",nickname);

	flag = 1;
	while (flag) {
		args = irc_recv_parse(c);
		if (!args[0])
			continue;
		/* zero argument items */
		if (!strcmp(args[0],"ERROR")) {
			irc_send_printf(c,"QUIT :error");
			irc_internal_disconnect(c,FE_CONNECT);
			firetalkerror = FE_CONNECT;
			return NULL;
		}

		if (!args[1])
			continue;
		/* one argument items */
		if (!strcmp(args[1],"ERROR")) {
			irc_send_printf(c,"QUIT :error");
			irc_internal_disconnect(c,FE_CONNECT);
			firetalkerror = FE_CONNECT;
			return NULL;
		}
		if (!strcmp(args[0],"PING"))
			irc_send_printf(c,"PONG %s",args[1]);
		else {
			switch (atoi(args[1])) {
				case 376:
					flag = 0;
					break;
				case 431:
				case 432:
				case 433:
				case 436:
				case 461:
					irc_send_printf(c,"QUIT :Invalid nickname");
					irc_internal_disconnect(c,FE_BADUSER);
					firetalkerror = FE_BADUSER;
					return NULL;
					break;
				case 465:
					irc_send_printf(c,"QUIT :banned");
					irc_internal_disconnect(c,FE_BLOCKED);
					firetalkerror = FE_BLOCKED;
					return NULL;
					break;
			}
		}
		
	}

	c->nickname = malloc(strlen(nickname) + 1);
	if (!c->nickname) {
		perror("malloc");
		exit(1);
	}
	strcpy(c->nickname,nickname);

	return c;
}

int irc_got_data(void * const c) {
	struct s_irc_connection *conn;
	char **args;
	char *tempchr, *tempchr2, *tempchr3;
	struct s_firetalk_buddy *buddyiter;
	struct s_irc_whois *whoisiter, *whoisiter2;
	struct s_firetalk_handle *fchandle;
	int tempint,tempint2,tempint3;
	char tempbuf[512];
	time_t temptime;

	fchandle = firetalk_find_handle(c);

	conn = c;

	args = irc_recv_parse(conn);
	while (args[0]) {
		/* zero argument items */
		if (!strcmp(args[0],"PING")) {
			if (args[1])
				irc_send_printf(c,"PONG %s",args[1]);
			else
				irc_send_printf(c,"PONG");
		}
		if (args[1]) {
			tempint = atoi(args[1]);
			if (!strcmp(args[1],"QUIT")) {
				firetalk_callback_im_buddyonline(conn,irc_get_nickname(args[0]),0);
				if (!irc_compare_nicks(conn->nickname,irc_get_nickname(args[0])))
					irc_internal_disconnect(conn,FE_DISCONNECT);
				else
					firetalk_callback_chat_user_quit(conn,irc_get_nickname(args[0]));
			}
		}
		if (args[1] && args[2]) {
			/* two-arg commands */
			if (!strcmp(args[1],"JOIN")) {
				firetalk_callback_im_buddyonline(conn,irc_get_nickname(args[0]),1);
				if (!irc_compare_nicks(conn->nickname,irc_get_nickname(args[0])))
					firetalk_callback_chat_joined(conn,&args[2][1]);
				else
					firetalk_callback_chat_user_joined(conn,&args[2][1],irc_get_nickname(args[0]));
			} else if (!strcmp(args[1],"PART")) {
				if (!irc_compare_nicks(conn->nickname,irc_get_nickname(args[0])))
					firetalk_callback_chat_left(conn,&args[2][1]);
				else
					firetalk_callback_chat_user_left(conn,&args[2][1],irc_get_nickname(args[0]));
			}
		}
		if (args[1] && args[2] && args[3]) {
			if (!strcmp(args[1],"PRIVMSG")) {
				/* scan for CTCP's */
				while ((tempchr = strchr(args[3],1)) && (tempchr2 = strchr(&tempchr[1],1))) {
					/* we have a ctcp */
					tempchr2[0] = '\0';
					if (!strcasecmp(&tempchr[1],"VERSION"))
						irc_send_printf(conn,"NOTICE %s :\001VERSION firetalk " LIBFIRETALK_VERSION "\001",irc_get_nickname(args[0]));
					else if (!strcasecmp(&tempchr[1],"SOURCE"))
						irc_send_printf(conn,"NOTICE %s :\001SOURCE " LIBFIRETALK_HOMEPAGE "\001",irc_get_nickname(args[0]));
					else if (!strcasecmp(&tempchr[1],"CLIENTINFO"))
						irc_send_printf(conn,"NOTICE %s :\001CLIENTINFO PING SOURCE TIME VERSION\001",irc_get_nickname(args[0]));
					else if (!strncasecmp(&tempchr[1],"PING ",5))
						irc_send_printf(conn,"NOTICE %s :\001PING %s\001",irc_get_nickname(args[0]),&tempchr[6]);
					else if (!strcasecmp(&tempchr[1],"TIME")) {
						time(&temptime);
						tempchr3 = ctime(&temptime);
						tempchr3[strlen(tempchr3)-1] = '\0';
						irc_send_printf(conn,"NOTICE %s :\001TIME %s\001",irc_get_nickname(args[0]),tempchr3);
					} else if (!strncasecmp(&tempchr[1],"ACTION ",7)) {
						if (args[2][0] == '#')
							firetalk_callback_chat_getaction(conn,&args[2][1],irc_get_nickname(args[0]),0,&tempchr[8]);
						else
							firetalk_callback_im_getaction(conn,irc_get_nickname(args[0]),0,&tempchr[8]);
					}
					strcpy(tempchr,&tempchr2[1]);
				}
				if (args[3][0]) {
					if (args[2][0] == '#')
						firetalk_callback_chat_getmessage(conn,&args[2][1],irc_get_nickname(args[0]),0,args[3]);
					else
						firetalk_callback_im_getmessage(conn,irc_get_nickname(args[0]),0,args[3]);
				}
			} else if (!strcmp(args[1],"NOTICE")) {
				if (conn->password && !strcasecmp(irc_get_nickname(args[0]),"NickServ") && strstr(args[3],"IDENTIFY"))
					/* nickserv seems to be asking us to identify ourselves, and we have a password */
					irc_send_printf(conn,"PRIVMSG NickServ :IDENTIFY %s",conn->password);
				if (args[2][0] == '#')
					firetalk_callback_chat_getmessage(conn,&args[2][1],irc_get_nickname(args[0]),1,args[3]);
				else
					firetalk_callback_im_getmessage(conn,irc_get_nickname(args[0]),1,args[3]);
			} else if (!strcmp(args[1],"TOPIC")) {
				firetalk_callback_chat_gottopic(conn,&args[2][1],args[3],irc_get_nickname(args[0]));
			} else {
				switch (tempint) {
					case 301: /* RPL_AWAY */
						buddyiter = fchandle->buddy_head;
						while (buddyiter) {
							if (!irc_compare_nicks(args[3],buddyiter->nickname)) {
								buddyiter->tempint2 = 1;
								firetalk_callback_im_buddyaway(conn,args[3],1);
							}
							buddyiter = buddyiter->next;
						}
						break;
					case 303: /* RPL_ISON */
						tempchr = args[3];
						while (tempchr && tempchr[0]) {
							tempchr2 = strchr(tempchr,' ');
							if (tempchr2)
								tempchr2[0] = '\0';
							buddyiter = fchandle->buddy_head;
							while (buddyiter) {
								if (!irc_compare_nicks(tempchr,buddyiter->nickname))
									buddyiter->tempint = 1;
								buddyiter = buddyiter->next;
							}
							if (tempchr2)
								tempchr = tempchr2 + 1;
							else
								tempchr = NULL;
						}
						if (--conn->isons <= 0) {
							conn->isons = 0;
							/* done, send the appropriate data */
							buddyiter = fchandle->buddy_head;
							while (buddyiter) {
								firetalk_callback_im_buddyonline(conn,buddyiter->nickname,buddyiter->tempint);
								if (buddyiter->tempint) {
									buddyiter->tempint2 = 0; /* away */
									irc_send_printf(conn,"WHOIS %s",buddyiter->nickname);
								}
								buddyiter = buddyiter->next;
							}
						}
						break;
					case 313: /* RPL_WHOISOPER */
						whoisiter = conn->whois_head;
						while (whoisiter) {
							if (!irc_compare_nicks(args[3],whoisiter->nickname))
								whoisiter->flags |= FF_ADMIN;
							whoisiter = whoisiter->next;
						}
						break;
					case 318: /* RPL_ENDOFWHOIS */
						whoisiter = conn->whois_head;
						whoisiter2 = NULL;
						while (whoisiter) {
							if (!irc_compare_nicks(args[3],whoisiter->nickname)) {
								/* manual whois */
								firetalk_callback_gotinfo(conn,whoisiter->nickname,whoisiter->info,0,whoisiter->idle,whoisiter->flags);
								free(whoisiter->nickname);
								if (whoisiter->info)
									free(whoisiter->info);
								if (whoisiter2)
									whoisiter2->next = whoisiter->next;
								else
									conn->whois_head = whoisiter->next;
								free(whoisiter);
								break;
							}
							whoisiter2 = whoisiter;
							whoisiter = whoisiter->next;
						}
						buddyiter = fchandle->buddy_head;
						while (buddyiter) {
							if (!irc_compare_nicks(args[3],buddyiter->nickname)) {
								if (!buddyiter->tempint2)
									firetalk_callback_im_buddyaway(conn,buddyiter->nickname,0);
							}
							buddyiter = buddyiter->next;
						}
						break;
					case 401: /* ERR_NOSUCHNICK */
					case 441: /* ERR_USERNOTINCHANNEL */
					case 443: /* ERR_USERONCHANNEL */
						firetalk_callback_im_buddyonline(conn,args[3],0);
						firetalk_callback_error(conn,FE_BADUSER,args[3]);
						whoisiter = conn->whois_head;
						whoisiter2 = NULL;
						while (whoisiter) {
							if (!irc_compare_nicks(args[3],whoisiter->nickname)) {
								free(whoisiter->nickname);
								if (whoisiter->info)
									free(whoisiter->info);
								if (whoisiter2)
									whoisiter2->next = whoisiter->next;
								else
									conn->whois_head = whoisiter->next;
								free(whoisiter);
								break;
							}
							whoisiter2 = whoisiter;
							whoisiter = whoisiter->next;
						}
						break;
					case 403: /* ERR_NOSUCHCHANNEL */
					case 442: /* ERR_NOTONCHANNEL */
						firetalk_callback_error(conn,FE_BADROOM,&args[3][1]);
						break;
					case 404: /* ERR_CANNOTSENDTOCHAN */
					case 405: /* ERR_TOOMANYCHANNELS */
					case 471: /* ERR_CHANNELISFULL */
					case 473: /* ERR_INVITEONLYCHAN */
					case 474: /* ERR_BANNEDFROMCHAN */
					case 475: /* ERR_BADCHANNELKEY */
						firetalk_callback_error(conn,FE_ROOMUNAVAILABLE,&args[3][1]);
						break;
					case 412: /* ERR_NOTEXTTOSEND */
						firetalk_callback_error(conn,FE_BADMESSAGE,NULL);
						break;
					case 482: /* ERR_CHANOPRIVSNEEDED */
						firetalk_callback_error(conn,FE_NOPERMS,&args[3][1]);
						break;
				}
			}
		}
		if (args[1] && args[2] && args[3] && args[4]) {
			if (!strcmp(args[1],"MODE")) {
				tempint = 0;
				tempint2 = 4;
				tempint3 = 0;
				while (args[tempint2] && args[3][tempint]) {
					switch (args[3][tempint++]) {
						case '+':
							tempint3 = 1;
							break;
						case '-':
							tempint3 = -1;
							break;
						case 'o':
							if (tempint3 == 1)
								firetalk_callback_chat_user_opped(conn,&args[2][1],args[tempint2++],irc_get_nickname(args[0]));
							else if (tempint3 == -1)
								firetalk_callback_chat_user_deopped(conn,&args[2][1],args[tempint2++],irc_get_nickname(args[0]));
							break;
						default:
							tempint2++;
							break;
					}
				}
			} else {
				switch (tempint) {
					case 317: /* RPL_WHOISIDLE */
						whoisiter = conn->whois_head;
						while (whoisiter) {
							if (!irc_compare_nicks(args[3],whoisiter->nickname))
								whoisiter->idle = atol(args[4])/60;
							whoisiter = whoisiter->next;
						}
						buddyiter = fchandle->buddy_head;
						while (buddyiter) {
							if (!irc_compare_nicks(args[3],buddyiter->nickname))
								firetalk_callback_idleinfo(conn,args[3],atol(args[4]) / 60);
							buddyiter = buddyiter->next;
						}
						break;
					case 332: /* RPL_TOPIC */
						firetalk_callback_chat_gottopic(conn,&args[3][1],args[4],NULL);
						break;
				}
			}
		}
		if (args[1] && args[2] && args[3] && args[4] && args[5]) {
			if (tempint == 353) {
				tempchr2 = args[5];
				while ((tempchr = strchr(tempchr2,' '))) {
					tempchr[0] = '\0';
					if (tempchr2[0] == '@' || tempchr2[0] == '+') {
						firetalk_callback_chat_user_joined(conn,&args[4][1],&tempchr2[1]);
						firetalk_callback_im_buddyonline(conn,&tempchr2[1],1);
						if (tempchr2[0] == '@')
							firetalk_callback_chat_user_opped(conn,&args[4][1],&tempchr2[1],NULL);
					} else {
						firetalk_callback_chat_user_joined(conn,&args[4][1],tempchr2);
						firetalk_callback_im_buddyonline(conn,tempchr2,1);
					}
					tempchr2 = tempchr + 1;
				}
			}
		}
		if (args[1] && args[2] && args[3] && args[4] && args[5] && args[6] && args[7]) {
			switch (tempint) {
				case 311: /* RPL_WHOISUSER */
					whoisiter = conn->whois_head;
					while (whoisiter) {
						if (!irc_compare_nicks(args[3],whoisiter->nickname)) {
							if (whoisiter->info)
								free(whoisiter->info);
							snprintf(tempbuf,512,"%s@%s: %s",args[4],args[5],args[7]);
							whoisiter->info = malloc(strlen(tempbuf)+1);
							strcpy(whoisiter->info,tempbuf);
						}
						whoisiter = whoisiter->next;
					}
				break;
			}
		}
		
		if (strstr(conn->buffer,"\r\n"))
			args = irc_recv_parse(conn);
		else
			args[0] = NULL;
	}
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int irc_compare_nicks(const char * const nick1, const char * const nick2) {
	return strcasecmp(nick1,nick2);
}

int irc_get_fd(void * const c) {
	return ((struct s_irc_connection *)c)->s;
}

int irc_chat_join(void * const c, const char * const room) {
	return irc_send_printf(c,"JOIN #%s",room);
}

int irc_chat_part(void * const c, const char * const room) {
	return irc_send_printf(c,"PART #%s",room);
}

int irc_chat_send_message(void * const c, const char * const room, const char * const message, const int auto_flag) {
	if (auto_flag)
		return irc_send_printf(c,"NOTICE #%s :%s",room,message);
	else
		return irc_send_printf(c,"PRIVMSG #%s :%s",room,message);
}

int irc_chat_send_action(void * const c, const char * const room, const char * const message, const int auto_flag) {
	if (auto_flag)
		return irc_send_printf(c,"NOTICE #%s :\001ACTION %s\001",room,message);
	else
		return irc_send_printf(c,"PRIVMSG #%s :\001ACTION %s\001",room,message);
}

int irc_chat_invite(void * const c, const char * const room, const char * const who) {
	return irc_send_printf(c,"INVITE %s #%s",who,room);
}

int irc_im_send_message(void * const c, const char * const dest, const char * const message, const int auto_flag) {
	if (auto_flag)
		return irc_send_printf(c,"NOTICE %s :%s",dest,message);
	else
		return irc_send_printf(c,"PRIVMSG %s :%s",dest,message);
}

int irc_im_send_action(void * const c, const char * const dest, const char * const message, const int auto_flag) {
	if (auto_flag)
		return irc_send_printf(c,"NOTICE %s :\001ACTION %s\001",dest,message);
	else
		return irc_send_printf(c,"PRIVMSG %s :\001ACTION %s\001",dest,message);
}

int irc_chat_set_topic(void * const c, const char * const room, const char * const topic) {
	return irc_send_printf(c,"TOPIC #%s :%s",room,topic);
}

int irc_chat_op(void * const c, const char * const room, const char * const who) {
	return irc_send_printf(c,"MODE #%s +o %s",room,who);
}

int irc_chat_deop(void * const c, const char * const room, const char * const who) {
	return irc_send_printf(c,"MODE #%s -o %s",room,who);
}

int irc_im_add_buddy(void * const c, const char * const nickname) {
	struct s_irc_connection *conn;
	conn = c;
	conn->isons++;
	return irc_send_printf(c,"ISON %s",nickname);
}

int irc_get_info(void * const c, const char * const nickname) {
	struct s_irc_connection *conn;
	struct s_irc_whois *whoistemp;
	conn = c;
	whoistemp = conn->whois_head;
	conn->whois_head = malloc(sizeof(struct s_irc_whois));
	conn->whois_head->nickname = malloc(strlen(nickname)+1);
	strcpy(conn->whois_head->nickname,nickname);
	conn->whois_head->flags = FF_NORMAL;
	conn->whois_head->idle = 0;
	conn->whois_head->info = NULL;
	conn->whois_head->next = whoistemp;
	return irc_send_printf(conn,"WHOIS %s",nickname);
}

int irc_check_buddies(struct s_firetalk_handle * const c) {
	struct s_irc_connection *conn;
	struct s_firetalk_buddy *buddyiter;
	char obuf[505];

	obuf[0] = '\0';

	conn = c->handle;
	if (conn->lasttime > (time(NULL) - 40))
		return FE_IDLEFAST;

	if (conn->isons > 0)
		return FE_IDLEFAST;

	buddyiter = c->buddy_head;
	while (buddyiter) {
		if (strlen(obuf) + strlen(buddyiter->nickname) > 502) {
			obuf[strlen(obuf)-1] = '\0';
			irc_send_printf(conn,"ISON %s",obuf);
			conn->isons++;
			obuf[0] = '\0';
		}
		strcat(obuf,buddyiter->nickname);
		strcat(obuf," ");
		buddyiter->tempint = 0;
		buddyiter = buddyiter->next;
	}

	obuf[strlen(obuf)-1] = '\0';
	if (strlen(obuf)) {
		irc_send_printf(conn,"ISON %s",obuf);
		conn->isons++;
	}

	time(&conn->lasttime);
	return FE_SUCCESS;
}
