/*
toc.c - FireTalk TOC protocol definitions
Copyright (C) 2000 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/poll.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include "toc.h"
#include "firetalk.h"
#include "firetalk-int.h"

#define SFLAP_FRAME_SIGNON 1
#define SFLAP_FRAME_DATA 2
#define SFLAP_FRAME_ERROR 3
#define SFLAP_FRAME_SIGNOFF 4
#define SFLAP_FRAME_KEEPALIVE 5

#define SIGNON_STRING "FLAPON\r\n\r\n"

#define TOC_SERVER "toc.oscar.aol.com"
#define TOC_PORT 9898

#define TOC_HEADER_LENGTH 6
#define TOC_SIGNON_LENGTH 24
#define TOC_HOST_SIGNON_LENGTH 4
#define TOC_USERNAME_MAXLEN 16

char lastinfo[TOC_USERNAME_MAXLEN + 1] = "";

/* Structures */

typedef struct s_toc_room {
	struct s_toc_room *next;
	char *name;
	char *id;
	int invited;
} s_toc_room;

typedef struct s_toc_connection {
	int s;                                               /* socket */
	unsigned short local_sequence;                       /* our sequence number */
	unsigned short remote_sequence;                      /* toc's sequence number */
	char *nickname;                                      /* our nickname (correctly spaced) */
	struct s_toc_room *room_head;
	time_t lasttalk;                                     /* last time we talked */
	time_t lasttime;                                     /* last time we set idles */
	int setnoidle;                                       /* whether we're set unidle */
} s_toc_connection;


/* Internal Function Declarations */

static unsigned short toc_fill_header(unsigned char * const header, unsigned char const frame_type, unsigned short const sequence, unsigned short const length);
static unsigned short toc_fill_signon(unsigned char * const signon, const char * const username);
static unsigned char toc_get_frame_type_from_header(const unsigned char * const header);
static unsigned short toc_get_sequence_from_header(const unsigned char * const header);
static unsigned short toc_get_length_from_header(const unsigned char * const header);
static char *toc_quote(const char * const string, const char outside_flag);
static char *toc_hash_password(const char * const password);
static char *toc_get_html(const char * const url);
static int toc_internal_disconnect(struct s_toc_connection * const c, const int error);
static int toc_internal_add_room(void * const c, const char * const name, const char * const id);
static char *toc_internal_find_room_id(void * const c, const char * const name);
static char *toc_internal_find_room_name(void * const c, const char * const id);
static int toc_send_printf(struct s_toc_connection * const c, const char * const format, ...);
static char **toc_parse_args(char * const instring, int maxargs);


/* Internal Function Definitions */

static unsigned short toc_fill_header(unsigned char * const header, unsigned char const frame_type, unsigned short const sequence, unsigned short const length) {
	header[0] = '*';            /* byte 0, length 1, magic 42 */
	header[1] = frame_type;     /* byte 1, length 1, frame type (defined above SFLAP_FRAME_* */
	header[2] = sequence/256;   /* byte 2, length 2, sequence number, network byte order */
	header[3] = sequence%256;   
	header[4] = length/256;     /* byte 4, length 2, length, network byte order */
	header[5] = length%256;
	return 6 + length;
}

static unsigned short toc_fill_signon(unsigned char * const signon, const char * const username) {
	unsigned short length;
	length = strlen(username);
	signon[0] = 0;              /* byte 0, length 4, flap version (1) */
	signon[1] = 0;
	signon[2] = 0;
	signon[3] = 1;
	signon[4] = 0;              /* byte 4, length 2, tlv tag (1) */
	signon[5]	= 1;
	signon[6] = length/256;     /* byte 6, length 2, username length, network byte order */
	signon[7] = length%256;
	memcpy(&signon[8],username,length);
	return(length + 8);
}

static unsigned char toc_get_frame_type_from_header(const unsigned char * const header) {
	return header[1];
}

static unsigned short toc_get_sequence_from_header(const unsigned char * const header) {
	unsigned short sequence;
	sequence = ntohs(* ((unsigned short *)(&header[2])));
	return sequence;
}

static unsigned short toc_get_length_from_header(const unsigned char * const header) {
	unsigned short length;
	length = ntohs(* ((unsigned short *)(&header[4])));
	return length;
}

static char *toc_quote(const char * const string, const char outside_flag) {
	static char output[2048];
	int length;
	int counter;
	int newcounter;

	length = strlen(string);
	if (outside_flag) {
 		newcounter = 1;
		output[0] = '"';
	} else
		newcounter = 0;

	for (counter = 0; counter < length; counter++) {
		if (string[counter] == '$' || string[counter] == '{' || string[counter] == '}' || string[counter] == '[' || string[counter] == ']' || string[counter] == '(' || string[counter] == ')' || string[counter] == '\'' || string[counter] == '`' || string[counter] == '"' || string[counter] == '\\') {
			if (newcounter > 2044)
				return NULL;
			output[newcounter++] = '\\';
			output[newcounter++] = string[counter];
		} else {
			if (newcounter > 2045)
				return NULL;
			output[newcounter++] = string[counter];
		}
	}

	if (outside_flag)
		output[newcounter++] = '"';
	output[newcounter] = '\0';

	return output;
}

static char *toc_hash_password(const char * const password) {
#define HASH "Tic/Toc"
	const char hash[sizeof(HASH)] = HASH;
	static char output[2048];
	int counter;
	int newcounter;
	int length;

	length = strlen(password);

	output[0] = '0';
	output[1] = 'x';

	newcounter = 2;

	for (counter = 0; counter < length; counter++) {
		if (newcounter > 2044)
			return NULL;
		sprintf(&output[newcounter],"%02x",password[counter] ^ hash[((counter) % (sizeof(HASH)-1))]);
		newcounter += 2;
	}

	output[newcounter] = '\0';

	return output;
}

static char *toc_get_html(const char * const url) {
#define TOC_HTML_MAXLEN 65536
	char data[TOC_HTML_MAXLEN];
	int length;
	int s;
  struct hostent *he;
  struct in_addr addr;
  struct sockaddr_in fulladdr;
	char *tempchr;

	if ( (s = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
		firetalkerror = FE_SOCKET;
		return NULL;
	}

	if (! (he = gethostbyname("toc-d01.blue.aol.com"))) {
		firetalkerror = FE_RESOLV;
		return NULL;
	}
	memcpy(&addr,he->h_addr_list[0],sizeof(struct in_addr));

	fulladdr.sin_family = AF_INET;
	fulladdr.sin_port = htons(80);
	memcpy(&fulladdr.sin_addr,&addr,sizeof(struct in_addr));

	if (connect(s,(const struct sockaddr *)&fulladdr,sizeof(struct sockaddr_in))) {
		firetalkerror = FE_CONNECT;
		return NULL;
	}

#define TOC_HTTP_REQUEST "GET /%s HTTP/1.0\r\n\r\n"

	sprintf(data,TOC_HTTP_REQUEST,url);
	length = strlen(data);
	if (send(s,data,length,0) != length) {
		firetalkerror = FE_PACKET;
		return NULL;
	}

	if ( (length = recv(s,data,TOC_HTML_MAXLEN,MSG_WAITALL)) < 1) {
		firetalkerror = FE_PACKET;
		return NULL;
	}

	data[length] = '\0';

	if (! (tempchr = strstr(data,"\r\n\r\n") )) {
		firetalkerror = FE_INVALIDFORMAT;
		return NULL;
	}

	return tempchr + 4;
}

static int toc_internal_disconnect(struct s_toc_connection * const c, const int error) {
	shutdown(c->s, SHUT_RDWR);
	if (c->nickname)
		free(c->nickname);
	free(c);
	firetalkerror = error;
	firetalk_callback_disconnect(c,error);
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

static int toc_internal_add_room(void * const c, const char * const name, const char * const id) {
	struct s_toc_room *iter;
	struct s_toc_connection *conn;
	conn = c;

	iter = conn->room_head;

	while (iter) {
		if (!toc_compare_nicks(iter->name,name)) {
			free(iter->id);
			iter->id = malloc(strlen(id) + 1);
			strcpy(iter->id,id);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		iter = iter->next;
	}

	iter = conn->room_head;
	conn->room_head = malloc(sizeof(struct s_toc_room));
	if (!conn->room_head) {
		perror("malloc");
		exit(1);
	}
	conn->room_head->next = iter;
	conn->room_head->name = malloc(strlen(name)+1);
	if (!conn->room_head->name) {
		perror("malloc");
		exit(1);
	}
	strcpy(conn->room_head->name,name);
	conn->room_head->id = malloc(strlen(id)+1);
	if (!conn->room_head->id) {
		perror("malloc");
		exit(1);
	}
	strcpy(conn->room_head->id,id);
	conn->room_head->invited = 0;
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

static char * toc_internal_find_room_id(void * const c, const char * const name) {
	struct s_toc_room *iter;
	struct s_toc_connection *conn;
	conn = c;

	iter = conn->room_head;

	while (iter) {
		if (!toc_compare_nicks(iter->name,name)) {
			firetalkerror = FE_SUCCESS;
			return iter->id;
		}
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return NULL;
}

static char * toc_internal_find_room_name(void * const c, const char * const id) {
	struct s_toc_room *iter;
	struct s_toc_connection *conn;
	conn = c;

	iter = conn->room_head;

	while (iter) {
		if (!toc_compare_nicks(iter->id,id)) {
			firetalkerror = FE_SUCCESS;
			return iter->name;
		}
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return NULL;
}

static int toc_internal_set_room_invited(void * const c, const char * const name, const int invited) {
	struct s_toc_room *iter;
	struct s_toc_connection *conn;
	conn = c;

	iter = conn->room_head;

	while (iter) {
		if (!toc_compare_nicks(iter->name,name)) {
			iter->invited = invited;
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return FE_NOTFOUND;
}

static int toc_internal_get_room_invited(void * const c, const char * const name) {
	struct s_toc_room *iter;
	struct s_toc_connection *conn;
	conn = c;

	iter = conn->room_head;

	while (iter) {
		if (!toc_compare_nicks(iter->name,name)) {
			firetalkerror = FE_SUCCESS;
			return iter->invited;
		}
		iter = iter->next;
	}

	firetalkerror = FE_NOTFOUND;
	return FE_NOTFOUND;
}

static int toc_send_printf(struct s_toc_connection * const c, const char * const format, ...) {
	va_list ap;
	int len;
	int i;
	char data[2048];
	short datai;
	char *tempchr;
	short length;

	va_start(ap,format);
	datai = TOC_HEADER_LENGTH;

	len = strlen(format);
	for (i = 0; i < len; i++) {
		if (format[i] == '%') {
			switch (format[++i]) {
				case 's':
					tempchr = va_arg(ap, char *);
					if (strchr(tempchr,' ') || strchr(tempchr,'\n') || strchr(tempchr,'\t'))
						tempchr = toc_quote(tempchr,1);
					if (!tempchr || (datai + strlen(tempchr) > 2046)) {
						firetalkerror = FE_PACKETSIZE;
						return FE_PACKETSIZE;
					}
					strcpy(&data[datai],tempchr);
					datai += strlen(tempchr);
					break;
				case '%':
					data[datai++] = '%';
					break;
			}
		} else {
			data[datai++] = format[i];
			if (datai > 2046) {
				firetalkerror = FE_PACKETSIZE;
				return FE_PACKETSIZE;
			}
		}
	}
	data[datai++] = '\0';

	length = toc_fill_header((unsigned char *)data,SFLAP_FRAME_DATA,++c->local_sequence,datai - TOC_HEADER_LENGTH);

	if (send(c->s,data,length,0) != length) {
		toc_internal_disconnect(c,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

static char *toc_get_arg0(char * const instring) {
	static char data[2048];
	char *tempchr;

	if (strlen(instring) > 2048) {
		firetalkerror = FE_PACKETSIZE;
		return NULL;
	}

	strcpy(data,instring);
	tempchr = strchr(data,':');
	if (tempchr)
		tempchr[0] = '\0';
	return data;
}

static char **toc_parse_args(char * const instring, const int maxargs) {
	static char *args[256];
	int curarg;
	char *tempchr;
	char *tempchr2;

	curarg = 0;
	tempchr = instring;

	while (curarg < (maxargs - 1) && curarg < 256 && (tempchr2 = strchr(tempchr,':'))) {
		args[curarg++] = tempchr;
		tempchr2[0] = '\0';
		tempchr = tempchr2 + 1;
	}
	args[curarg++] = tempchr;
	args[curarg] = NULL;
	return args;
}

/* External Function Definitions */

int toc_compare_nicks (const char * const nick1, const char * const nick2) {
	const char * tempchr1;
	const char * tempchr2;

	tempchr1 = nick1;
	tempchr2 = nick2;

	if (!nick1 || !nick2) {
		firetalkerror = FE_NOMATCH;
		return FE_NOMATCH;
	}

	while (tempchr1[0]) {
		while (tempchr1[0] == ' ')
			tempchr1++;
		while (tempchr2[0] == ' ')
			tempchr2++;
		if (tolower(tempchr1[0]) != tolower(tempchr2[0])) {
			firetalkerror = FE_NOMATCH;
			return FE_NOMATCH;
		}
		tempchr1++;
		tempchr2++;
	}
	if (tempchr2[0]) {
		firetalkerror = FE_NOMATCH;
		return FE_NOMATCH;
	}

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int toc_disconnect(void * const c) {
	return toc_internal_disconnect(c,FE_USERDISCONNECT);
}

void *toc_connect_server(const char * server, short port) {
	struct s_toc_connection *c;
	struct hostent *he;
	struct in_addr addr;
	struct sockaddr_in fulladdr;
	unsigned char header[TOC_HEADER_LENGTH];
	char host_signon[TOC_HOST_SIGNON_LENGTH];
	short length;
	char ourserver[sizeof(TOC_SERVER)] = TOC_SERVER;

	if (!server) {
		server = ourserver;
		port = TOC_PORT;
	}
	
	c = malloc(sizeof(struct s_toc_connection));
	if (!c) {
		perror("malloc");
		exit(1);
	}
	c->nickname = NULL;
	c->room_head = NULL;
	c->lasttalk = time(NULL);
	c->lasttime = 0;
	c->setnoidle = 0;

	if ( (c->s = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
		toc_internal_disconnect(c,FE_SOCKET);
		firetalkerror = FE_SOCKET;
		return NULL;
	}

	if (! inet_aton(server,&addr)) {
		if (! (he = gethostbyname(server))) {
			toc_internal_disconnect(c,FE_RESOLV);
			firetalkerror = FE_RESOLV;
			return NULL;
		}
		memcpy(&addr,he->h_addr_list[0],sizeof(struct in_addr));
	}

	fulladdr.sin_family = AF_INET;
	fulladdr.sin_port = htons(port);
	memcpy(&fulladdr.sin_addr,&addr,sizeof(struct in_addr));

	if (connect(c->s,(const struct sockaddr *)&fulladdr,sizeof(struct sockaddr_in))) {
		toc_internal_disconnect(c,FE_CONNECT);
		firetalkerror = FE_CONNECT;
		return NULL;
	}

	/* send the signon string to indicate that we're speaking FLAP here */

	if (send(c->s,SIGNON_STRING,strlen(SIGNON_STRING),0) != strlen(SIGNON_STRING)) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(c,FE_PACKET);
		firetalkerror = FE_PACKET;
		return NULL;
	}

	/* get the flap version packet from the server */

	if (recv(c->s,&header,TOC_HEADER_LENGTH,MSG_WAITALL) != TOC_HEADER_LENGTH) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(c,FE_PACKET);
		firetalkerror = FE_PACKET;
		return NULL;
	}

	if (toc_get_frame_type_from_header(header) != SFLAP_FRAME_SIGNON) {
		firetalkerror = FE_FRAMETYPE;
		toc_internal_disconnect(c,FE_FRAMETYPE);
		firetalkerror = FE_FRAMETYPE;
		return NULL;
	}

	length = toc_get_length_from_header(header);

	if (length != TOC_HOST_SIGNON_LENGTH) {
		firetalkerror = FE_PACKETSIZE;
		toc_internal_disconnect(c,FE_PACKETSIZE);
		firetalkerror = FE_PACKETSIZE;
		return NULL;
	}

	if (recv(c->s,&host_signon,TOC_HOST_SIGNON_LENGTH,MSG_WAITALL) != TOC_HOST_SIGNON_LENGTH) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(c,FE_PACKET);
		firetalkerror = FE_PACKET;
		return NULL;
	}

	/* if this isn't our flap version, we're leaving */

	if (host_signon[0] != 0 || host_signon[1] != 0 || host_signon[2] != 0 || host_signon[3] != 1) {
		firetalkerror = FE_VERSION;
		toc_internal_disconnect(c,FE_VERSION);
		firetalkerror = FE_VERSION;
		return NULL;
	}

	/* ok, its ours, grab their initial sequence number so we can start checking */

	c->remote_sequence = toc_get_sequence_from_header(header);
	
	/* and seed our own sequence generation */

	srand(time(NULL));
	c->local_sequence = 1+(short) (65536.0*rand()/(RAND_MAX+1.0));

	return c;
}

int toc_signon(void * const c, const char * const username, const char * const password) {
	char data[8192];
	int flag;
	char *tempchr1;
	char *tempchr2;
	unsigned short length;
	struct pollfd pl;
	struct s_toc_connection *conn;
	int gotconfig;
	unsigned char permit_mode;
	int ret;
  char **args;
	char *arg0;
	struct s_firetalk_handle *fchandle;

	conn = c;

	fchandle = firetalk_find_handle(c);

	/* fill & send the flap signon packet */

	conn->lasttalk = time(NULL);

	length = toc_fill_header((unsigned char *) data,SFLAP_FRAME_SIGNON,++conn->local_sequence,toc_fill_signon((unsigned char *)&data[TOC_HEADER_LENGTH],username));

	if (send(conn->s,data,length,0) != length) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(conn,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}

	ret = toc_send_printf(conn,"toc_signon login.oscar.aol.com 5190 %s %s english \"libfiretalk v" LIBFIRETALK_VERSION "\"",username,toc_hash_password(password));
	if (ret != FE_SUCCESS)
		return ret;

	/* get the SIGN_ON packet back from the server */

	if (recv(conn->s,data,TOC_HEADER_LENGTH,MSG_WAITALL) != TOC_HEADER_LENGTH) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(conn,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}

	if (toc_get_sequence_from_header((unsigned char *)data) != ++conn->remote_sequence) {
		firetalkerror = FE_SEQUENCE;
		toc_internal_disconnect(conn,FE_SEQUENCE);
		firetalkerror = FE_SEQUENCE;
		return FE_SEQUENCE;
	}

	if (toc_get_frame_type_from_header((unsigned char *)data) != SFLAP_FRAME_DATA) {
		firetalkerror = FE_FRAMETYPE;
		toc_internal_disconnect(conn,FE_FRAMETYPE);
		firetalkerror = FE_FRAMETYPE;
		return FE_FRAMETYPE;
	}

	length = toc_get_length_from_header((unsigned char *)data);

	if (length > (8192 - TOC_HEADER_LENGTH)) {
		firetalkerror = FE_PACKETSIZE;
		toc_internal_disconnect(conn,FE_PACKETSIZE);
		firetalkerror = FE_PACKETSIZE;
		return FE_PACKETSIZE;
	}

	if (recv(conn->s,&data[TOC_HEADER_LENGTH],length,MSG_WAITALL) != length) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(conn,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}

	data[TOC_HEADER_LENGTH + length] = '\0';
	arg0 = toc_get_arg0(&data[TOC_HEADER_LENGTH]);

	if (strcmp(arg0,"SIGN_ON")) {
		if (!strcmp(arg0,"ERROR")) {
			args = toc_parse_args(&data[TOC_HEADER_LENGTH],3);
			if (args[1]) {
				switch (atoi(args[1])) {
					case 980:
						toc_internal_disconnect(conn,FE_BADUSERPASS);
						firetalkerror = FE_BADUSERPASS;
						return FE_BADUSERPASS;
						break;
					case 981:
						toc_internal_disconnect(conn,FE_SERVER);
						firetalkerror = FE_SERVER;
						return FE_SERVER;
						break;
					case 982:
						toc_internal_disconnect(conn,FE_BLOCKED);
						firetalkerror = FE_BLOCKED;
						return FE_BLOCKED;
						break;
					case 983:
						toc_internal_disconnect(conn,FE_BLOCKED);
						firetalkerror = FE_BLOCKED;
						return FE_BLOCKED;
						break;
				}
			}
		}
		toc_internal_disconnect(conn,FE_UNKNOWN);
		firetalkerror = FE_UNKNOWN;
		return FE_UNKNOWN;
	}

	/* poll until we have critical packets */

	flag = 1;
	pl.fd = conn->s;
	pl.events = POLLIN;
	gotconfig = 0;
	while (flag) {
		if (poll(&pl,1,30000) > 0) {
			if (recv(conn->s,data,TOC_HEADER_LENGTH,MSG_WAITALL) != TOC_HEADER_LENGTH) {
				toc_internal_disconnect(conn,FE_PACKET);
				firetalkerror = FE_PACKET;
				return FE_PACKET;
			}
			if (toc_get_sequence_from_header((unsigned char *)data) != ++conn->remote_sequence) {
				toc_internal_disconnect(conn,FE_SEQUENCE);
				firetalkerror = FE_SEQUENCE;
				return FE_SEQUENCE;
			}
			if (toc_get_frame_type_from_header((unsigned char *)data) != SFLAP_FRAME_DATA) {
				toc_internal_disconnect(conn,FE_FRAMETYPE);
				firetalkerror = FE_FRAMETYPE;
				return FE_FRAMETYPE;
			}
			length = toc_get_length_from_header((unsigned char *)data);
			if (length > (8192 - TOC_HEADER_LENGTH)) {
				toc_internal_disconnect(conn,FE_PACKETSIZE);
				firetalkerror = FE_PACKETSIZE;
				return FE_PACKETSIZE;
			}
			if (recv(conn->s,&data[TOC_HEADER_LENGTH],length,MSG_WAITALL) != length) {
				toc_internal_disconnect(conn,FE_PACKET);
				firetalkerror = FE_PACKET;
				return FE_PACKET;
			}
			data[TOC_HEADER_LENGTH + length] = '\0';
			arg0 = toc_get_arg0(&data[TOC_HEADER_LENGTH]);
			if (!strcmp(arg0,"NICK")) {
				args = toc_parse_args(&data[TOC_HEADER_LENGTH],2);
				if (args[1]) {
					conn->nickname = malloc(strlen(args[1]) + 1);
					if (!conn->nickname) {
						perror("malloc");
						exit(1);
					}
					strcpy(conn->nickname,args[1]);
				}
			} else if (!strcmp(arg0,"CONFIG")) {
				args = toc_parse_args(&data[TOC_HEADER_LENGTH],2);
				if (!args[1]) {
					toc_internal_disconnect(conn,FE_INVALIDFORMAT);
					firetalkerror = FE_INVALIDFORMAT;
					return FE_INVALIDFORMAT;
				}
				tempchr1 = args[1];
				gotconfig = 1;
				permit_mode = 0;
				while ((tempchr2 = strchr(tempchr1,'\n'))) {
					/* rather stupidly, we have to tell it everything back that it just told us.  how dumb */
					tempchr2[0] = '\0';
					switch (tempchr1[0]) {
						case 'm':    /* permit mode */
							permit_mode = (tempchr1[2] - '1') + 1;
							break;
						case 'd':    /* deny */
							firetalk_im_internal_add_deny(fchandle,&tempchr1[2]);
							break;
						case 'b':    /* buddy */
							firetalk_im_internal_add_buddy(fchandle,&tempchr1[2]);
							break;
					}
					tempchr1 = &tempchr2[1];
				}
				if (permit_mode != 4)
					ret = toc_send_printf(conn,"toc_add_deny");
					if (ret != FE_SUCCESS)
						return ret;
			} else {
				toc_internal_disconnect(c,FE_WIERDPACKET);
				firetalkerror = FE_WIERDPACKET;
				return FE_WIERDPACKET;
			}
			if (conn->nickname && gotconfig)
				flag = 0;
		} else {
			toc_internal_disconnect(conn,FE_PACKET);
			firetalkerror = FE_PACKET;
			return FE_PACKET;
		}
	}

	/* ask the client to handle its init */
	firetalk_callback_doinit(conn);

	/* upload the lists */
	toc_im_upload_buddies(conn);
	toc_im_upload_denies(conn);

	/* complete the login process with a toc_init_done */

	return toc_send_printf(conn,"toc_init_done");
}

int toc_im_add_buddy(void * const c, const char * const nickname) {
	return toc_send_printf(c,"toc_add_buddy %s",nickname);
}

int toc_im_remove_buddy(void * const c, const char * const nickname) {
	return toc_send_printf(c,"toc_remove_buddy %s",nickname);
}

int toc_im_add_deny(void * const c, const char * const nickname) {
	return toc_send_printf(c,"toc_add_deny %s",nickname);
}

int toc_im_remove_deny(void * const c, const char * const nickname) {
	/* this doesn't work in TOC, pretend that it did */
	return FE_SUCCESS;
}

int toc_save_config(void * const c) {
	char data[2048];
	struct s_firetalk_handle *conn;
	struct s_firetalk_buddy *buddyiter;
	struct s_firetalk_deny *denyiter;

	conn = firetalk_find_handle(c);

	strcpy(data,"m 4\ng Buddies\n");
	buddyiter = conn->buddy_head;
	while (buddyiter) {
		if (strlen(data) > 2000) {
			firetalkerror = FE_PACKETSIZE;
			return FE_PACKETSIZE;
		}
		strcat(data,"b ");
		strcat(data,buddyiter->nickname);
		strcat(data,"\n");
		buddyiter = buddyiter->next;
	}
	denyiter = conn->deny_head;
	while (denyiter) {
		if (strlen(data) > 2000) {
			firetalkerror = FE_PACKETSIZE;
			return FE_PACKETSIZE;
		}
		strcat(data,"d ");
		strcat(data,denyiter->nickname);
		strcat(data,"\n");
		denyiter = denyiter->next;
	}
	return toc_send_printf(c,"toc_set_config %s",data);
}

int toc_im_upload_buddies(void * const c) {
	char data[2048];
	struct s_toc_connection *conn;
	struct s_firetalk_buddy *buddyiter;
	struct s_firetalk_handle *fchandle;
	int length; 

	conn = c;

	fchandle = firetalk_find_handle(c);

	strcpy(&data[TOC_HEADER_LENGTH],"toc_add_buddy");
	buddyiter = fchandle->buddy_head;
	while (buddyiter) {
		strcat(&data[TOC_HEADER_LENGTH]," ");
		strcat(&data[TOC_HEADER_LENGTH],toc_quote(buddyiter->nickname,0));
		if (strlen(&data[TOC_HEADER_LENGTH]) > 2000) {
			length = toc_fill_header((unsigned char *)data,SFLAP_FRAME_DATA,++conn->local_sequence,strlen(&data[TOC_HEADER_LENGTH])+1);

			if (send(conn->s,data,length,0) != length) {
				firetalkerror = FE_PACKET;
				toc_internal_disconnect(conn,FE_PACKET);
				firetalkerror = FE_PACKET;
				return FE_PACKET;
			}
			strcpy(&data[TOC_HEADER_LENGTH],"toc_add_buddy");
		}
		buddyiter = buddyiter->next;
	}
	length = toc_fill_header((unsigned char *)data,SFLAP_FRAME_DATA,++conn->local_sequence,strlen(&data[TOC_HEADER_LENGTH])+1);

	if (send(conn->s,data,length,0) != length) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(conn,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int toc_im_upload_denies(void * const c) {
	char data[2048];
	struct s_firetalk_deny *denyiter;
	int length; 
	struct s_toc_connection *conn;
	struct s_firetalk_handle *fchandle;

	conn = c;

	fchandle = firetalk_find_handle(c);

	strcpy(&data[TOC_HEADER_LENGTH],"toc_add_deny");
	denyiter = fchandle->deny_head;
	while (denyiter) {
		strcat(&data[TOC_HEADER_LENGTH]," ");
		strcat(&data[TOC_HEADER_LENGTH],toc_quote(denyiter->nickname,0));
		if (strlen(&data[TOC_HEADER_LENGTH]) > 2000) {
			length = toc_fill_header((unsigned char *)data,SFLAP_FRAME_DATA,++conn->local_sequence,strlen(&data[TOC_HEADER_LENGTH])+1);

			if (send(conn->s,data,length,0) != length) {
				firetalkerror = FE_PACKET;
				toc_internal_disconnect(conn,FE_PACKET);
				firetalkerror = FE_PACKET;
				return FE_PACKET;
			}
			strcpy(&data[TOC_HEADER_LENGTH],"toc_add_deny");
		}
		denyiter = denyiter->next;
	}
	length = toc_fill_header((unsigned char *)data,SFLAP_FRAME_DATA,++conn->local_sequence,strlen(&data[TOC_HEADER_LENGTH])+1);

	if (send(conn->s,data,length,0) != length) {
		firetalkerror = FE_PACKET;
		toc_internal_disconnect(conn,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_PACKET;
	}
	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int toc_im_send_message(void * const c, const char * const dest, const char * const message, const int auto_flag) {
	struct s_toc_connection *conn;

	conn = c;

	if (!auto_flag) {
		if (time(NULL) - conn->lasttalk >= 600) {
			conn->lasttime = 0;
			conn->lasttalk = time(NULL);
			toc_set_idle(conn);
		} else
			conn->lasttalk = time(NULL);
	}
	
	if (auto_flag)
		return toc_send_printf(conn,"toc_send_im %s %s auto",dest,message);
	else
		return toc_send_printf(conn,"toc_send_im %s %s",dest,message);
}

int toc_im_send_action(void * const c, const char * const dest, const char * const message, const int auto_flag) {
	struct s_toc_connection *conn;
	char tempbuf[2048]; 

	if (strlen(message) > 2042) {
		firetalkerror = FE_PACKETSIZE;
		return FE_PACKETSIZE;
	}

	strcpy(tempbuf,"/me ");
	strcat(tempbuf,message);

	conn = c;

	if (!auto_flag) {
		if (time(NULL) - conn->lasttalk >= 600) {
			conn->lasttime = 0;
			conn->lasttalk = time(NULL);
			toc_set_idle(conn);
		} else
			conn->lasttalk = time(NULL);
	}
	
	if (auto_flag)
		return toc_send_printf(conn,"toc_send_im %s %s auto",dest,tempbuf);
	else
		return toc_send_printf(conn,"toc_send_im %s %s",dest,tempbuf);
}

int toc_get_info(void * const c, const char * const nickname) {
	return toc_send_printf(c,"toc_get_info %s",nickname);
}

int toc_set_info(void * const c, const char * const info) {
	return toc_send_printf(c,"toc_set_info %s",info);
}

int toc_set_away(void * const c, const char * const message) {
	if (message)
		return toc_send_printf(c,"toc_set_away %s",message);
	else
		return toc_send_printf(c,"toc_set_away");
}

int toc_im_evil(void * const c, const char * const who) {
	return toc_send_printf(c,"toc_evil %s norm",who);
}

int toc_got_data(void * const c) {
	struct s_toc_connection *conn;
	char *tempchr1;
	char *tempchr2;
	char *tempchr3;
	char *tempchr4;
	int tempint;
	int tempint2;
	char data[8192];
	unsigned short length;
	char *arg0;
	char **args;

	conn = c;

	if (recv(conn->s,data,TOC_HEADER_LENGTH,MSG_WAITALL) != TOC_HEADER_LENGTH) {
		toc_internal_disconnect(conn,FE_DISCONNECT);
		firetalkerror = FE_DISCONNECT;
		return FE_DISCONNECT;
	}
	if (toc_get_sequence_from_header((unsigned char *)data) != ++conn->remote_sequence) {
		toc_internal_disconnect(conn,FE_SEQUENCE);
		firetalkerror = FE_SEQUENCE;
		return FE_DISCONNECT;
	}
	if (toc_get_frame_type_from_header((unsigned char *)data) != SFLAP_FRAME_DATA) {
		toc_internal_disconnect(conn,FE_FRAMETYPE);
		firetalkerror = FE_FRAMETYPE;
		return FE_DISCONNECT;
	}
	length = toc_get_length_from_header((unsigned char *)data);
	if (length > (8192 - TOC_HEADER_LENGTH)) {
		toc_internal_disconnect(conn,FE_PACKETSIZE);
		firetalkerror = FE_PACKETSIZE;
		return FE_DISCONNECT;
	}
	if (recv(conn->s,&data[TOC_HEADER_LENGTH],length,MSG_WAITALL) != length) {
		toc_internal_disconnect(conn,FE_PACKET);
		firetalkerror = FE_PACKET;
		return FE_DISCONNECT;
	}
	data[TOC_HEADER_LENGTH + length] = '\0';
	arg0 = toc_get_arg0(&data[TOC_HEADER_LENGTH]);
	
	if (!strcmp(arg0,"ERROR")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],3);
		if (!args[1]) {
			toc_internal_disconnect(conn,FE_INVALIDFORMAT);
			firetalkerror = FE_INVALIDFORMAT;
			return FE_INVALIDFORMAT;
		}
		switch (atoi(args[1])) {
			case 901:
				firetalk_callback_error(conn,FE_USERUNAVAILABLE,args[2]);
				break;
			case 902:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,args[2]);
				break;
			case 903:
				firetalk_callback_error(conn,FE_TOOFAST,NULL);
				break;
			case 950:
				firetalk_callback_error(conn,FE_ROOMUNAVAILABLE,args[2]);
				break;
			case 960:
				firetalk_callback_error(conn,FE_TOOFAST,args[2]);
				break;
			case 961:
				firetalk_callback_error(conn,FE_INCOMINGERROR,args[2]);
				break;
			case 962:
				firetalk_callback_error(conn,FE_INCOMINGERROR,args[2]);
				break;
			case 970:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 971:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 972:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 973:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 974:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 975:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 976:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 977:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 978:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 979:
				firetalk_callback_error(conn,FE_USERINFOUNAVAILABLE,NULL);
				break;
			case 980:
				firetalk_callback_error(conn,FE_BADUSERPASS,NULL);
				break;
			case 981:
				firetalk_callback_error(conn,FE_UNKNOWN,NULL);
				break;
			case 982:
				firetalk_callback_error(conn,FE_BLOCKED,NULL);
				break;
			case 983:
				firetalk_callback_error(conn,FE_BLOCKED,NULL);
				break;
			case 989:
				firetalk_callback_error(conn,FE_UNKNOWN,NULL);
				break;
			default:
				firetalk_callback_error(conn,FE_UNKNOWN,NULL);
				break;
		}
	} else if (!strcmp(arg0,"IM_IN")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],4);
		if (!args[1] || !args[2] || !args[3]) {
			toc_internal_disconnect(conn,FE_INVALIDFORMAT);
			firetalkerror = FE_INVALIDFORMAT;
			return FE_INVALIDFORMAT;
		}
		if (strncasecmp(args[3],"/me ",4))
			firetalk_callback_im_getmessage(conn,args[1],args[2][0] == 'T',args[3]);
		else
			firetalk_callback_im_getaction(conn,args[1],args[2][0] == 'T',&args[3][4]);
	} else if (!strcmp(arg0,"UPDATE_BUDDY")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],7);
		if (!args[1] || !args[2] || !args[3] || !args[4] || !args[5] || !args[6]) {
			toc_internal_disconnect(conn,FE_INVALIDFORMAT);
			firetalkerror = FE_INVALIDFORMAT;
			return FE_INVALIDFORMAT;
		}
		firetalk_callback_im_buddyonline(conn,args[1],args[2][0] == 'T');
		firetalk_callback_im_buddyaway(conn,args[1],args[6][2] == 'U');
		firetalk_callback_idleinfo(conn,args[1],atol(args[5]));
	} else if (!strcmp(arg0,"GOTO_URL")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],3);
		if (!args[1] || !args[2]) {
			toc_internal_disconnect(conn,FE_INVALIDFORMAT);
			firetalkerror = FE_INVALIDFORMAT;
			return FE_INVALIDFORMAT;
		}
		tempchr2 = toc_get_html(args[2]);
		if (!tempchr2) {
			firetalk_callback_error(conn,firetalkerror,lastinfo[0] ? lastinfo : NULL);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
#define USER_STRING "Username : <B>"
		tempchr1 = strstr(tempchr2,USER_STRING);
		if (!tempchr1) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,lastinfo[0] ? lastinfo : NULL);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
		tempchr1 += sizeof(USER_STRING) - 1;
		tempchr2 = strchr(tempchr1,'<');
		if (!tempchr2) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,lastinfo[0] ? lastinfo : NULL);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
		tempchr2[0] = '\0';
		tempchr2++;
#define WARNING_STRING "Warning Level : <B>"
		tempchr3 = strstr(tempchr2,WARNING_STRING);
		if (!tempchr3) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,tempchr1);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
		tempchr3[0] = '\0';
		tempint = 0;
		if (strstr(tempchr2,"free") || strstr(tempchr2,"dt"))
			tempint |= FF_SUBSTANDARD;
		if (strstr(tempchr2,"aol"))
			tempint |= FF_NORMAL;
		if (strstr(tempchr2,"admin"))
			tempint |= FF_ADMIN;
		tempchr3 += sizeof(WARNING_STRING) - 1;
		tempchr2 = strchr(tempchr3,'%');
		if (!tempchr2) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,tempchr1);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
		tempchr2[0] = '\0';
		tempchr2++;
		tempint2 = atoi(tempchr3);
#define IDLE_STRING "Idle Minutes : <B>" 
		tempchr3 = strstr(tempchr2,IDLE_STRING);
		if (!tempchr3) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,tempchr1);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
		tempchr3 += sizeof(IDLE_STRING) - 1;
		tempchr2 = strchr(tempchr3,'<');
		if (!tempchr2) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,tempchr1);
			firetalkerror = FE_SUCCESS; /* select call actually succeeded */
			return FE_SUCCESS;
		}
		tempchr2[0] = '\0';
		tempchr2++;
#define INFO_STRING "<hr><br>\n"
#define INFO_END_STRING "<br><hr>"
		tempchr4 = strstr(tempchr2,INFO_STRING);
		if (!tempchr4) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,tempchr1);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		tempchr4 += sizeof(INFO_STRING) - 1;
		tempchr2 = strstr(tempchr4,INFO_END_STRING);
		if (!tempchr2) {
			/* this is ok, just means they have no profile */
			firetalk_callback_gotinfo(conn,tempchr1,NULL,tempint2,atoi(tempchr3),tempint);
		} else {
			tempchr2[0] = '\0';
			firetalk_callback_gotinfo(conn,tempchr1,tempchr4,tempint2,atoi(tempchr3),tempint);
		}
	} else if (!strcmp(arg0,"EVILED")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],3);
		if (!args[1]) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,NULL);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		firetalk_callback_eviled(conn,atoi(args[1]),args[2]);
	} else if (!strcmp(arg0,"CHAT_JOIN")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],3);
		if (!args[1] || !args[2]) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,NULL);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		toc_internal_add_room(conn,args[2],args[1]);
		firetalk_callback_chat_joined(conn,args[2]);
	} else if (!strcmp(arg0,"CHAT_LEFT")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],2);
		if (!args[1]) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,NULL);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		firetalk_callback_chat_left(conn,toc_internal_find_room_name(conn,args[1]));
	} else if (!strcmp(arg0,"CHAT_IN")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],5);
		if (!args[1] || !args[2] || !args[3] || !args[4]) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,NULL);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		if (!strncasecmp(args[4],"<HTML><PRE>",11)) {
			args[4] = &args[4][11];
			if ((tempchr1 = strchr(args[4],'<')))
				tempchr1[0] = '\0';
		}
		if (strncasecmp(args[4],"/me ",4))
			firetalk_callback_chat_getmessage(conn,toc_internal_find_room_name(conn,args[1]),args[2],0,args[4]);
		else
			firetalk_callback_chat_getaction(conn,toc_internal_find_room_name(conn,args[1]),args[2],0,&args[4][4]);
	} else if (!strcmp(arg0,"CHAT_INVITE")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],5);
		if (!args[1] || !args[2] || !args[3] || !args[4]) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,NULL);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		toc_internal_add_room(conn,args[1],args[2]);
		toc_internal_set_room_invited(conn,args[1],1);
		firetalk_callback_chat_invited(conn,args[1],args[3],args[4]);
	} else if (!strcmp(arg0,"CHAT_UPDATE_BUDDY")) {
		args = toc_parse_args(&data[TOC_HEADER_LENGTH],4);
		if (!args[1] || !args[2] || !args[3]) {
			firetalk_callback_error(conn,FE_INVALIDFORMAT,NULL);
			firetalkerror = FE_SUCCESS;
			return FE_SUCCESS;
		}
		tempchr1 = args[3];
		tempchr3 = toc_internal_find_room_name(conn,args[1]);
		while ((tempchr2 = strchr(tempchr1,':'))) {
			/* cycle through list of buddies */
			tempchr2[0] = '\0';
			if (args[2][0] == 'T')
				firetalk_callback_chat_user_joined(conn,tempchr3,tempchr1);
			else
				firetalk_callback_chat_user_left(conn,tempchr3,tempchr1);
			tempchr1 = tempchr2 + 1;
		}
		if (args[2][0] == 'T')
			firetalk_callback_chat_user_joined(conn,tempchr3,tempchr1);
		else
			firetalk_callback_chat_user_left(conn,tempchr3,tempchr1);
	} else
		firetalk_callback_error(conn,FE_WIERDPACKET,NULL);

	firetalkerror = FE_SUCCESS;
	return FE_SUCCESS;
}

int toc_get_fd(void * const c) {
	return (*(struct s_toc_connection *)c).s;
}

int toc_set_idle(void * const c) {
	struct s_toc_connection *conn;
	long idle;
	char data[32];

	conn = c;

	if (conn->lasttime > (time(NULL) - 40))
		return FE_IDLEFAST;

	idle = time(NULL) - conn->lasttalk;

	if (idle < 600) {
		if (conn->setnoidle)
			return FE_SUCCESS;
		else
			conn->setnoidle = 1;
	} else
		conn->setnoidle = 0;

	firetalk_callback_setidle(conn,&idle);

	sprintf(data,"%ld",idle);
	conn->lasttime = time(NULL);
	return toc_send_printf(conn,"toc_set_idle %s",data);
}

int toc_chat_join(void * const c, const char * const room) {
	int i;
	i = toc_internal_get_room_invited(c,room);
	if (firetalkerror == FE_SUCCESS && i) {
		toc_internal_set_room_invited(c,room,0);
		return toc_send_printf(c,"toc_chat_accept %s",toc_internal_find_room_id(c,room));
	} else
		return toc_send_printf(c,"toc_chat_join 4 %s",room);
}

int toc_chat_part(void * const c, const char * const room) {
	return toc_send_printf(c,"toc_chat_leave %s",toc_internal_find_room_id(c,room));
}

int toc_chat_send_message(void * const c, const char * const room, const char * const message) {
	return toc_send_printf(c,"toc_chat_send %s %s",toc_internal_find_room_id(c,room),message);
}

int toc_chat_send_action(void * const c, const char * const room, const char * const message) {
	char tempbuf[2048];

	if (strlen(message) > 2042) {
		firetalkerror = FE_PACKETSIZE;
		return FE_PACKETSIZE;
	}
		
	strcpy(tempbuf,"/me ");
	strcat(tempbuf,message);
	return toc_send_printf(c,"toc_chat_send %s %s",toc_internal_find_room_id(c,room),tempbuf);
}

int toc_chat_invite(void * const c, const char * const room, const char * const who, const char * const message) {
	return toc_send_printf(c,"toc_chat_invite %s %s %s",toc_internal_find_room_id(c,room),message,who);
}
