package Freenet;
import Freenet.support.*;
import java.io.*;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
		
  Requires Classes: java.io.* (JDK 1.1)

 */

/**
 * This class creates a conduit between an InputStream and an 
 * OutputStream and when the feed method of this object is called
 * the InputStream will be fed into the output stream of this
 * object.
 *
 * @version 1.0
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author oskar
 **/

public class Conduit extends Thread {
    private static int num = 0;

    private MessageHandler mh;
    private int cnum;
    private InputStream in;
    private OutputStream out;
    private MessageObject mo;
    private long bytesToMove = -1;
    private long bytesMoved = 0;


    /**
     * @param i The stream to read data from
     * @param o The stream to send data to
     * @param mh  A MessageHandler to feed the completed MessageObjects to.
     **/
    public Conduit(InputStream i, OutputStream o, MessageHandler mh) {
	super("Conduit#" + (++num));
	setDaemon(false);
	this.in = i;
	this.out = o;
	this.mh = mh;
    }

    /**
     * Returns the number of bytes that have been read and written.
     * @return  The number of bytes moved.
     */
    public long bytesMoved() {
	return bytesMoved;
    }

    /**
     * Commences the transfer from Input to Output Streams asynchronously
     * @param mo A MessageObject which will be handled when the stream 
     *           finnishes. If the Conduit fails for some reason, the 
     *           getException() method will be called with a ConduitException
     *           before the MessageObject is handled.
     *
     *           This MessageObject will be handled in the copyign thread,
     *           and before releasing the locks on the stream.
     **/
    public void asyncFeed(MessageObject mo) {
	if (!isAlive()) {
	    this.mo=mo;
	    this.start();
	}
    }

    /**
     * Commences the transfer from Input to Output Stream asynchronously 
     * until a number of bytes have been transfered.
     *
     * @param mo A MessageObject which will be handled when the stream 
     *           finnishes. If the Conduit fails for some reason, the 
     *           getException() method will be called with a ConduitException
     *           before the MessageObject is handled.
     *
     *           This MessageObject will be handled in the copying thread,
     *           and before releasing the locks on the stream.
     * @param n  the number of bytes to read
     **/
    public void asyncFeed(MessageObject mo, long n) {
	if (!isAlive()) {
	    this.mo=mo;
	    this.bytesToMove = n;
	    this.start();
	}
    }
    
    public void run() {
	byte[] buffer = new byte[Core.bufferSize];
	int bytes_read;
	Core.logger.log(this,cnum + "- Conduit starting, reading " + 
			bytesToMove + " bytes.",Logger.DEBUG);
	synchronized(in) {
	    synchronized(out) {
		try {
		    do {
			try {
			    bytes_read = in.read(buffer, 0, bytesToMove < 0 ||
						 (bytesToMove - bytesMoved >= 
						  buffer.length)
						 ? buffer.length
						 : (int) (bytesToMove - 
							  bytesMoved));
			} catch (IOException e) {
			    throw new ConduitException(e,in,out,true,
						       bytesMoved);
			    //System.out.println(new String(buffer));
			}
			try {
			    if (bytes_read > 0)
                            {
				out.write(buffer, 0, bytes_read);
			        bytesMoved = bytesMoved + bytes_read;
                            }
			} catch (IOException e) {
			    throw new ConduitException(e,in,out,false, 
						       bytesMoved);
			}
		    } while (bytes_read >= 0 && bytesMoved != bytesToMove);
		    
		    try {
			out.flush();
		    } catch (IOException e) {
			throw new ConduitException(e,in,out,false, bytesMoved);
		    }
		    if (bytesToMove != 0 && bytesMoved != bytesToMove) {
			throw new ConduitException(new EOFException(),
						   in,out,true, bytesMoved);
		    }
		} catch (ConduitException e) {
		    Core.logger.log(this,cnum + "- " + 
				    e.toString(),Logger.MINOR);
		    if (mo != null)
			mo.setException(e);
		} finally {
		    try {
			out.flush();
		    } catch (IOException ie) {
			ie.printStackTrace();
		    }
		    

		    // Finally handle the MessageObject.
		    if (mo != null)
			mh.handle(mo);

		    Core.logger.log(this,cnum + "- Conduit finished moved " + bytesMoved + " bytes.",Logger.DEBUG);
		}
	    }
	}
    }
}
