package Freenet;
import Freenet.crypt.EntropySource;
import Freenet.support.*;
import Freenet.thread.*;
import java.io.*;
import java.io.EOFException;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * Handles both sending and receiving messages on a connection.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 **/

public class ConnectionHandler implements ERunnable
{
    // Protected/Private Fields

    private static int highid = 0x8001;

    private Connection c;
    private Presentation t;
    private MessageHandler mh;
    private int id;

    // state variables
    private volatile long lastActiveTime;
    private volatile boolean closeOnInactive = false;
    private volatile boolean closed;
    private volatile boolean writing = false;
    private boolean persist = false;

    private Thread exec_instance;
    private static EntropySource sendTimer=new EntropySource(), 
	recvTimer=new EntropySource();
    public Object initLock=new Object();
    public Object sendLock=new Object();
    public boolean ready;

    // Constructors
    public ConnectionHandler(Presentation t, Connection c, MessageHandler mh) {
	closed=false;
	this.id = highid;
	if (++highid > 0x10000)
	    highid = 0x8001;

	this.t = t;
	this.c = c;
	this.mh = mh;
	lastActiveTime = System.currentTimeMillis();

	Core.logger.log(this,"New connectionhandler with " 
			+ peer(),Logger.DEBUGGING);
    }


    public void start() {
	exec_instance=new Thread(this, peer() + " ConnectionHandler");
	exec_instance.start();
    }

    public void setExecutionInstance(EThread e) {
	exec_instance=e.getThread();
    }

    // Public Methods
    public void run() {
	synchronized(sendLock) {
	    try {
		Core.logger.log(this, "Authenticating "+c, Logger.DEBUG);
		t.initConnection(c);
		ready=true;
		Core.logger.log(this, c+" authenticated", Logger.DEBUG);
	    } catch (IOException e) {
		Core.logger.log(this,"Failed to initialize connection: " + e,
				Logger.MINOR);
		closed = true;
		synchronized (c.initLock) {
		    c.initLock.notifyAll();
		}
		forceClose();
		return;
	    } finally {
		sendLock.notifyAll();
		if (c != null) {
		    synchronized (c.initLock) {
			c.initLock.notifyAll();
		    }
		}
		synchronized (initLock) {
		    initLock.notifyAll();
		}
	    }
	}	

	c.setReady();
	
	// Get the inputstream for messages (decrypted);
	PushbackInputStream in;
	try {
	    in = new PushbackInputStream(t.getReceiveStream(c));
	} catch (IOException e) {
	    Core.logger.log(this,"Stream died right away",
			    Logger.DEBUGGING);
	    forceClose();
	    return;
	}


	// loop over each message
	message: do {
	    RawMessage m=null;
	    Message msg = null;
	    try {
		if(c==null) 
		    break message;
		
		lastActiveTime = System.currentTimeMillis();
		c.setSoTimeout(1000);
		waiting: while (!closed) {
		    int i;
		    try {
			i = in.read();
			if (i == -1)
			    closed = true;
			else if (i==0) {
			    //ignore null between messages (for cipher padding)
			} else {
			    in.unread(i);
			    break waiting;
			}
		    } catch (InterruptedIOException e) {
			if (!writing && closeOnInactive && System.currentTimeMillis() >= (lastActiveTime + Core.connectionTimeout)) {
			    close();
			}
		    }
		}

		if (closed) break message;

		c.setSoTimeout(600000);/* 10 minutes */
	    
		// read message from protocol
		m = t.readMessage(in);
		if (m.persist < 0)
		    persist = false;
		else if (m.persist > 0)
		    persist = true;

		closed = !persist && !m.keepAlive;
		Core.randSource.acceptTimerEntropy(recvTimer);

		Core.logger.log(this,"Rawmessage:\n"+m,Logger.DEBUG);
		msg = MessageFactory.toMessage(m);

	    } catch (InvalidMessageException e) {
		Core.logger.log(this,"Invalid message: " + e.toString(),
				Logger.MINOR);
		continue message; 
	    } catch (EOFException e) {
		break message; // this stream is over
	    } catch (IOException e) {
		break message;
	    }

	    String logstr=m.messageType+" <- "+c.getPeerAddress();
	    logstr=Long.toHexString(msg.id)+" - "+logstr;
	    Core.logger.log(this,logstr,Logger.MINOR);
	    Core.logger.log(this,"Message:\n"+msg,Logger.DEBUG);
	    
	    // set the receivedAt, source, and receivedWith fields of message

	    // if there was a source and we weren't told otherwise, it is ok 
	    // to close.
	    closeOnInactive = m.source != null && !persist;

	    
	    Core.logger.log(this,"Got on " + c.getMyAddress() + " from " + 
		       (m.source != null ? 
			c.getPeerAddress(m.source.listenPart()) : 
			c.getPeerAddress()), Logger.DEBUGGING);
	    
	    msg.initSources(c.getMyAddress(),m.source == null ? 
			    null : 
			    c.getPeerAddress(m.source.listenPart()), 
			    m.keepAlive ? this : null);
	    
	    // Handle

	    if (m.trailingFieldLength == 0) {
		mh.handle(msg);
	    } else {
		synchronized(m.trailingFieldStream) {
		    mh.handle(msg);
		    try {
			m.trailingFieldStream.wait();
		    } catch (InterruptedException e) {
			Core.logger.log(this,"I got interrupted!",
					Logger.DEBUGGING);
		    }
		    Core.logger.log(this,"I am awake!",Logger.DEBUGGING);
		}
	    }
	    Core.logger.log(this,"Finished with message",
			    Logger.DEBUGGING);
	    lastActiveTime = System.currentTimeMillis();
	} while (!closed);

	if (c != null && c.out != null) {
            try {
                synchronized(t.getSendStream(c)) {
                    Core.logger.log(this,"Finished with connection - closing",
                                    Logger.DEBUGGING);
                    forceClose();
                }
            } catch (IOException e) {
                Core.logger.log(this,"IOException caught during close",
                                Logger.DEBUGGING);
            }
	}
    }

     /**
     * Sends a message using this connection. This will return after the 
     * message headers have been sent. This should not be used for messages
     * with trailing fields.
     * @param m            the message to send
     */   
    public void sendMessage(Message m) throws SendFailedException {
	sendMessage(m,null);
    }

    /**
     * Sends a message using this connection. This will return after the 
     * message headers have been sent, but will send any trailing field
     * in another thread.
     * @param m      the Message to send
     * @param mo     A MessageObject to handle when the sending of the
     *               the trailing field (if there is one) finishes. 
     */
    public void sendMessage(Message m, MessageObject mo) 
	throws SendFailedException {
	while (!closed && !c.ready()) {
	    synchronized(initLock) {
		try {
		    initLock.wait();
		} catch (InterruptedException ie) {}
	    }
	}

	if (closed)
	    throw new SendFailedException((c == null) ? null : c.getPeerAddress());

	synchronized(sendLock) {
	    unlockedSend(m, mo);
	    // allow any trailing field to start before returning the 
	    // sendlock
	    Thread.currentThread().yield(); 
	}
    }

    /**
     * This is the same as sendMessage(Message, MessageObject) but will
     * not attempt to get a lock for sending before doing so. This
     * is somewhat dangerous (the actual write to the sending stream is
     * still locked) but is present to allow Threads that already own
     * the monitor for the sending stream (mostly MessageObjects executed
     * by a completed Conduit) to shortcut infront of messages waiting
     * for a lock. In general, this method should not be used.
     * @param m      the Message to send
     * @param mo     A MessageObject to handle when the sending of the
     *               the trailing field (if there is one) finishes. 
     */
    public void unlockedSend(Message m, MessageObject mo) 
	throws SendFailedException {
	if (closed) throw new SendFailedException(c.getPeerAddress());
	Core.logger.log(this,"Sending a message",Logger.DEBUGGING);
	RawMessage raw=m.toRawMessage(t);
	Address peerAddress = c.getPeerAddress();
	String logstr=raw.messageType+" -> " + peerAddress;
	logstr = Long.toHexString(m.id)+" - "+logstr;
	Core.logger.log(this,logstr,Logger.MINOR);
	
	OutputStream mout;
	try {
	    mout = t.getSendStream(c);
	} catch (IOException e) {
	    throw new SendFailedException(peerAddress);
	}
	synchronized(mout) {
	    try {
		raw.writeMessage(mout);
		mout.flush();
	    } catch (IOException e) {
		throw new SendFailedException(peerAddress);
	    }
	    InputStream data = raw.trailingFieldStream;
	    if (data != null) {
		SendObjectWrapper sow = new 
		    SendObjectWrapper(mo,
				      raw.keepAlive || persist);
		// don't close connection, as I'm writing to it
		writing = true;
		Conduit conduit = new Conduit(data, mout, 
					      new SendObjectHandler());
		conduit.asyncFeed(sow , raw.trailingFieldLength);
	    } else {
		lastActiveTime = System.currentTimeMillis();
		if (!raw.keepAlive && !persist)
		    close();
	    }
	    Core.logger.log(this, "Message sent.",Logger.DEBUGGING);
	}
	// the conduit is synchronized on the output stream too, so it should not start until now - there is a small danger here that another sendMessage would win the race instead of the conduit and start instead...
	// this was happening, so I tried adding the sendlock which should allow the trailing to start first (though it still isn't completely safe)
	Thread.currentThread().yield();
	Core.randSource.acceptTimerEntropy(sendTimer);
    }
    
    public void close() {
	closed = true;
    }

    /**
     * forces this connection to close. Bad things will happen
     * if you use this carelessly.
     **/
    public void forceClose() {
	closed = true;
        if(c!=null)
	  c.close();
	c = null;
	t = null;
	mh = null;
	if (exec_instance!=null)
	    exec_instance.interrupt();
    }

    /**
     * Checks whether the connection is alive
     **/

    public boolean isOpen() {
	return exec_instance.isAlive() && !closed;
    }

    public Address peer() {
	return c.getPeerAddress();
    }

    public Address peer(ListeningAddress laddr) {
	return c.getPeerAddress(laddr);
    }

    public Address local() {
	return c.getMyAddress();
    }

    public Address local(ListeningAddress laddr) {
	return c.getMyAddress(laddr);
    }

    /**
     * The ConnectioHandler has it's own MessageHandler for the MessageObjects
     * used when sends are generated. This is a hack, but will have to do
     * for now
     */
    protected class SendObjectHandler implements MessageHandler {
	
	public void handle(MessageObject mo) {
	    if (mo instanceof SendObjectWrapper) {
		((SendObjectWrapper) mo).done();
	    } else {
		Core.logger.log(this, "Got MessageObject I can't handle: " +
				mo, Logger.ERROR);
	    }
	}
    }

    /**
     * This object is wrapped around the MessageObject called after writing so
     * as to make sure that the ConnectionHandler get's notified as well.
     */
    protected class SendObjectWrapper implements MessageObject {
	MessageObject mo;
	boolean keepAlive;

	public SendObjectWrapper(MessageObject mo, boolean keepAlive) {
	    this.mo = mo;
	    this.keepAlive = keepAlive;
	}

	public long id() {
	    return id;
	}

	public void done() {
	    lastActiveTime = System.currentTimeMillis();
	    writing = false;
	    // now send the actual MessageObject to the real MessageHandler
	    if (mo != null)
		mh.handle(mo);
	    if (!keepAlive)
		close();
	}

	public void setException(Exception e) {
	    if (mo != null)
		mo.setException(e);
	}
    }
}







