package Freenet;
import Freenet.support.*;
import Freenet.thread.*;
import Freenet.crypt.*;
import java.util.*;
//import java.net.*;
import java.io.*;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This is a Wrapper object that contains the components of a Node in the
 * Adaptive Network.  It uses a Network object to communicate with other
 * Nodes in the Network.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 * @author oskar
 **/

public abstract class Core {

    static public final long defaultTickerTime = 500;
    static public final int defaultConnectTimeout = 30000;
    static public final int defaultAuthTimeout = 30000; // 30 seconds
    static public final int defaultConnectionTimeout = 180000; // 3 min
    static public final int defaultHandshakeTimeout = 30000;
    static public final int defaultHandshakeLife = 10000000; // about 3 hours
    static public final int defaultHopTimeExpected = 12000;
    static public final int defaultHopTimeDeviation = 12000;
    static public final int defaultMaximumConnectionThreads = 50;
    static public final int defaultBufferSize = 0x1000;
    static public final boolean defaultTransience = false;
    static public final int defaultListenPort = 19114;  //why was that set to 0 ???

    /** The current version tree **/ 
    public static final String freenetVersion = "0.3.9.2";
    /** The protocolversion supported **/
    public static final String protocolVersion = "1.299";
    /** The build number of the current revision **/
    public static final String buildNumber = "392"; 
    /** Revision number as read from CVS **/
    public static String serverRevision;
    /** Debug mode (Not used AFAIK) **/
    public static boolean debug;
    /** The number of milliseconds to wait for connections to open **/
    public static int connectTimeout;
    public static int authTimeout;
    /** The number of milliseconds between each tick of the ticker **/
    public static long tickerTime;
    /** The number of milliseconds to leave a silent connection open **/ 
    public static int connectionTimeout;
    /** The number of milliseconds to wait for a handshake reply **/
    public static int handshakeTimeout;
    /** How long a handshake successful stays in the memory of the node **/
    public static int handshakeLife;
    /** Whether the node is transient (doesn't advertise itself) **/
    public boolean getTransience() {return transience;}
    private static boolean transience;
    /** Maximum threads to assign to incoming Messages **/
    public static int maximumConnectionThreads;
    /** The Expected per Node time of a hop **/
    public static int hopTimeExpected;
    /** The expected standard deviation from the per hop time **/
    public static int hopTimeDeviation;
    /** The size used for memory buffers when copying data **/
    public static int bufferSize;

    /** The central PRNG **/
    public static RandomSource randSource;
    /** The object that logs events **/
    public static Logger logger = new SuppressedLogger();

    static private boolean initialized=false;

    static {
	randSource=
	    new Yarrow( ((new File("/dev/urandom").exists()) ?
			 "/dev/urandom" : "prng.seed"),
			"SHA1", "Rijndael");
    }

    private static class VoidMessageHandler implements MessageHandler {
	public void handle(MessageObject mo) {}
    }

    /**
     * Sets the logging object to be used for logging messages
     * @param log a Logger object that will log messages to
     an output stream.
    **/
    public static void setLogger(Logger log) {
	
	Core.logger = log;
	
    }

    /**
     * Sets the core properties of Freenet.
     * @param params a Params object containing any command line or
     *                 config file settings. 
     **/
    public static void init(Params params) {
        
        if(initialized) return;
        else initialized=true;
	
	tickerTime = params.getlong("tickerTime", defaultTickerTime);

	transience = params.getboolean("transient",defaultTransience);

        connectTimeout = params.getint("connectTimeout", defaultConnectTimeout);
	authTimeout = params.getint("authTimeout", defaultAuthTimeout);

	connectionTimeout = params.getint("connectionTimeout",defaultConnectionTimeout);
        handshakeTimeout = params.getint("handshakeTimeout", defaultHandshakeTimeout);
	handshakeLife = params.getint("handshakeLife", defaultHandshakeLife);
	hopTimeExpected = params.getint("hopTimeExpected",defaultHopTimeExpected);
	hopTimeDeviation = params.getint("hopTimeDeviation",defaultHopTimeDeviation);
	maximumConnectionThreads = 
	    params.getint("maximumConnectionThreads",
			  defaultMaximumConnectionThreads);

	bufferSize = params.getint("bufferSize",defaultBufferSize);

	// set keytypes 

	Key.addKeyType(Freenet.keys.KHK.keyNumber,Freenet.keys.KHK.class);
	Key.addKeyType(Freenet.keys.KSK.keyNumber,Freenet.keys.KSK.class);
	Key.addKeyType(Freenet.keys.SVK.keyNumber,Freenet.keys.SVK.class);
	Key.addKeyType(Freenet.keys.CHK.keyNumber,Freenet.keys.CHK.class);
	String[] keyTypes = params.getlist("keyTypes");
	for (int i = 1; keyTypes != null && i < keyTypes.length ; i += 2) { 
	    try {
		Key.addKeyType(Integer.parseInt(keyTypes[i-1]),
			       Class.forName(keyTypes[i]));
	    } catch (ClassNotFoundException e) {
		Core.logger.log(null,"No such class: " + keyTypes[i] +
				" for Key type " + keyTypes[i-1],Logger.ERROR);
	    }
	}
    }

    /** The Ticker to schedule delayed execution of MessageObjects with **/
    public Ticker timer;
    /** The ThreadManager to assign threads for handeling incoming messages
	This can be null in which case plain java threads should be used **/
    public ThreadManager threadManager;
    /** The ThreadGroup used by this Core **/
    public ThreadGroup threadGroup;
    /** The local ListeningAddress (port) listened to **/
    public ListeningAddress myAddress;
    /** The protocol presentation for used **/
    public Presentation presentation;
    /** The MessageHandler that incoming messages are handled by **/
    public MessageHandler mh;
    /** The HandShakeHandler used to send, receive, and remember handshakes **/
    public HandshakeHandler hh;
    /** The object that listens to connections **/
    public Listener listener;
    /** provides a way to turn off listening **/
    public boolean listen = true;

    /**
     * Create a new Freenet Core.
     * @param myAddress  The ListeningAddress (port) to listen on.
     * @param presentation  The protocol presentation for this core to use
     * @param hh         A unit that handles the receiving and storing of
     *                   node handshakes.
     * @exception ListenException Thrown if the node cannot listen on the given
     *                            address.
     **/
    public Core(ListeningAddress myAddress, Presentation presentation, HandshakeHandler hh) throws ListenException {
	this.listener = ListenerFactory.listen(myAddress);
	this.presentation = presentation;
	this.myAddress = listener.address;
	this.hh = hh;
	this.mh = new VoidMessageHandler();

	//Start the thread manager
	threadGroup = new ThreadGroup("Freenet on " + myAddress);
	if (maximumConnectionThreads > 0) {
	    threadManager=new ThreadPool(threadGroup, 5,
					 maximumConnectionThreads,
					 maximumConnectionThreads);
	    threadManager.start();
	}

	// initialize Ticker. Note it has a VoidMessageHandler, so don't start
	// it until we have a real one.
	timer = new Ticker(mh, threadManager, tickerTime);
	timer.setDaemon(true);

	logger.log(this,"Freenet Core running on "+listener+
		   " (build "+buildNumber+")",Logger.NORMAL);
    }
    
    /**
     * Accepts connections to this Core and gives lets the provided
     * MessageHandler take care of messages received. 
     * this method does not return.
     *
     * @param mh   A MessageHandler that from now is used to handle messages
     *             received and internal MessageObjects from now on.
     **/
    public void acceptConnections(MessageHandler mh) {
	acceptConnections(mh, false, true);
    }

    /**
     * Accepts connections to this Core and gives lets the provided
     * MessageHandler take care of messages received. 
     *
     * @param mh   A MessageHandler that from now is used to handle messages
     *             received and internal MessageObjects from now on.
     * @param daemon  Should the thread that accepts connections be a daemon
     *                thread?
     * @param join     Whether the currentthread should join the thread that
     *                 accepts connections (ie, not return).
     **/
    public void acceptConnections(MessageHandler mh, boolean daemon,
				  boolean join) {
	this.mh = mh;
	// start the Timer
	timer.setMessageHandler(mh);
	timer.start();
	Thread t = new ConnectionAcceptor(threadGroup, listener);
	t.setDaemon(daemon);
	t.start();
	if (join) {
	    try {
		t.join();
	    } catch (InterruptedException e) {
	    }
	}
    }

    private class ConnectionAcceptor extends Thread {

	public ConnectionAcceptor(ThreadGroup threadGroup, Listener listener) {
	    super(threadGroup,"Freenet Core connection listener on " 
		  + listener);
	}

	public void run() {
	    logger.log(this, "Accepting connections.",Logger.MINOR);
	    
	    Connection conn;
	    RawMessage m;
	    ConnectionHandler c;
		
	    while(listen) {
		try {
		    conn = listener.accept();
		    logger.log(this,"Accepted connection:"
			       +conn,Logger.DEBUGGING);
		} catch (IOException e) {
		    e.printStackTrace(logger.getOut());
		    throw new RuntimeException("Problem accepting " 
					       + "next connection:" + e);
		}
		c = new ConnectionHandler(presentation, conn,mh);
		if (maximumConnectionThreads>0 && threadManager!=null) {
		    //Try to run the thread in the pool		    
		    if (!threadManager.run(c)) { 
			if (conn.privileged()) {
			    // Run anyway if this is a localhost connection.
			    logger.log(this,"Accepted privileged connection: "+conn+
				       " on an unpooled thread"+
				       " (connection limit reached)",
				       Logger.NORMAL);
			    c.start();
			}
			else {
			    // Give up.
			    logger.log(this,"Rejected connection:"+conn+
				       " (connection limit reached)",
				       Logger.NORMAL);
			    c.forceClose();
			}
		    }
		} else c.start(); // No thread management, start it.
	    }
	}
    }
    

    /**
     * Makes a connection from this Node to another
     * @param peer The Address of the other Freenet node/client
     * @return A ConnectionHandler that handles the new connection
     * @exception ConnectFailedException  if the Connection could not be opened
     *                                    or a new node did not respond to 
     *                                    handshake.
     **/
    public ConnectionHandler makeConnection(Address peer) throws ConnectFailedException {
	
	if(peer==null) throw new ConnectFailedException(peer);

	ConnectionHandler ch = connect(peer);

	if (!hh.getHandshake(this,ch))
	    throw new ConnectFailedException(peer);
    
	if (!ch.isOpen()) { // didn't stay open after handshake
	    ch = connect(peer);
	}

	return ch;
    }

    /**
     * This is the raw connection maker, it will not handshake
     * or check the status of the connection in any way.
     * @param peer  The address to connect to.
     * @exception ConnectFailedException if the Connection could not be opened.
     **/
    public ConnectionHandler connect(Address peer) throws ConnectFailedException {
	Connection c = null;
	try {
	    c = ConnectionFactory.connect(peer);
	} catch(Throwable e) {
	    String descr = e.toString()+" when attempting to connect to "+peer;
	    logger.log(this, descr, Logger.MINOR);
	    throw new ConnectFailedException(peer, descr);
	}

	logger.log(this,
		   "Connection between: " + c.getMyAddress(myAddress) + 
		   " and " + c.getPeerAddress(),Logger.DEBUGGING);
	if (c.getMyAddress(myAddress).equals(c.getPeerAddress())) {
	    logger.log(this,"No connecting to yourself",Logger.MINOR);	    
	    c.close();
	    throw new ConnectFailedException(peer);
	}
	
	ConnectionHandler ch = new ConnectionHandler(presentation, c, mh);
	ch.start(); // start listening on this connection
	return ch;
    }

  /**
   * Send the message using the appropriate protocol
   * @deprecated Use makeConnection() to grab a ConnectionHandler and 
   *             send from that instead.
   **/
    public void sendMessage(Message m, Address destination) throws SendFailedException {
	try {
	    ConnectionHandler ch = makeConnection(destination);
	    ch.sendMessage(m);
	} catch (ConnectFailedException e) {
	    throw new SendFailedException(destination);
	}
    }
}













