/*  Enhanced Params Object for Freenet
 *  Copyright (C) 2001  Timm Murray
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package Freenet;

import java.util.*;
import java.io.*;

/**
 * An Enhanced Params class. Allows for setting valid options and disallowing 
 * invalid ones.
 * @author <a href="mailto:hardburn@users.sourceforge.net">Timm Murray</a>
 */
public class EParams extends Params
{
	private Hashtable ht = new Hashtable();
	private boolean useValid = false;
	private String[] valid;
	private File file;

	public EParams()
	{
		// Do nothing, succsessfuly
	}

	public EParams(String file)
		throws InvalidParamException, IOException, FileNotFoundException
	{
		//System.out.println("Call EParams(String file)");
		this(new File(file));
	}
	
	public EParams(File file)
		throws InvalidParamException, IOException, FileNotFoundException
	{
		System.out.println("Call EParams(File file)");
		if(file != null) {
			this.file = file;
			readFile();
		}
	}

	public EParams(String file, String[] args)
		throws InvalidParamException, IOException, FileNotFoundException
	{
		//System.out.println("Call EParams(String file, String[] args)");
		this(new File(file), args);
	}

	public EParams(File file, String[] args)
		throws InvalidParamException, IOException, FileNotFoundException
	{
		System.out.println("Call EParams(File file, String[] args)");
		if(args.length > 0) parseArgs(args);
		if(file != null) {
			this.file = file;
			readFile();
		}
	}

	/**
	 * Read in a given file, CLI arguments, and give a String[] of valid 
	 * options.
	 * @param file Path to the file to read in
	 * @param args CLI arguments
	 * @param valid Valid arguments to be set
	 */
	public EParams(String file, String[] args, String[] valid)
			throws InvalidParamException, IOException, FileNotFoundException
	{
		//System.out.println("Call EParams(String file, String[] args, String[] valid)");
		this(new File(file), args, valid);
	}
		
	/**
	 * Read in a given file, CLI arguments, and give String[] of valid
	 * options
	 * @param file File to be read
	 * @param args CLI arguments
	 * @param valid Valid arguments to be set
	 */
	public EParams(File file, String[] args, String[] valid) 
			throws InvalidParamException, IOException, FileNotFoundException
	{
		System.out.println("Call EParams(File file, String[] args, String[] valid)");
		if(file != null) {
			this.file = file;
			readFile();
		}
		if(args.length > 0) parseArgs(args);
		if(valid.length > 0) {
			useValid = true;
			this.valid = new String[valid.length];
			for(int i = 0; i < valid.length; i++) this.valid[i] = valid[i];
		}
	}
	
	/**
	 * Read in CLI arguments and give a String[] of valid options.
	 * @param args CLI arguments
	 * @param valid Valid arguments to be set
	 */
	public EParams(String[] args, String[] valid)
			throws InvalidParamException
	{
		System.out.println("Call EParams(String[] args, String[] valid)");
		if(args.length > 0) parseArgs(args);
		if(valid.length > 0) {
			useValid = true;
			this.valid = new String[valid.length];
			for(int i = 0; i < valid.length; i++) this.valid[i] = valid[i];
		}
	}

	private void readFile() throws InvalidParamException, 
			FileNotFoundException, IOException
	{
		// # and ; marks a comment in the file, allow whitespace
		BufferedReader in = new BufferedReader(new FileReader(file));
		String line;
		while((line = in.readLine()) != null) {
			line = line.trim();

			// Takes care of blank lines
			if(line.length() < 1) continue;

			// If it starts with a # or ; skip to the next line automaticly
			if(line.charAt(0) == '#' || line.charAt(0) == ';') continue;
		
			StringTokenizer st = new StringTokenizer(line);
			int num = st.countTokens();
			String[] tokens = new String[num];
			for(int i = 0; i < num; i++) {
				tokens[i] = "";
			}
			
			for(int i = 0; i < num; i++) {
				tokens[i] = st.nextToken().trim();
				if(tokens[i].charAt(0) == '#' || tokens[i].charAt(0) == ';')
					break;
			}

			// tokens[0] should now contain a string in name=value form
			st = new StringTokenizer(tokens[0], "=");
			if(st.hasMoreTokens()) {
				String name = st.nextToken().trim();
				if(st.hasMoreTokens()) {
					String value = st.nextToken().trim();
					
					// make yes=true, no=false
					if(value.equals("yes")) value = "true";
					else if(value.equals("no")) value = "false";
					set(name, value);
				}
				// A lone name should be considered a "true" value
				else {
					set(name, new Boolean(true));
				}
			}
		}
	}

	private void parseArgs(String[] args) throws InvalidParamException
	{
		for(int i = 0; i < args.length; i++) {
			String name = args[i];
			if(name.charAt(0) == '-') {
				name = name.substring(1);
				String value;
				if(! (++i >= args.length)) {
					value = args[i];
					if(value.charAt(0) != '-') set(name, value);
					else {
						set(name, new Boolean(true));
						--i;
					}
				}
				else {
					set(name, new Boolean(true));
				}
			}
			else {
				set(name, new Boolean(true));
			}
		}
	}

	private boolean checkValid(String name)
	{
		if(! useValid) return true;
		for(int i = 0; i < valid.length; i++) {
			if(valid[i].equals(name)) return true;
		}
		return false;
	}

	public String[] getNames()
	{
		Enumeration e = ht.keys();
		String[] names = new String[ht.size()];
		for(int i = 0; e.hasMoreElements(); i++) {
			names[i] = (String) e.nextElement();
		}
		return names;
	}

	public Object[] getValues()
	{
		Enumeration e = ht.elements();
		Object[] values = new String[ht.size()];
		for(int i = 0; e.hasMoreElements(); i++) {
			Object o = e.nextElement();
			System.out.println(o.toString());
			System.out.println(String.valueOf(i));

			// It was choking on booleans for some reason, hopefuly this
			// fixes it
			if(o instanceof Boolean) values[i] = o.toString();
			else values[i] = o;
			System.out.println("End");
		}
		return values;
	}

	public int getSize() { return ht.size(); }

	public void set(String name, String value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, value);
		}
		else {
			if(checkValid(name)) ht.put(name, value);
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}
	
	public void set(String name, int value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, new Integer(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Integer(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}
	
	public void set(String name, long value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, new Long(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Long(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}

	public void set(String name, short value) throws InvalidParamException
	{
		if(! useValid) { 
			ht.put(name, new Short(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Short(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}

	public void set(String name, byte value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, new Byte(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Byte(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}

	public void set(String name, boolean value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, new Boolean(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Boolean(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}

	public void set(String name, double value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, new Double(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Double(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}

	public void set(String name, float value) throws InvalidParamException
	{
		if(! useValid) {
			ht.put(name, new Float(value));
		}
		else {
			if(checkValid(name)) ht.put(name, new Float(value));
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}

	public void set(String name, Object value) throws InvalidParamException
	{ 
		if(! useValid) {
			ht.put(name, value);
		}
		else {
			if(checkValid(name)) ht.put(name, value);
			else throw new InvalidParamException("Invalid param: " + name);
		}
	}
	
	public String getParam(String name)
	{
		return (String) ht.get(name);
	}
	public int getInt(String name)
	{
		Integer i = (Integer) ht.get(name);
		return i.intValue();
	}
	public long getLong(String name)
	{
		Long l = (Long) ht.get(name);
		return l.longValue();
	}
	public short getShort(String name)
	{
		Short s = (Short) ht.get(name);
		return s.shortValue();
	}
	public byte getByte(String name)
	{
		Byte b = (Byte) ht.get(name);
		return b.byteValue();
	}
	public boolean getBoolean(String name)
	{
		Boolean b = (Boolean) ht.get(name);
		return b.booleanValue();
	}
	public Object getObject(String name) { return ht.get(name); }
}
