package Freenet;
import Freenet.support.Logger;
import Freenet.support.io.ReadInputStream;
import Freenet.support.io.WriteOutputStream;
import java.io.IOException;
import java.io.EOFException;
import java.io.ByteArrayOutputStream;

import java.util.Hashtable;
import java.util.Enumeration;

/**
 * This is a wrapper for a Hashtable containing strings and other FieldsSets
 * that also includes a parser/printer in the format used for fields in normal
 *  Freenet messages. FieldSets are used for almost all data serializing 
 * (DataProperties, DataStore to disk, client metadata, etc) in Freenet.
 *
 * @author oskar
 **/
public class FieldSet {

    /** The maximum size in chars that a fieldset read by the parseFields 
     *  method can have 
     **/
    public static long MAX_PARSE_SIZE = 65536;

    private Hashtable fields = new Hashtable();
    private int size = 0;


    /**
     *  Construct an empty FieldSet
     **/
    public FieldSet() {}

    /**
     *  Construct a FieldSet from the given stream using the default separators
     **/
    public FieldSet(ReadInputStream in) throws IOException {
	parseFields(in);
    }

    /**
     * Get a string value
     * @param name the name of the field
     * @param def  the default string to return
     * @return The field interpreted as a string
     **/
    public String get(String s) {
	Object o = fields.get(s);
	if (o == null || !(o instanceof String))
	    return null;
	String value = (String) fields.get(s);
	return value;
    }

    /**
     * Get a field subset.
     * @param name the name of the field
     * @return The field interpreted as a subset of fields. Null if no
     *         such field is found.
     **/
    public FieldSet getSet(String s) {
	Object o = fields.get(s);
	if (o == null || !(o instanceof FieldSet))
	    return null;
	return (FieldSet) o;
    }

    /**
     * Remove a field
     * @param name The name of the field to remove
     **/
    public void remove(String s) {
	fields.remove(s);
    }

    /**
     * Alias for add. Here to make FieldSet conform to the Hashtable API.
     * I suppose it would be better to replace all add() calls with put()
     * class instead of having two methods that do the same thing, but
     * what a pain.
     * @param name	The name of the field.
     * @param value	The value to set to the field.
     */
     public void put(String name, String value)
     {
       add(name, value);
     }

    /**
     * Another alias for add() so that FieldSet with conform to the Hashtable
     * API.
     * @param name   The name of the field.
     * @param value  The value to set to the field.
     **/
    public void put(String name, FieldSet fs) {
	put(name, fs);
    }

    /**
     * Add a string value. This will overwrite any old values for this field.
     * @param name   The name of the field.
     * @param value  The value to set to the field.
     **/
    public void add(String name, String value) {
//	fields.put(name, value);

       // Careful, I've changed it so that all field names with '.' in them will
       // be stored as sets instead of strings.
       readField(name, value, '.');
    }

    /**
     * Add a subset. This will overwrite any old values for this field.
     * @param name   The name of the field.
     * @param value  The value to set to the field.
     **/
    public void add(String name, FieldSet fs) {
	fields.put(name, fs);
    }

    /**
     * Whether a field can be read as a string. 
     * @param   name The fields name.
     * @return  Whether the data in the field can be returned as a string.
     **/
    public boolean isString(String name) {
	return fields.get(name) != null && fields.get(name) instanceof String;
    }

    /**
     * Whether a field can be read as a set. 
     * @param   name The fields name.
     * @return  Whether the data in the field can be returned as a FieldSet.
     **/
    public boolean isSet(String name) {
	return fields.get(name) != null && fields.get(name) instanceof FieldSet;
    }
	    

    /**
     * @return  Whether this fieldset is empty.
     **/
    public boolean isEmpty() {
	return fields.isEmpty();
    }


    /**
     * Writes the fields to a stream with a standard syntax, and the given
     * semantics, using the standard separators.
     *
     * @param w         The stream to write to.
     **/
    public void writeFields(WriteOutputStream w) throws IOException {
	writeFields(w, "End", '\n', '=', '.');
    }

    /**
     * Writes the fields to a stream with a standard syntax, and the given
     * semantics.
     * @param w         The stream to write to.
     * @param sep    The character used to delimit the end of a field name value
     *             pair.
     * @param equal  The character that delimits the field name from the field
     *             value.
     * @param subset The character used to delimit subsets in the field name.
     * @param terminate The string to write at the end to terminate the fieldset
     *                this must not include the character "equal" used to 
     *                delimit the name from the value in pairs.
     **/
    public void writeFields(WriteOutputStream w, String terminate, char sep, char equal, char subset) throws IOException {
	inWriteFields(w,"", sep, equal, subset);
	w.writeUTF(terminate,sep);
    }

    /* internal version, adds the preceding string necessary for recursion */
    private void inWriteFields(WriteOutputStream w, String pre, char sep, char equal, char subset) throws IOException {

	for (Enumeration e = fieldNames() ; e.hasMoreElements() ;) {
	    String name = (String) e.nextElement();
	    if (isSet(name)) {
		getSet(name).inWriteFields(w, pre + name + subset, sep, equal, subset);
	    } else if (isString(name)) {
		w.writeUTF(pre + name + equal + get(name), sep);
	    } else {
		Core.logger.log(this,"Could not interpret field as string " + 
				"when trying to send.",Logger.MINOR);
	    }
	}
    }

    /**
     * Parses fields from a stream using the standard separators.
     *
     * @param in     The stream to read.
     * @return       The string encountered that lacks a field name/value 
     *               delimiter and that therefore is assumed to terminate 
     *               the fieldset.  (<code>null</code> if terminated by EOF.)
     * @exception    IOException  if something goes wrong.
     **/
    public String parseFields(ReadInputStream in) throws IOException {
	return privParse(in, '\n', '\r', '=', '.', true);
    }

    /**
     * Parses fields from a stream in a standard syntax with given
     * semantics. The easiest way to see the syntax is probably to
     * look at the output of writeFields() or look at the Freenet
     * protocol specs.
     *
     * @param in     The stream to read.
     * @param sep    The character used to delimit the end of a field name value
     *             pair.
     * @param equal  The character that delimits the field name from the field
     *             value.
     * @param subset The character used to delimit subsets in the field name.
     * @return  The string encountered that lacks a field name/value delimiter
     *          and that therefore is assumed to terminate the fieldset.
     *          (<code>null</code> if terminated by EOF.)
     * @exception  IOException  if something goes wrong.
     **/
    public String parseFields(ReadInputStream in, char sep, char equal, char subset) throws IOException {
	return privParse(in, sep, (char)0, equal, subset, false);
    }

    /**
     * Parses fields from a stream in a standard syntax with given
     * semantics. The easiest way to see the syntax is probably to
     * look at the output of writeFields() or look at the Freenet
     * protocol specs.
     *
     * @param in     The stream to read.
     * @param sep    The character used to delimit the end of a field name value
     *             pair.
     * @param ignore The character that should be ignored if it directly precedes
     *             the preceding seperator (used for \r)
     * @param equal  The character that delimits the field name from the field
     *             value
     * @param subset The character used to delimit subsets in the field name.
     * @return  The string encountered that lacks a field name/value delimiter
     *          and that therefore is assumed to terminate the fieldset.
     *          (<code>null</code> if terminated by EOF.)
     * @exception  IOException  if something goes wrong.
     **/
    public String parseFields(ReadInputStream in, char sep, char ignore, char equal, char subset) throws IOException {
	return privParse(in,sep,ignore,equal,subset,true);
    }

    private String privParse(ReadInputStream in, char sep, char ignore, char equal, char subset, boolean useignore) throws IOException {
	// Now read field/data pairs
	String s, name, data;
	int n;
	long read = 0;
	while(true) {
	    try {
		if (useignore)
		    s = in.readToEOF(sep,ignore);
		else
		    s = in.readToEOF(sep);
	    }
	    catch (EOFException e) {
		return null;
	    }
	    read += s.length();
	    n = s.indexOf(equal);
	    if (n >= 0) { // field
		name = s.substring(0,n);
		data = s.substring(n + 1);
		readField(name,data,subset);
		if (read > MAX_PARSE_SIZE)
		    throw new IOException("Message too large");
	    } else { // trailing
		return s;
	    }
	}
    }

    /**
     * An alias to fieldNames() to make FieldSet conform to the Hashtable API.
     *
     **/
     public Enumeration keys() { return fieldNames(); }

    /**
     * @return An enumeration of the field names in this FieldSet.
     **/
    public Enumeration fieldNames() {
	return fields.keys();
    }


    /**
     * @return A string representation of the FieldSet.
     **/
    public String toString() {
	try {
	    ByteArrayOutputStream sw = new ByteArrayOutputStream();
	    WriteOutputStream pr = new  WriteOutputStream(sw);
	    inWriteFields(pr,"",',','=','.');
	    pr.flush();
	    return "{" + new String(sw.toByteArray(),"UTF8") + "}";
	} catch (IOException e) { // shouldn't ever happen
	    return null;
	}
    }
    

    /**
     * Reads a field off a name and value
     **/
    protected void readField(String name, String value, char sep) {
	int dot = name.indexOf(sep);
	if (dot < 0) { // value
	    fields.put(name, value);
	} else { // subset
	    String fname = name.substring(0,dot);
	    Object o = fields.get(fname);
	    FieldSet fs;
	    if (o == null || !(o instanceof FieldSet)) {
		fs = new FieldSet();
		fields.put(fname,fs);
	    } else {
		fs = (FieldSet) o;
	    }
	    fs.readField(name.substring(dot+1),value,sep);
	}
    }
}






