package Freenet;
import Freenet.node.*;
import Freenet.client.ClientMessageObject;
import Freenet.message.TimedOut;
import Freenet.support.*;
import Freenet.crypt.EntropySource;
import java.lang.reflect.*;
import java.net.*;
import java.util.Hashtable;
import java.util.Enumeration;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
                
  Requires Classes: Node (1.0)
                    Address (1.0)
 */

/**
 * This is the abstract superclass of all messages
 *
 * @see Node
 * @see Address
 * @author Brandon Wiley (blanu@uts.cc.utexas.edu)
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

public abstract class Message implements NodeMessageObject, ClientMessageObject
{
    public static final String messageName="";

    private static EntropySource uniqueIdEntropy=new EntropySource();

    /** The Address of the node where this message originated **/
    public Address source;
    /** Address of this node on the network that this message came from **/
    public Address receivedAt = null;
    /** ConnectionHandler that accepted that received message **/
    public ConnectionHandler receivedWith = null;
    /** The number of times this message should be forwarded before
	generating a TimedOutMessage **/
    public long hopsToLive;
    /** The number number of hops this message is from its originating node **/
    public long depth;
    /** A randomly generated unique ID used to identify related messages **/
    public long id;
    /** Whether the connection should be kept open after sending this **/
    public boolean keepAlive = true;
    /** Any unknown / unused fields **/
    public FieldSet otherFields;
    
    /**
     * Creates a new message.
     * @param  idnum   The messages Unique ID, should be a random long > 65536
     * @param  htl     The number of hops this message should live.
     * @param  dpth    The number of hops from the originator to claim this
     *                 message is.
     * @param  otherFields   The remaining set of fields which are not directly
     *                       related to routing the message.
     */
    protected Message(long idnum, long htl, long dpth, FieldSet otherFields)
    {
	id=idnum;	
	hopsToLive=htl;
	depth=dpth;
	this.otherFields = otherFields;
    }
    
    /**
     * Creates a message from a RawMessage, as read from a presentation
     * @param raw The RawMessage. Please note that the RawMessage is 
     *            consumed after this. Both the normal and trailing
     *            fields are corrupted. Use toRawMessage() to restore it.
     **/   
    protected Message(RawMessage raw) throws InvalidMessageException {
	try {
	    otherFields = raw.fs;
	    id = Fields.stringToLong(otherFields.get("UniqueID"));
	    if (id >= 0 && id < 0x10000)
		throw new InvalidMessageException("Message id out of range");
	    Core.randSource.acceptEntropy(uniqueIdEntropy, id, 64);
	    otherFields.remove("UniquieID");
	    hopsToLive = Fields.stringToLong(otherFields.get("HopsToLive"));
	    otherFields.remove("HopsToLive");
	    depth = Fields.stringToLong(otherFields.get("Depth"));
	    otherFields.remove("Depth");
	} catch (NumberFormatException e) {
	    throw new InvalidMessageException("Bad number");
	} catch (NullPointerException e) {
	    // this is a little ugly
	    e.printStackTrace();
	    throw new InvalidMessageException("Necessary Field Missing");
	}
    }

    public RawMessage toRawMessage(Presentation t) {
	RawMessage r = t.newMessage("",source,keepAlive,
				    otherFields,0,"EndMessage",null);

	r.fs.add("UniqueID",Fields.longToString(id));
	r.fs.add("HopsToLive",Fields.longToString(hopsToLive));
	r.fs.add("Depth",Fields.longToString(depth));
	return r;
    }


    /** 
     * Called by the connection handler after right constructing
     * the message, while still in touch with the connection
     * @param me The address of this node that the message was
     *           received on.
     * @param peer The address of the node the message was received
     *             from.
     **/
    public void initSources(Address me, Address peer, ConnectionHandler ch) {
	receivedAt = me;
	source = peer;
	receivedWith = ch;
    }

    public long id() {
	return id;
    }

    /**
     * Called by a node after this message is received.  This allows
     * message handling code to be packaged within the message class
     * that is being handled.
     * @param n The node that called this message.  This should be used
     *          to make any nescessary modifications to the DataStore etc.
     * @param sb Null unless this node has been seen before.  If non-null it
     *           is the Object returned by the received method of the
     *           last message seen with this ID.
     * @return The object to be passed to any messages received in the near
     *         future with the same ID as this one.  Null if no object 
     *         should be passed.
     **/
    public final MessageMemory received(Node n, MessageMemory sb) { 
	// Reduce the lifespan of this message
	hopsToLive = hopsToLive > Node.maxHopsToLive ? Node.maxHopsToLive : --hopsToLive;
	depth++;
	return 	(hopsToLive<=0 ?
		 timeOut(n, sb) :
		 pReceived(n, sb)); // Actually do something with the message
    }

    /**
     * Messages normally do nothing about Exceptions.
     **/
    public void setException(Exception e) {
    }

    public void sending(Core n, ConnectionHandler ch) throws SendFailedException {
	sending(n,ch.peer(),ch.local(n.myAddress));
    }

    /**
     * Called just before the message is sent, and makes any last minute
     * changes.
     *
     * @param n          The node that is sending the message
     * @param peer       The address to which the message is being sent
     * @param myAddress  The sending nodes address with respect to peer
     **/

    public void sending(Core n, Address peer, Address myAddress) throws SendFailedException
    {
	if (myAddress.equals(peer))
	    throw new SendFailedException(peer); // don't send to yourself
	source = myAddress;
    }

  /**
   * An overridable version of the received message.
   * @param n The node that called this message.  This should be used
   *          to make any nescessary modifications to the DataStore etc.
   * @param sb Null unless this node has been seen before.  If non-null it
   *           is the Object returned by the received method of the
   *           last message seen with this ID.
   * @return The object to be passed to any messages received in the near
   *         future with the same ID as this one.  Null if no object 
   *         should be passed.
   **/

    abstract protected MessageMemory pReceived(Node n, MessageMemory sb);


    protected MessageMemory timeOut(Node n, MessageMemory sb)
    {
	Core.logger.log(this,"Message timed out, sending timeout to original source",Logger.MINOR);
	TimedOut to=new TimedOut(id, depth); // this message is going in
    
	try {
	    ConnectionHandler ch = sendReply(n, to);
	    if (ch != null)
	    	ch.close();
	} catch(SendFailedException sfe) {
	    Core.logger.log(this,"Send of timeout failed",Logger.NORMAL);
	}
	
	return sb;
    }
    
    /**
     * Sends a message back to the node from which this message was received.
     * If a connection is open, that will be used, otherwise a new one opened
     * @param n this node
     * @param m the message to send.
     * @return if there was an open connection back, or if there was a
     *         a connection back that was closed, the connection over which
     *         the message was sent is returned. If there was no connection
     *         back, the new connection is closed after send, and null
     *         is returned.
     *         
     **/

    protected ConnectionHandler sendReply(Node n, Message m) throws SendFailedException {
	try {
	    ConnectionHandler ch;
	    if (receivedWith == null) {
		ch = n.makeConnection(source);
		m.keepAlive = false; // close again after sending message
	    } else if (!receivedWith.isOpen())
		ch = n.makeConnection(source);
	    else 
		ch = receivedWith;

	    m.sending(n, ch);
	    ch.sendMessage(m);

	    return receivedWith == null ? null : ch;
	} catch (ConnectFailedException e) {
	    throw new SendFailedException(e.peer);
	}
    }

    /**
     * Sends this message back to the node from which the chain was originally
     * received.
     * @param n   this node.
     * @param mm  The MessageMemory for this chain of messages.
     * @return    If there was an open connection back in the MessageMemory,
     *            or if there was a connection back that had been closed,
     *            the connection which this message is sent on is returned. 
     *            If there was no  connection back stored, null is returned.
     **/

    public ConnectionHandler sendBack(Node n, MessageMemory mm) throws SendFailedException {
	try {
	    ConnectionHandler ch;
	    if (mm.replyCon == null) {
		ch = n.makeConnection(mm.origRec);
		keepAlive = false;
	    } else if (!mm.replyCon.isOpen()) {
		ch = n.makeConnection(mm.origRec);
	    } else {
		ch = mm.replyCon;
	    }
	    sending(n, ch);
	    ch.sendMessage(this);
	    return mm.replyCon == null ? null : ch;
	} catch (ConnectFailedException e) {
	    throw new SendFailedException(e.peer);
	}
    }

    public String toString()
    {
	//        RawMessage r = toRawMessage();
	return "Src:"+source+" htl:"+hopsToLive+
	    " depth:" + depth + " id:"+Long.toHexString(id) +
	    " type:" + this.getClass().getName();
    }
}


