package Freenet;

import Freenet.support.*;
import java.util.*;
import java.io.*;

/**
 * Stores Freenet parameters
 **/
public class Params
{
    /**
     * The default names of files to use when constructing a Params object. (Moved here by SB).
     */
    static public final String[] defaultRCFiles = {".freenetrc", "freenet.ini"};

    FieldSet params = new FieldSet();
    Vector argVector = new Vector();
    private String savedOption = null;
    private boolean noMoreOptions = false;


    public static void main(String[] args)
    {
	try {
	    Params p = new Params(".freenetrc", args);
	    System.out.println("Params: " + p.params);
	    System.out.println("Args: " + p.argVector);
	}
	catch (IOException e) {
	    System.out.println ("Error reading .freenetrc: " + e);
	}
    }

    public Params() { }
    
    public Params(String filename) throws FileNotFoundException
    {
	init(ArrayFactory.makeArray(filename), null);
    }

    public Params(String[] args)
    {
        try {
            init((String[]) null, args);
        }
          // Won't happen because we're not passing any filenames.
        catch (FileNotFoundException e) {}
    }

    public Params(String filename, String[] args) throws FileNotFoundException
    {
	init(ArrayFactory.makeArray(filename), args);
    }


    /**
     * Read in Freenet parameters
     * 
     * Really, this is kind of a hack and needs proper semantic option
     * handling to distinguish between those that take values and
     * those that don't.
     *
     **/
    public Params(String[] files, String[] args) throws FileNotFoundException
    {
        init(files, args);
    }

    private void init(String[] files, String[] args) throws FileNotFoundException
    {
	// read params from command line (overrides file)
	if (args != null) {
	    for (int x = 0; x < args.length; x++) {
		if (args[x] != null) {
		    if (isOption (args[x])) {
			// process previous option, if any
			if (savedOption != null) {
			    setParam (savedOption, "");
			}

			// save for next iteration
			if (!args[x].equals ("--")) {
			    savedOption = args[x].substring(1);
			}
			else {
			    noMoreOptions = true;
			    savedOption = null;
			}
		    }
		    else {
			if (savedOption != null) {
			    // store value for previous option
			    setParam (savedOption, args[x]);
			}
			else {
			    // store command argument
			    argVector.addElement (args[x]);
			}
			savedOption = null;
		    }
		}
	    }

	    // process last option, if any
	    if (savedOption != null)
		setParam (savedOption, "");
	}

	// get config file specified on command line, if any

	String paramFile = getParam("paramFile");

	if (paramFile != null) {
	    if (files != null) {
		String[] tmp = new String[files.length + 1];
		System.arraycopy(files, 0, tmp, 0, files.length);
		tmp[files.length] = paramFile;
		files = tmp;
	    } else {
		files = ArrayFactory.makeArray(paramFile);
	    }
	}

	// read params from files
	if (files != null) {
            boolean foundRCFile = false;
	    for (int x = files.length - 1; x >= 0; x--) {
		if (files[x] != null) {
		    try {
			Core.logger.log(this, "Reading config file " + 
					files[x], Logger.DEBUGGING);
			readParams(files[x], false);
                        foundRCFile = true;
		    } catch (FileNotFoundException e) {
			Core.logger.log(this, "Couldn't load config file (this is not a problem, and if Freenet procedes not to work for you 100/1 says this is not the reason)." + 
					files[x], Logger.MINOR);
		    }
		}
	    }
            if (!foundRCFile) {
                StringBuffer sb = new StringBuffer();
                if (files.length > 1)
                  sb.append("any of ");
                for (int i = 0; i < files.length; i++) {
                  if (i > 0) sb.append(", ");
                  sb.append(files[i]);
                }
                throw new FileNotFoundException(sb.toString());
            }
	}
    }

    public void readParams(String filename) throws FileNotFoundException {
	readParams(filename, true);
    }

    public void readParams(String filename, boolean overwrite) throws FileNotFoundException {
	FileReader fr = null;
	BufferedReader br = null;
	try {
	    fr = new FileReader(filename);
	    br = new BufferedReader(fr);
	    String s;
    	    while ((s = br.readLine()) != null) {
		String trim = s.trim();
		if (trim.length() != 0 && "#;*".indexOf(trim.charAt(0)) < 0) { // if it's not a comment
		    int index = s.indexOf('=');
		    if (index >= 0) {
			String name = s.substring(0, index);
			String value = s.substring(index + 1);

			String exactName = name.trim().toLowerCase();
			String exactValue = value.trim();
			
			if (overwrite || getParam(exactName) == null) {
			    params.put(exactName, exactValue);
			}
		    }
		}
	    }
	}
	catch (FileNotFoundException fnf) {
	    throw fnf;
	}
	catch (IOException e) {
	    Core.logger.log(this, "Error while reading config file", Logger.ERROR);
	}
	finally {
	    try {
		if (br != null) br.close();
		if (fr != null) fr.close();
	    }
	    catch (IOException e) {
		Core.logger.log(this, "Error while closing config file", Logger.ERROR);
	    }
	}
    }

    public void removeParam(String name) {
	params.remove(name.toLowerCase());
    }

  public void setParam(String name, String value)
  {
      Core.logger.log(this, "set \""+name+"\" to \""+value+"\"", Logger.MINOR);
      params.put(name.toLowerCase(), value);
  }

  public String getParam(String name, String def)
  {
    String temp=getParam(name);
    if(temp==null) return def;
    else return temp;
  }

  // get number of command arguments
  public int getNumArgs()
  {
    return argVector.size();
  }

  // get command argument by position
  public String getArg(int position)
  {
    if (position >= 0 && position < argVector.size()) {
	return (String) argVector.elementAt (position);
    } else {
	// no such argument
	return null;
    }
  }

    public void addArg(String argument)
    {
	argVector.addElement (argument);
	return;
    }

//  public String getParam(String name)
//  {
//    String r = (String)params.get(name.toLowerCase());
//    if (r == null) {
//	Core.logger.log(this, 
//			"No value specified for parameter '"+name+"'",
//			Logger.DEBUGGING);
//    }
//    return r;
//  }

    public String[] getlist(String name, String[] def) {
	if (getParam(name) == null) return def;
	else return getlist(name);
    }

    public FieldSet getset(String name, FieldSet def) {
    	FieldSet f=getset(name);
        if(f==null) return def;
        else return f;
    }

    public FieldSet getset(String name) {
      return params.getSet(name);
    }

    public String[] getlist(String name) {
	String s = getParam(name);
	if (s == null) return null;
	else {
	    StringTokenizer st = new StringTokenizer(s,",:;");
	    String[] list = new String[st.countTokens()];
	    for (int i = 0 ; st.hasMoreTokens() ; i++) {
		list[i] = st.nextToken().trim();
	    }
	    return list;
	}
    }

  public long getlong(String name, long def)
  {
    String temp=getParam(name);
    if(temp==null) return def;
    else return getlong(name);
  }

  public long getlong(String name)
  {
    return (Long.parseLong(getParam(name)));
  }

  public int getint(String name, int def)
  {
    String temp=getParam(name);
    if(temp==null) return def;
    else return getint(name);
  }

  public int getint(String name)
  {
    return (Integer.parseInt(getParam(name)));
  }

  public short getshort(String name, short def)
  {
    String temp=getParam(name);
    if(temp==null) return def;
    else return getshort(name);
  }

  public short getshort(String name)
  {
    return (Short.parseShort(getParam(name)));
  }

    public boolean getboolean(String name, boolean def)
    {
	String temp = getParam(name);
	if (temp == null) {
	    return def;
	}
	else {
	    if (def == false) {
		return temp.equalsIgnoreCase("yes") || temp.equalsIgnoreCase("true");
	    }
	    else {
		return !(temp.equalsIgnoreCase("no") || temp.equalsIgnoreCase("false"));
	    }
	}
    }

    public boolean getboolean(String name)
    {
	return (getParam(name).equalsIgnoreCase("yes") || getParam(name).equalsIgnoreCase("true"));
    }

    public String getParam(String name) {
	FieldSet fieldSet = params;
	String prefix = "";
	while (true) {
	    int dot = name.indexOf('.');
	    if (dot < 0) {
		String result = fieldSet.get(name.toLowerCase());
		if (result == null)
		    Core.logger.log(this, 
			"No value specified for parameter '" + prefix + name + "'",
			Logger.DEBUGGING);
		return result;
	    }
	    fieldSet = fieldSet.getSet(name.substring(0, dot));
	    if (fieldSet == null) {
		Core.logger.log(this,
		    "No set specified for parameter '" + prefix + name.substring(0, dot) + "'",
		    Logger.DEBUGGING);
		return null;
	    }
	    prefix = prefix + name.substring(0, dot) + ".";
	    name = name.substring(dot + 1);
	}
    }

    private void recursiveWriteParams(String prefix, FieldSet fieldSet, Hashtable processed, PrintWriter pw) {
	Enumeration e = fieldSet.fieldNames();
	while (e.hasMoreElements()) {
	    String name = (String) e.nextElement();
	    if (fieldSet.isSet(name)) {
		recursiveWriteParams(prefix + name + ".", fieldSet.getSet(name), processed, pw);
	    }
	    else {
		String fullyQualifiedName = prefix + name;
		if (!processed.containsKey(fullyQualifiedName)) {
		    pw.println(fullyQualifiedName + "=" + fieldSet.get(name));
		}
	    }
	}
    }

    public void writeParams(String filename) {
	// The way that this function works is that it reads the old
	// configuration file and replaces the setting values as it
	// goes.  All new settings are just dumped at the bottom of 
	// the file.  The idea is to keep all the comments and the
	// ordering of the settings in tact.

	File file = new File(filename);

	// Mark all the fields which have already been added
	Hashtable processed = new Hashtable(); 

	// Read the old file and put in the new values each setting.
	// Stick the output into a StringBuffer so that it can be
	// easily dumped out to the new configuration file.  This saves
	// the hassle of renaming the old config file, etc.
	StringBuffer configFile = new StringBuffer();
	FileReader fr = null;
	BufferedReader br = null;
	try {
	    if (file.exists()) {
		fr = new FileReader(file);
		br = new BufferedReader(fr);
		String s;
		while ((s = br.readLine()) != null) {
		    int index = s.indexOf('=');
		    if (index >= 0) {
			String name = s.substring(0, index);
			if (!processed.containsKey(name)) {
			    String value = getParam(name.trim());
			    if (value == null)
				configFile.append(s);
			    else
				configFile.append(name + "=" + value);
			    processed.put(name.toLowerCase(), name);
			}
		    }
		    else {
			configFile.append(s);
		    }
		    configFile.append("\r\n");
		}
	    }
	}
	catch (IOException e) {
	    Core.logger.log(this, "Error while reading old config file during config write", Logger.ERROR);
	}
	finally {
	    try {
		if (br != null) br.close();
		if (fr != null) fr.close();
	    }
	    catch (IOException e) {
		Core.logger.log(this, "Error while closing old config file during config write", Logger.ERROR);
	    }
	}

	// The actual writing of the config file
	try {
	    PrintWriter pw = new PrintWriter(new FileWriter(file));

	    // Write the contents of the old config
	    pw.print(configFile.toString());

	    // Write out any properties which weren't in the input file
	    recursiveWriteParams("", params, processed, pw);
	    pw.close();
	}
	catch(IOException e) {
	    Core.logger.log(this, "Could not write to config file", Logger.ERROR);
	}
    }

    private boolean isOption (String arg) {
	return (!noMoreOptions && arg.startsWith("-") && !arg.equals("-"));
    }

    /**
     * Hack to make an array on the fly
     **/
    static class ArrayFactory {
	public static String[] makeArray (String filename) {
	    String[] arr = {filename};
	    return arr;
	}
    }
}



