package Freenet;

import java.util.Hashtable;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * The Peer class wraps all pertinant information about peer nodes in Freenet.
 * Currently, this means it holds the address of the peer, and a dictionary
 * of possible performance or reliability metrics.
 *
 * @author Scott G. Miller <scgmille@indiana.edu>
 */
public class Peer {
    protected Address addr;
    protected Hashtable metrics;

    public Peer(Address addr) {
	this.addr=addr;
	metrics=new Hashtable();
    }

    public Peer(Address addr, Hashtable metrics) {
	this(addr);
	this.metrics=metrics;
    }

    /**
     * Returns the address of this peer
     */
    public Address getAddress() {
	return addr;
    }

    /**
     * Gets the unrefined Object metric with key <b>name</b>
     *
     * @param name The key of the metric you wish to retrieve
     * @return The value of a given metric
     */
    public Object getMetric(String name) {
	return metrics.get(name);
    }

    /**
     * Sets the unrefined Object metric with key <b>name</b>
     *
     * @param name The key of the metric you wish to set
     * @param newValue The new value for the metric
     * @return The previous value of this metric
     */
    public Object setMetric(String name, Object newValue) {
	return metrics.put(name, newValue);
    }

    /**
     * Sets the metric with key <b>name</b> to the integer value 
     * <b>newValue</b>
     *
     * @param name The key of the metric you wish to set
     * @param newValue The new value for the metric
     */
    public void setMetric(String name, int newValue) {
	setMetric(name, new Integer(newValue));
    }

    /**
     * Sets the metric with key <b>name</b> to the double value 
     * <b>newValue</b>
     *
     * @param name The key of the metric you wish to set
     * @param newValue The new value for the metric
     */
    public void setMetric(String name, double newValue) {
	setMetric(name, new Double(newValue));
    }

    /**
     * Gets the value of the metric <b>name</b> as an integer.  If the
     * metric is unset, <b>defaultValue</b> is returned as the value of this
     * metric.
     *
     * @param name The key of the metric you wish to retrieve
     * @param defaultValue The default value of the metric
     */
    public int getMetric_int(String name, int defaultValue) {
	Integer i=(Integer)getMetric(name);
	return (i==null ? defaultValue : i.intValue());
    }

    /**
     * Adds <b>delta</b> to the integer metric <b>name</b>.  If the
     * metric is unset, the new value becomes simply <b>delta</b>.  Otherwise
     * the new value is defined as (<i>previous value</i> + <b>delta</b>).
     *
     * @param name The key of the metric you wish to retrieve
     * @param delta The quantity to add to the metric
     */
    public int metricAdd_int(String name, int delta) {
	int v=getMetric_int(name, 0);
	v+=delta;
	setMetric(name, v);
	return v;
    }
	
    /**
     * Multiplies the integer metric <b>name</b> by 
     * <b>multiplier</b>.
     * 
     * If the metric is unset, the new value becomes simply <b>0</b>.  
     * Otherwise the new value is defined as 
     * (<i>previous value</i> * <b>multiplier</b>).
     *
     * @param name The key of the metric you wish to retrieve
     * @param multiplier The quantity to multiply to the metric
     */
    public int metricMul_int(String name, double multiplier) {
	int v=getMetric_int(name, 0);
	v*=multiplier;
	setMetric(name, v);
	return v;
    }

    /**
     * Gets the value of the metric <b>name</b> as an double.  If the
     * metric is unset, <b>defaultValue</b> is returned as the value of this
     * metric.
     *
     * @param name The key of the metric you wish to retrieve
     * @param defaultValue The defaultValue value of the metric
     */
    public double getMetric_double(String name, double defaultValue) {
	Double d=(Double)getMetric(name);
	return (d==null ? defaultValue : d.doubleValue());
    }

    /**
     * Adds <b>delta</b> to the double metric <b>name</b>.  If the
     * metric is unset, the new value becomes simply <b>delta</b>.  Otherwise
     * the new value is defined as (<i>previous value</i> + <b>delta</b>).
     *
     * @param name The key of the metric you wish to retrieve
     * @param delta The quantity to add to the metric
     */
    public double metricAdd_double(String name, double delta) {
	double v=getMetric_double(name, 0);
	v+=delta;
	setMetric(name, v);
	return v;
    }
	
    /**
     * Multiplies the double metric <b>name</b> by 
     * <b>multiplier</b>.
     * 
     * If the metric is unset, the new value becomes simply <b>0</b>.  
     * Otherwise the new value is defined as 
     * (<i>previous value</i> * <b>multiplier</b>).
     *
     * @param name The key of the metric you wish to retrieve
     * @param multiplier The quantity to multiply to the metric
     */
    public double metricMul_double(String name, double multiplier) {
	double v=getMetric_double(name, 0);
	v*=multiplier;
	setMetric(name, v);
	return v;
    }

    public String toString() {
	return "Peer["+addr.toString()+"]";
    }
}

    
    
