package Freenet;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.io.EOFException;

/**
 * Implementations of Presentation are used to handle connections
 * with a certain message protocol. It is used used to initialize
 * new connections, read messages off the connection into the RawMessage 
 * format, and creating new RawMessages for writing to a stream with
 * this protocol.
 **/

public abstract class Presentation {

    /**
     * These are control byte values used in the streams of all presentations 
     **/
    public static final byte
	CB_OK        = 0x00,
	CB_RESTARTED = 0x01;
    //Values over 128 are internal
    public static final int 
	CB_BAD_DATA = 0x81,
	CB_CONN_DIED = 0x82,
	CB_BAD_KEY = 0x83;

    /**
     * Performs a key exchange or other initalization of a new connection
     * @param c   The Connection to initialize
     **/
    public abstract void initConnection(Connection c) throws IOException;

    /**
     * Return an InputStream that reads data off the connection with
     * this presentation.
     * param c The connection you wish to read the message to
     * return  An inputStream from which the the RawMessage can be read.
     **/
    public abstract InputStream getReceiveStream(Connection c) throws IOException;

    /**
     * Return an OutputStream that writes data to the connection with
     * this presentation.
     * param c  The connection you wish to write the message to
     * return  An outputstream that to which the RawMessage and trailing
     *         field can be written.
     **/
    public abstract OutputStream getSendStream(Connection c) throws IOException;

    /**
     * Creates a new RawMessage of a stream by reading this presentation. This
     * method locks until an entire message has been read (excluding trailing)
     * @param in    The stream to read from.
     * @return      A new raw message.
     **/
    public abstract RawMessage readMessage(InputStream in) throws InvalidMessageException, EOFException;

    /** 
     * Creates a new RawMessage of a given type that uses this presentation
     * @param messageType   The name of the message type
     * @param source        The address from which this message claims to come.
     *                      May be null.
     * @param keepAlive     Whether to keep alive the connection after 
     *                      receiving or sending this message.
     * @param fs            A set of message specific fields.
     * @param trailingLength The length of the trailing data, or 0 if there 
     *                       is no trailing data
     * @param trailingName   The name of the trailing field, or null if there
     *                       is no trailing data
     * @param data           An inputstream containing the trailing data,
     *                       straight off the DataStore (decrypted).
     * @return     A new raw message
     **/
    public abstract RawMessage newMessage(String messageType, Address Source, boolean keepAlive, FieldSet fs, long trailingLength, String trailingName, InputStream data);
}
